/***************************************************************************
                    musicbrainzquery_dialog.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 2004
    copyright            : (C) 2004 by Markus Kalkbrenner
    email                : mkalkbrenner@users.sourceforge.net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "musicbrainzquery_dialog.h"

#if HAVE_LIBMUSICBRAINZ

#include "prokyon.h"
#include "musicbrainzclient.h"
#include "qmbmetadata.h"
#include "jobmanager.h"
#include "gui.h"
#include "lvi_musicbrainzresult.h"
#include "pixmapcache.h"

#include <qvaluevector.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qmessagebox.h>

#endif /*HAVE_LIBMUSICBRAINZ*/

MusicBrainzQuery_Dialog::MusicBrainzQuery_Dialog(QWidget *parent, const QString &path, const char *name)
  : QDialog(parent, name, true)
{    
#if HAVE_LIBMUSICBRAINZ
    setCaption(/* TRANSLATORS: filename */ QString(_("Querying MusicBrainz for %1")).arg(path));

    QGridLayout *grid = new QGridLayout(this, 2, 2);
    
    result_view = new ExtendedListView(this);
    
    result_view->addColumn(_("Artist"), 120);
    result_view->addColumn(_("Title"), 160);
    result_view->addColumn(_("Length"), -1);
    result_view->setColumnAlignment(LVI_MusicBrainzResult::LengthColumn, Qt::AlignRight);
    result_view->addColumn(_("Album"), 160);
    
    result_view->setItemMargin(0);
    result_view->setAllColumnsShowFocus(true);
    result_view->setSelectionMode(QListView::Single);
    result_view->setSorting(LVI_MusicBrainzResult::ArtistColumn);
    result_view->setShowSortIndicator(true);
    
    grid->addMultiCellWidget(result_view, 0, 0, 0, 2);
    
    button_select = new QPushButton(pixmapcache->getFadeIconSet("filesave.png"), _("Select"), this);
    button_cancel = new QPushButton(pixmapcache->getFadeIconSet("action_discard.png"), _("Cancel"), this);
    button_skip_remaining = new QPushButton(pixmapcache->getFadeIconSet("action_discard.png"), _("Skip Remaining"), this);
    button_select->setEnabled(false);
    button_cancel->setEnabled(false);
    button_skip_remaining->setEnabled(false);
    
    grid->addWidget(button_select, 1, 0);
    grid->addWidget(button_cancel, 1, 1);
    grid->addWidget(button_skip_remaining, 1, 2);
    
    connect(result_view, SIGNAL(selectionChanged()),
      this, SLOT(slot_selectionChanged()));
    connect(button_select, SIGNAL(clicked()),
      this, SLOT(slot_select()));
    connect(button_cancel, SIGNAL(clicked()),
            this, SLOT(slot_cancel()));
    connect(button_skip_remaining, SIGNAL(clicked()),
            this, SLOT(slot_skipRemaining()));
    
    selected = NULL;
    fullPath = path;
    
    // show dialog before proceeding
    QTimer::singleShot( 100, this, SLOT(slot_queryServer()) );
#endif /*HAVE_LIBMUSICBRAINZ*/
}

#if HAVE_LIBMUSICBRAINZ
LVI_MusicBrainzResult* MusicBrainzQuery_Dialog::getSelected()
{
    return selected;
}
#endif /*HAVE_LIBMUSICBRAINZ*/

void MusicBrainzQuery_Dialog::slot_queryServer()
{
#if HAVE_LIBMUSICBRAINZ
    if(jobman->getNumberOfJobsRunning() > 0) {
       QMessageBox::information(parentWidget(), _("Note"),
           QString(_("MusicBrainz can't be queried if there still jobs running in background.\nCurrently there are %1 jobs.")).arg(jobman->getNumberOfJobsRunning()),  QMessageBox::Ok);
       reject();
       return;
    }
    
    LVI_MusicBrainzResult *item = static_cast<LVI_MusicBrainzResult*>(gui->getListing()->list()->currentItem());
    if(item == NULL) return;
    
    pauseWatchers();
    QValueVector<QMBMetadata> result = musicBrainzClient->queryServer(fullPath, item->getOriginalTrack());
    proceedWatchers();
    
    setCaption(/* TRANSLATORS: filename */ QString(_("Result(s) from MusicBrainz for %1")).arg(fullPath));
    
    if(!result.empty()) {
        QValueVector<QMBMetadata>::iterator lastElement = result.end();
        for(QValueVector<QMBMetadata>::iterator it = result.begin(); it != lastElement; ++it) {
            if(verbose == 10) {
                qWarning("\nmusicbrainz result:");
                it->debugDump();
            }
            TRACK *track = new TRACK();
            track->length = QString(it->GetDuration()).toInt() / 1000;
            track->artist = it->GetArtistName().stripWhiteSpace();
            track->title = it->GetTrackName().stripWhiteSpace();
            track->album = it->GetAlbumName().stripWhiteSpace();
            track->tracknumber = QString(it->GetTrackNumber()).toInt();
            new LVI_MusicBrainzResult(result_view, track);
         }
    } else {
        reject();
    }
    
    button_cancel->setEnabled(true);
    button_skip_remaining->setEnabled(true);
#endif /*HAVE_LIBMUSICBRAINZ*/
}

void MusicBrainzQuery_Dialog::slot_selectionChanged()
{
#if HAVE_LIBMUSICBRAINZ
    button_select->setEnabled(true);
#endif /*HAVE_LIBMUSICBRAINZ*/
}

void MusicBrainzQuery_Dialog::slot_select()
{
#if HAVE_LIBMUSICBRAINZ
    selected = dynamic_cast<LVI_MusicBrainzResult*>(result_view->selectedItem());
    accept();
#endif /*HAVE_LIBMUSICBRAINZ*/
}

void MusicBrainzQuery_Dialog::slot_cancel()
{
#if HAVE_LIBMUSICBRAINZ
    reject();
#endif /*HAVE_LIBMUSICBRAINZ*/
}

void MusicBrainzQuery_Dialog::slot_skipRemaining() {
#if HAVE_LIBMUSICBRAINZ
    done(MusicBrainzQuery_Dialog::SkipRemaining);
#endif /*HAVE_LIBMUSICBRAINZ*/
}

MusicBrainzQuery_Dialog::~MusicBrainzQuery_Dialog()
{
}

