/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "gui_mixxxqueue.h"
#include "mixxxclient.h"
#include "pixmapcache.h"

#include <qlayout.h>
#include <qgroupbox.h>

#include <prokyon.h>

// ##############################################
// # initialize search gui
// ##############################################
GUI_MixxxQueue::GUI_MixxxQueue(QWidget *parent, const char *name)
	: QFrame(parent, name)
{
    int lineSpace = fontMetrics().lineSpacing();
  
	QGridLayout *grid = new QGridLayout(this, 2, 3);
    grid->addColSpacing(1, lineSpace / 2);
    grid->setColStretch(0, 1);
    grid->setColStretch(2, 1);
    // channel 1

    QGroupBox *ch1_frame = new QGroupBox(_("Channel 1"), this);
    QGridLayout *ch1_grid = new QGridLayout(ch1_frame, 5, 9);
    ch1_grid->addColSpacing(0, lineSpace / 2);
    ch1_grid->addColSpacing(8, lineSpace / 2);
    ch1_grid->addRowSpacing(0, lineSpace);
    ch1_grid->addRowSpacing(4, lineSpace / 2);
    ch1_grid->setColStretch(1, 1);
    ch1_grid->setColStretch(2, 1);
    ch1_grid->setColStretch(3, 1);
    ch1_grid->setColStretch(4, 1);
    ch1_grid->setColStretch(5, 1);
    ch1_grid->setColStretch(6, 1);
    ch1_grid->setColStretch(7, 1);
    
    ch1_view = new ExtendedListView(ch1_frame);
    ch1_view->addColumn("", 16);
    ch1_view->addColumn(pixmapcache->getFadeIconSet("stock_music-library.png"), _("Tracks in Playlist"), -1);
    
    ch1_view->setItemMargin(0);
    ch1_view->setSelectionMode(QListView::Single);
    ch1_view->setSorting(-1);
    ch1_view->setAllColumnsShowFocus(true);
  
    ch1_grid->addMultiCellWidget(ch1_view, 1, 1, 1, 7);
    
    ch1_up   = new QPushButton(_("Up"), ch1_frame);
    ch1_up->setPixmap(pixmapcache->get("up.png"));
    ch1_down = new QPushButton(_("Down"), ch1_frame);
    ch1_down->setPixmap(pixmapcache->get("down.png"));
    ch1_jump = new QPushButton(_("Jump"), ch1_frame);
    ch1_jump->setPixmap(pixmapcache->get("jump.png"));
    ch1_del = new QPushButton(_("Down"), ch1_frame);
    ch1_del->setPixmap(pixmapcache->get("action_remove.png"));
    ch1_clear = new QPushButton(_("Jump"), ch1_frame);
    ch1_clear->setPixmap(pixmapcache->get("filenew.png"));
    
    ch1_grid->addMultiCellWidget(ch1_up, 2, 2, 1, 2);
    ch1_grid->addMultiCellWidget(ch1_down, 2, 2, 3, 4);
    ch1_grid->addWidget(ch1_jump, 2, 5);
    ch1_grid->addWidget(ch1_del, 2, 6);
    ch1_grid->addWidget(ch1_clear, 2, 7);
    
    ch1_prev  = new QPushButton(_("Previous"), ch1_frame);
    ch1_prev->setPixmap(pixmapcache->get("stock_media-prev.png"));
    ch1_rew   = new QPushButton(_("Rewind"), ch1_frame);
    ch1_rew->setPixmap(pixmapcache->get("stock_media-rew.png"));
    ch1_play  = new QPushButton(_("Play"), ch1_frame);
    ch1_play->setPixmap(pixmapcache->get("stock_media-play.png"));
    ch1_play->setToggleButton(true);
    ch1_pause = new QPushButton(_("Pause"), ch1_frame);
    ch1_pause->setPixmap(pixmapcache->get("stock_media-pause.png"));
    ch1_pause->setToggleButton(true);
    ch1_stop  = new QPushButton(_("Stop"), ch1_frame);
    ch1_stop->setPixmap(pixmapcache->get("stock_media-stop.png"));
    ch1_fwd   = new QPushButton(_("Forward"), ch1_frame);
    ch1_fwd->setPixmap(pixmapcache->get("stock_media-fwd.png"));
    ch1_next  = new QPushButton(_("Next"), ch1_frame);
    ch1_next->setPixmap(pixmapcache->get("stock_media-next.png"));
    
    ch1_grid->addWidget(ch1_prev, 3, 1);
    ch1_grid->addWidget(ch1_rew, 3, 2);
    ch1_grid->addWidget(ch1_play, 3, 3);
    ch1_grid->addWidget(ch1_pause, 3, 4);
    ch1_grid->addWidget(ch1_stop, 3, 5);
    ch1_grid->addWidget(ch1_fwd, 3, 6);
    ch1_grid->addWidget(ch1_next, 3, 7);
    
    grid->addWidget(ch1_frame, 0, 0);
    
    // channel 2

    QGroupBox *ch2_frame = new QGroupBox(_("Channel 2"), this);
    QGridLayout *ch2_grid = new QGridLayout(ch2_frame, 5, 9);
    ch2_grid->addColSpacing(0, lineSpace / 2);
    ch2_grid->addColSpacing(8, lineSpace / 2);
    ch2_grid->addRowSpacing(0, lineSpace);
    ch2_grid->addRowSpacing(4, lineSpace / 2);
    ch2_grid->setColStretch(1, 1);
    ch2_grid->setColStretch(2, 1);
    ch2_grid->setColStretch(3, 1);
    ch2_grid->setColStretch(4, 1);
    ch2_grid->setColStretch(5, 1);
    ch2_grid->setColStretch(6, 1);
    ch2_grid->setColStretch(7, 1);
    
    ch2_view = new ExtendedListView(ch2_frame);
    ch2_view->addColumn("", 16);
    ch2_view->addColumn(pixmapcache->getFadeIconSet("stock_music-library.png"), _("Tracks in Playlist"), -1);
    
    ch2_view->setItemMargin(0);
    ch2_view->setSelectionMode(QListView::Single);
    ch2_view->setSorting(-1);
    ch2_view->setAllColumnsShowFocus(true);
  
    ch2_grid->addMultiCellWidget(ch2_view, 1, 1, 1, 7);
    
    ch2_up   = new QPushButton(_("Up"), ch2_frame);
    ch2_up->setPixmap(pixmapcache->get("up.png"));
    ch2_down = new QPushButton(_("Down"), ch2_frame);
    ch2_down->setPixmap(pixmapcache->get("down.png"));
    ch2_jump = new QPushButton(_("Jump"), ch2_frame);
    ch2_jump->setPixmap(pixmapcache->get("jump.png"));
    ch2_del = new QPushButton(_("Down"), ch2_frame);
    ch2_del->setPixmap(pixmapcache->get("action_remove.png"));
    ch2_clear = new QPushButton(_("Jump"), ch2_frame);
    ch2_clear->setPixmap(pixmapcache->get("filenew.png"));
    
    ch2_grid->addMultiCellWidget(ch2_up, 2, 2, 1, 2);
    ch2_grid->addMultiCellWidget(ch2_down, 2, 2, 3, 4);
    ch2_grid->addWidget(ch2_jump, 2, 5);
    ch2_grid->addWidget(ch2_del, 2, 6);
    ch2_grid->addWidget(ch2_clear, 2, 7);
    
    ch2_prev  = new QPushButton(_("Previous"), ch2_frame);
    ch2_prev->setPixmap(pixmapcache->get("/stock_media-prev.png"));
    ch2_rew   = new QPushButton(_("Rewind"), ch2_frame);
    ch2_rew->setPixmap(pixmapcache->get("stock_media-rew.png"));
    ch2_play  = new QPushButton(_("Play"), ch2_frame);
    ch2_play->setPixmap(pixmapcache->get("stock_media-play.png"));
    ch2_play->setToggleButton(true);
    ch2_pause = new QPushButton(_("Pause"), ch2_frame);
    ch2_pause->setPixmap(pixmapcache->get("stock_media-pause.png"));
    ch2_pause->setToggleButton(true);
    ch2_stop  = new QPushButton(_("Stop"), ch2_frame);
    ch2_stop->setPixmap(pixmapcache->get("stock_media-stop.png"));
    ch2_fwd   = new QPushButton(_("Forward"), ch2_frame);
    ch2_fwd->setPixmap(pixmapcache->get("stock_media-fwd.png"));
    ch2_next  = new QPushButton(_("Next"), ch2_frame);
    ch2_next->setPixmap(pixmapcache->get("stock_media-next.png"));
    
    ch2_grid->addWidget(ch2_prev, 3, 1);
    ch2_grid->addWidget(ch2_rew, 3, 2);
    ch2_grid->addWidget(ch2_play, 3, 3);
    ch2_grid->addWidget(ch2_pause, 3, 4);
    ch2_grid->addWidget(ch2_stop, 3, 5);
    ch2_grid->addWidget(ch2_fwd, 3, 6);
    ch2_grid->addWidget(ch2_next, 3, 7);
    
    grid->addWidget(ch2_frame, 0, 2);
    
    // alternate play
    
    QGroupBox *alternate_frame = new QGroupBox(_("Alternate Play"), this);
    QGridLayout *alternate_grid = new QGridLayout(alternate_frame, 3, 6);
    alternate_grid->addColSpacing(0, lineSpace / 2);
    alternate_grid->addColSpacing(5, lineSpace / 2);
    alternate_grid->addRowSpacing(0, lineSpace);
    alternate_grid->addRowSpacing(2, lineSpace / 2);
    
    alternate_play  = new QPushButton(_("Play"), alternate_frame);
    alternate_play->setPixmap(pixmapcache->get("stock_media-play.png"));
    alternate_play->setToggleButton(true);
    alternate_pause = new QPushButton(_("Pause"), alternate_frame);
    alternate_pause->setPixmap(pixmapcache->get("stock_media-pause.png"));
    alternate_pause->setToggleButton(true);
    alternate_stop  = new QPushButton(_("Stop"), alternate_frame);
    alternate_stop->setPixmap(pixmapcache->get("stock_media-stop.png"));
    
    autoCrossFading = new QCheckBox(_("Auto Cross Fading"), alternate_frame);
    
    alternate_grid->addWidget(alternate_play, 1, 1);
    alternate_grid->addWidget(alternate_pause, 1, 2);
    alternate_grid->addWidget(alternate_stop, 1, 3);
    alternate_grid->addWidget(autoCrossFading, 1, 4);
    
    grid->addMultiCellWidget(alternate_frame, 1, 1 , 0, 2);
    
    connect(ch1_view, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
      this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
    connect(ch1_view, SIGNAL(doubleClicked(QListViewItem*)),
      this, SLOT(slot_doubleClicked(QListViewItem*)));
    connect(ch2_view, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
      this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
    connect(ch2_view, SIGNAL(doubleClicked(QListViewItem*)),
      this, SLOT(slot_doubleClicked(QListViewItem*)));
    connect(ch1_play, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_play()));
    connect(ch1_pause, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_pause()));
    connect(ch1_stop, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_stop()));
    connect(ch1_prev, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_prev()));
    connect(ch1_next, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_next()));
    
    connect(ch2_play, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_play()));
    connect(ch2_pause, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_pause()));
    connect(ch2_stop, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_stop()));
    connect(ch2_prev, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_prev()));
    connect(ch2_next, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_next()));

    connect(ch1_jump, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_jump()));
    connect(ch1_del, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_del()));
    connect(ch1_clear, SIGNAL(clicked()),
      this, SLOT(slot_button_ch1_clear()));
    
    connect(ch2_jump, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_jump()));
    connect(ch2_del, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_del()));
    connect(ch2_clear, SIGNAL(clicked()),
      this, SLOT(slot_button_ch2_clear()));
    
    connect(alternate_play, SIGNAL(clicked()),
      this, SLOT(slot_button_alternate_play()));
    connect(alternate_pause, SIGNAL(clicked()),
      this, SLOT(slot_button_alternate_pause()));
    connect(alternate_stop, SIGNAL(clicked()),
      this, SLOT(slot_button_alternate_stop()));
    
    connect(ch1_up, SIGNAL(clicked()),
      this, SLOT(slot_movePlaylistTrackUpCh1()));
    connect(ch2_up, SIGNAL(clicked()),
      this, SLOT(slot_movePlaylistTrackUpCh2()));
    connect(ch1_down, SIGNAL(clicked()),
      this, SLOT(slot_movePlaylistTrackDownCh1()));
    connect(ch2_down, SIGNAL(clicked()),
      this, SLOT(slot_movePlaylistTrackDownCh2()));
    
    play_lvi.resize(2);
    next_lvi.resize(2);
    ch_view.resize(2);
    ch_view.insert(0, ch1_view);
    ch_view.insert(1, ch2_view);
    
    isConnected      = false;
    isMixxxConnected = false;
    isPaused[0]      = false;
    isPaused[1]      = false;
    
    setMixxxConnectionState(false);
}

void GUI_MixxxQueue::slot_setConnectionState(bool state)
{
    isConnected = state;
}

void GUI_MixxxQueue::setMixxxConnectionState(bool state)
{
    isMixxxConnected = state;
    turnOnChannelControls(state);
    turnOnAlternatePlayControls(state);
    
    if(state) {
        mixxxClient->registerListener(this);
    }
        
}

void GUI_MixxxQueue::enqueueAtChannel(QList<LVI_Track> tracks, int channel)
{
    LVI_Track *lvi_track;
    for ( lvi_track = tracks.first(); lvi_track; lvi_track = tracks.next() ) {
        TRACK *track = lvi_track->getOriginalTrack();
        PLAYLIST_TRACK * p_track = getPlaylistTrackFromTrack(track);
        LVI_PlaylistTrack *lvi_p_track;
        lvi_p_track = new LVI_PlaylistTrack(ch_view.at(channel - 1), p_track, p_track->medium_id);
        
        if(lvi_p_track->itemPos() == 0) {
            lvi_p_track->adjustMixxxStatus(2);
            next_lvi.insert(channel - 1, lvi_p_track);
        }
        
        lvi_p_track->adjustIcon();
    }
}

void GUI_MixxxQueue::enqueueAtChannel(QList<LVI_PlaylistTrack> tracks, int channel)
{
    LVI_PlaylistTrack *p_track;
    for ( p_track = tracks.first(); p_track; p_track = tracks.next() ) {
        enqueueAtChannel(p_track, channel);
    }
}

void GUI_MixxxQueue::enqueueAtChannel(LVI_PlaylistTrack* track, int channel)
{
    ExtendedListView *view = ch_view.at(channel -1);
    // clone the ListViewItem, otherwise it will be removed from it's original parent!
    LVI_PlaylistTrack *lvi = new LVI_PlaylistTrack(view, track->getPlaylistTrack(), track->getMediumType());
    lvi->adjustIcon();
    view->insertItem(lvi);
    if(view->childCount() > 1) {
        lvi->moveItem( view->lastItem() );
    } else {
        lvi->adjustMixxxStatus(2);
        next_lvi.insert(channel -1, lvi);
    }
}

void GUI_MixxxQueue::notifyFinished(int channel)
{
    if(alternate_play->isOn()) {
        int otherChannel = (channel % 2) + 1;
        if(next_lvi.at(otherChannel - 1)) {
            mixxxClient->continueAtChannel(otherChannel);
            play_lvi.insert(otherChannel - 1, next_lvi.at(otherChannel - 1));
            play_lvi.at(otherChannel - 1)->adjustMixxxStatus(1);
            if(next_lvi.at(otherChannel - 1)->itemBelow()) {
                next_lvi.insert(otherChannel - 1, dynamic_cast<LVI_PlaylistTrack*>(next_lvi.at(otherChannel - 1)->itemBelow()));
                next_lvi.at(otherChannel - 1)->adjustMixxxStatus(2);
            } else {
                next_lvi.remove(otherChannel - 1);
            }
            
            play_lvi.at(channel - 1)->adjustMixxxStatus(0);
            play_lvi.remove(channel - 1);
            preloadChannel(channel);
        } else if(next_lvi.at(channel - 1)) {
            button_next(channel);
        } else {
            slot_button_ch1_stop();
            slot_button_ch2_stop();
        }
    } else if(play_lvi.at(channel - 1)) {
        if(next_lvi.at(channel - 1)) {
            if(channel == 1) slot_button_ch1_next();
            else             slot_button_ch2_next();
        } else {
            if(channel == 1) slot_button_ch1_stop();
            else             slot_button_ch2_stop();
        }
    }
}

QString GUI_MixxxQueue::getFile(LVI_PlaylistTrack *lvi)
{
    QString file = NULL;
    
    PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
//    if(lvi->getIsAvailable()) {
        file = track->path + "/" + track->filename;
//    } else {
//        QMessageBox::information(this, "Message", _("File(s) not available for playing"), QMessageBox::Ok);
//    }
    
    return file;
}

void GUI_MixxxQueue::preloadChannel(int channel)
{
    if(next_lvi.at(channel - 1)) {
        QString file = getFile(next_lvi.at(channel - 1));
        if(file.ascii()) {
            mixxxClient->loadAtChannel(file, channel);
        }
    }
}

void GUI_MixxxQueue::turnOnChannelControls(bool state)
{
    ch1_play->setEnabled(state);
    ch1_pause->setEnabled(state);
    ch1_stop->setEnabled(state);
    ch1_fwd->setEnabled(state);
    ch1_rew->setEnabled(state);
    ch1_next->setEnabled(state);
    ch1_prev->setEnabled(state);
    ch2_play->setEnabled(state);
    ch2_pause->setEnabled(state);
    ch2_stop->setEnabled(state);
    ch2_fwd->setEnabled(state);
    ch2_rew->setEnabled(state);
    ch2_next->setEnabled(state);
    ch2_prev->setEnabled(state);
    
    // currently not supported by mixxx
    ch1_fwd->setEnabled(false);
    ch1_rew->setEnabled(false);
    ch2_fwd->setEnabled(false);
    ch2_rew->setEnabled(false);
}

void GUI_MixxxQueue::turnOnAlternatePlayControls(bool state)
{
    alternate_play->setEnabled(state);
    alternate_pause->setEnabled(state);
    alternate_stop->setEnabled(state);
    
    // currently not supported by mixxx
    autoCrossFading->setEnabled(false);
}

// ##############################################
// # callbacks for the buttons
// ##############################################
void GUI_MixxxQueue::slot_movePlaylistTrackUpCh1()
{
    movePlaylistTrackUp(ch1_view);
}

void GUI_MixxxQueue::slot_movePlaylistTrackUpCh2()
{
    movePlaylistTrackUp(ch2_view);
}

void GUI_MixxxQueue::movePlaylistTrackUp(QListView* view)
{
    QListViewItem *item;
    item = view->currentItem();
    if(item == 0) return;
    if(item == view->firstChild()) return;
     
    if(item->itemAbove() == view->firstChild()) {
        view->firstChild()->moveItem(item);
    } else {
        item->moveItem(item->itemAbove()->itemAbove());
    }
    
    adjustPlayNext(item);
    view->ensureItemVisible(item);
}

void GUI_MixxxQueue::slot_movePlaylistTrackDownCh1()
{
    movePlaylistTrackDown(ch1_view);
}

void GUI_MixxxQueue::slot_movePlaylistTrackDownCh2()
{
    movePlaylistTrackDown(ch2_view);
}

void GUI_MixxxQueue::movePlaylistTrackDown(QListView* view)
{
    QListViewItem *item;
    item = view->currentItem();
    if(item == 0) return;
    if(item->itemBelow() == 0) return;
    
    item->moveItem(item->itemBelow());
    
    adjustPlayNext(item);
    view->ensureItemVisible(item);
}

void GUI_MixxxQueue::adjustPlayNext(QListViewItem* item)
{
    for(int i = 1; i >= 0; --i) {
        if((item == next_lvi.at(i) && play_lvi.at(i)) ||
           (item == play_lvi.at(i) && item->itemBelow()) ||
           (item->itemAbove() && item->itemAbove() == play_lvi.at(i))) {
            if(next_lvi.at(i)) {
                next_lvi.at(i)->adjustMixxxStatus(0);
            }
            next_lvi.insert(i, dynamic_cast<LVI_PlaylistTrack*>(play_lvi.at(i)->itemBelow()));
            next_lvi.at(i)->adjustMixxxStatus(2);
        } else if(item == play_lvi.at(i) && next_lvi.at(i) && !item->itemBelow()) {
            next_lvi.at(i)->adjustMixxxStatus(0);
            next_lvi.remove(i);
        }
    }
}

void GUI_MixxxQueue::slot_button_ch1_jump()
{
    button_jump(1);
}

void GUI_MixxxQueue::slot_button_ch2_jump()
{
    button_jump(2);
}

void GUI_MixxxQueue::button_jump(int channel)
{
    LVI_PlaylistTrack *item = dynamic_cast<LVI_PlaylistTrack*>(ch_view.at(channel - 1)->currentItem());
    if(play_lvi.at(channel - 1)) {
        play_lvi.at(channel - 1)->adjustMixxxStatus(0);
        play_lvi.remove(channel - 1);
        if(next_lvi.at(channel - 1)) {
            next_lvi.at(channel - 1)->adjustMixxxStatus(0);
        }
        next_lvi.insert(channel - 1, item);
        mixxxClient->stopAtChannel(channel);
        if(channel == 1) slot_button_ch1_play();
        else             slot_button_ch2_play();
    } else if(next_lvi.at(channel - 1)) {
        next_lvi.at(channel - 1)->adjustMixxxStatus(0);
        next_lvi.insert(channel - 1, item);
        next_lvi.at(channel - 1)->adjustMixxxStatus(2);
    }
}

void GUI_MixxxQueue::slot_button_ch1_del()
{
    button_del(1);
}

void GUI_MixxxQueue::slot_button_ch2_del()
{
    button_del(2);
}

void GUI_MixxxQueue::button_del(int channel)
{
    LVI_PlaylistTrack *item = dynamic_cast<LVI_PlaylistTrack*>(ch_view.at(channel - 1)->currentItem());
    if(play_lvi.at(channel - 1) && item == play_lvi.at(channel - 1)) {
        // don't allow deletion of curently played track
        return;
    }
    if(next_lvi.at(channel - 1) && item == next_lvi.at(channel - 1)) {
        if(item->itemBelow()) {
            next_lvi.insert(channel - 1, dynamic_cast<LVI_PlaylistTrack*>(item->itemBelow()));
            next_lvi.at(channel - 1)->adjustMixxxStatus(2);
        } else {
            next_lvi.remove(channel - 1);
        }
    }
    ch_view.at(channel - 1)->takeItem(item);
}

void GUI_MixxxQueue::slot_button_ch1_clear()
{
    button_clear(1);
}

void GUI_MixxxQueue::slot_button_ch2_clear()
{
    button_clear(2);
}

void GUI_MixxxQueue::button_clear(int channel)
{
    ch_view.at(channel - 1)->clear();
}

void GUI_MixxxQueue::slot_button_ch1_play()
{
    if(next_lvi.at(0)) {
        button_play(1);
        ch1_play->setEnabled(false);
        ch1_pause->setEnabled(true);
        ch1_pause->setOn(false);
    } else {
        ch1_play->setOn(false);
    }
}

void GUI_MixxxQueue::slot_button_ch2_play()
{
    if(next_lvi.at(1)) {
        button_play(2);
        ch2_play->setEnabled(false);
        ch2_pause->setEnabled(true);
        ch2_pause->setOn(false);
    } else {
        ch2_play->setOn(false);
    }
}

void GUI_MixxxQueue::button_play(int channel)
{
    if(isPaused[channel - 1]) {
        mixxxClient->continueAtChannel(channel);
    } else {
        QString file = getFile(next_lvi.at(channel - 1));
        if(file.ascii()) {
            mixxxClient->loadAndPlayAtChannel(file, channel);
        }
    }
    play_lvi.insert(channel - 1, next_lvi.at(channel - 1));
    play_lvi.at(channel - 1)->adjustMixxxStatus(1);
    if(next_lvi.at(channel - 1)->itemBelow()) {
        next_lvi.insert(channel - 1, dynamic_cast<LVI_PlaylistTrack*>(next_lvi.at(channel - 1)->itemBelow()));
        next_lvi.at(channel - 1)->adjustMixxxStatus(2);
    } else {
        next_lvi.remove(channel - 1);
    }
}

void GUI_MixxxQueue::slot_button_ch1_stop()
{
    ch1_play->setOn(false);
    ch1_play->setEnabled(true);
    ch1_pause->setOn(false);
    ch1_pause->setEnabled(true);
    
    button_stop(1);
}

void GUI_MixxxQueue::slot_button_ch2_stop()
{
    ch2_play->setOn(false);
    ch2_play->setEnabled(true);
    ch2_pause->setOn(false);
    ch2_pause->setEnabled(true);
    
    button_stop(2);
}

void GUI_MixxxQueue::button_stop(int channel)
{
    mixxxClient->stopAtChannel(channel);
    isPaused[channel - 1] = false;
    if(play_lvi.at(channel - 1)) {
        play_lvi.at(channel - 1)->adjustMixxxStatus(0);
        play_lvi.remove(channel - 1);
    }
    if(next_lvi.at(channel - 1)) {
        next_lvi.at(channel - 1)->adjustMixxxStatus(0);
        next_lvi.remove(channel - 1);
    }
    if(ch_view.at(channel - 1)->firstChild()) {
        next_lvi.insert(channel - 1, dynamic_cast<LVI_PlaylistTrack*>(ch_view.at(channel - 1)->firstChild()));
        next_lvi.at(channel - 1)->adjustMixxxStatus(2);
    }
}

void GUI_MixxxQueue::slot_button_ch1_pause()
{
    if(play_lvi.at(0)) {
        button_pause(1);
        ch1_pause->setEnabled(false);
        ch1_play->setOn(false);
        ch1_play->setEnabled(true);
    } else {
        ch1_pause->setOn(false);
    }
}

void GUI_MixxxQueue::slot_button_ch2_pause()
{
    if(play_lvi.at(1)) {
        button_pause(2);
        ch2_pause->setEnabled(false);
        ch2_play->setOn(false);
        ch2_play->setEnabled(true);
    } else {
        ch2_pause->setOn(false);
    }
}

void GUI_MixxxQueue::button_pause(int channel)
{
    mixxxClient->pauseAtChannel(channel);
    isPaused[channel - 1] = true;
    if(next_lvi.at(channel - 1)) {
        next_lvi.at(channel - 1)->adjustMixxxStatus(0);
    }
    next_lvi.insert(channel - 1, play_lvi.at(channel - 1));
    play_lvi.remove(channel - 1);
    next_lvi.at(channel - 1)->adjustMixxxStatus(2);
}

void GUI_MixxxQueue::slot_button_ch1_fwd()
{
}

void GUI_MixxxQueue::slot_button_ch2_fwd()
{
}

void GUI_MixxxQueue::slot_button_ch1_rew()
{
}

void GUI_MixxxQueue::slot_button_ch2_rew()
{
}

void GUI_MixxxQueue::slot_button_ch1_next()
{
    button_next(1);
}

void GUI_MixxxQueue::slot_button_ch2_next()
{
    button_next(2);
}

void GUI_MixxxQueue::button_next(int channel)
{
    if(play_lvi.at(channel - 1) && next_lvi.at(channel - 1)) {
        play_lvi.at(channel - 1)->adjustMixxxStatus(0);
        mixxxClient->stopAtChannel(channel);
        if(channel == 1) slot_button_ch1_play();
        else             slot_button_ch2_play();
        ch_view.at(channel - 1)->ensureItemVisible(play_lvi.at(channel - 1));
    }
}

void GUI_MixxxQueue::slot_button_ch1_prev()
{
    button_prev(1);
}

void GUI_MixxxQueue::slot_button_ch2_prev()
{
    button_prev(2);
}

void GUI_MixxxQueue::button_prev(int channel)
{
    if(play_lvi.at(channel - 1) && play_lvi.at(channel - 1)->itemPos() != 0) {
        play_lvi.at(channel - 1)->adjustMixxxStatus(0);
        if(next_lvi.at(channel - 1)) {
            next_lvi.at(channel - 1)->adjustMixxxStatus(0);
        }
        next_lvi.insert(channel - 1, dynamic_cast<LVI_PlaylistTrack*>(play_lvi.at(channel - 1)->itemAbove()));
        play_lvi.remove(channel - 1);
        mixxxClient->stopAtChannel(channel);
        if(channel == 1) slot_button_ch1_play();
        else             slot_button_ch2_play();
        ch_view.at(channel - 1)->ensureItemVisible(play_lvi.at(channel - 1));
    }
}

void GUI_MixxxQueue:: slot_button_alternate_play()
{
    if((next_lvi.at(0) || next_lvi.at(1)) &&
      !(play_lvi.at(0) && play_lvi.at(1))) {
        
        if(!play_lvi.at(0) && !play_lvi.at(1)) { // both channels are stoped
            if(isPaused[0]) {
                button_play(1);
            } else if(next_lvi.at(0)) {
                button_play(1);
                preloadChannel(2);
            } else {
                button_play(2);
            }
        } else if(play_lvi.at(0)) { // only channel 1 is playing
            preloadChannel(2);
        } else { // only channel 2 is playing
            preloadChannel(1);
        }
        
        isPaused[0] = false;
        isPaused[1] = false;
        
        alternate_pause->setOn(false);
        alternate_play->setEnabled(false);
        alternate_pause->setEnabled(true);
        turnOnChannelControls(false);
    } else {
        alternate_play->setOn(false);
    }
}

void GUI_MixxxQueue:: slot_button_alternate_pause()
{
    if(!alternate_play->isOn()) {
        alternate_pause->setOn(false);
        return;
    }
    
    if(play_lvi.at(0)) { // channel 1 is playing
        button_pause(1);
    } else if (play_lvi.at(1)) { // channel 2 is playing
        button_pause(2);
    }
    
    alternate_play->setOn(false);
    alternate_play->setEnabled(true);
    alternate_pause->setEnabled(false);
}

void GUI_MixxxQueue:: slot_button_alternate_stop()
{
    isPaused[0] = false;
    isPaused[1] = false;
    
    alternate_play->setOn(false);
    alternate_pause->setOn(false);
    alternate_play->setEnabled(true);
    alternate_pause->setEnabled(true);
    turnOnChannelControls(true);
    
    slot_button_ch1_stop();
    slot_button_ch2_stop();
}

void GUI_MixxxQueue:: slot_doubleClicked(QListViewItem*)
{
}

void GUI_MixxxQueue:: slot_contextMenu(QListViewItem *item, const QPoint &point, int id)
{
}

GUI_MixxxQueue::~GUI_MixxxQueue()
{
}
