/* prismstumbler.c - Utility Scan for 802_11 networks 
 * 
 * This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA). 
 * The linux driver for these cards can be found on www.linux-wlan.com
 * It has been verified with a D-LINK DWL-650 WL100 IEEE 802.11 adapter.
 * Created by: Jan Fernquist  Jan.B.Fernquist@telia.com
 * 
 *  The main idea for this program create a NetStumbler like utlity to find W-LANS
 *
 * Base code was from PrismDump.
 * Code also Snarfed from Ethreal and airsnort
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED

#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>
#include <poll.h>
#include <features.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>

#include "prismstumbler.h"
#include "fxgps.h"
#include "analyzer.h"
#include "iface-scan.h"
#include "iface-wlan-14.h"
#include "cards.h"

/*
 * How many packet should we read if we have a signal on a channel????
 * After that we force a channel change anyway
 */
#define CHANNEL_READ_COUNT 10

int g_offset = 0;
char *buf;
extern psconfig_t cfg;
extern unsigned int stop_sniffing;
extern int QUIET_MODE;
/*
 * Use globals, ugly but efficient
 */
int RawSock = -1;
ScanResult_t Res;
static int gps_active = 0;
static int gps_fix_ctr = 0;
static int scan_avail = 0;
FILE *WPFile = NULL;

static psnetinfo_t **netlist = NULL;
static int netcount = 0;
static usernetinfo_t usernet;

/*
 * Some forward refs
 */

static void send_config (int sock);


int getPacket (unsigned char *buf, int maxlen, int timeout);


/* checks if a string is not emty, valid and displayable */
inline int 
check_str_content(char *astr)
{
	int len,i;
	if (!astr) return FALSE;
	if ((len = strlen(astr)) == 0) return FALSE;
	if (!isalnum(astr[0])) return FALSE;
	for (i=0;i<len;i++){
		if (isalnum(astr[i]) &&  
			!isspace(astr[i])) return TRUE;
	}	
	return FALSE;
}


/* sends a command, here for command status reply */
static void
send_info (info_t info, char* message, int sock)
{
	psmessage_t msg;
	msg.type = msg_info;
	msg.content.info.info = info;
	snprintf(msg.content.info.message,100,"%s",message);

	if (write (sock, (void *) &msg, sizeof (psmessage_t)) < 0)
	{
		perror ("err sending info");
	}
}


void
send_gpsdata (float Long, float Lat, int q, int sock)
{
	psmessage_t msg;
	msg.type = msg_gps;
	msg.content.gps.Long = Long;
	msg.content.gps.Lat = Lat;
	msg.content.gps.quality = q;
	if (write (sock, (void *) &msg, sizeof (psmessage_t)) < 0)
	{
		perror ("err sending stream data");
	}
}


static void
send_config (int sock)
{
	psmessage_t msg;
	msg.type = msg_config;
	msg.content.cfg = cfg;
	if (write (sock, (void *) &msg, sizeof (psmessage_t)) < 0)
	{
		perror ("err sending config data");
	}
}


void
send_netlist (int sock, int all)
{
	psmessage_t msg;
	int i;
	for (i = 0; i < netcount; i++)
		if (netlist[i]->isvalid || all)
		{
			msg.type = msg_network;
			msg.content.net = *netlist[i];
			if (write (sock, (void *) &msg, sizeof (psmessage_t))
			    < 0)
			{
				perror ("err sending stream data");
			}
			/* reset signal levels */
			netlist[i]->isvalid = FALSE;
		}
}


/* connects to gpsd, init gps usage */
void
do_initgps(int ps_socket)
{
	float Long, Lat;
	
	if (gps_active) return;
	if (gps_initialize () > 0)
	{
		gps_active = 1;
#ifdef DEBUG
		fprintf (stderr, "GPSd available... trying to use. :-)\n");
#endif
		gps_fix_ctr = gps_get_xy (&Long, &Lat);
		if (gps_fix_ctr)
			gps_fix_ctr = 100;
		send_gpsdata (Long, Lat, gps_fix_ctr, ps_socket);
	}
	else /* init clients gps data with zeros */
	{
		send_gpsdata (0.0, 0.0, 0, ps_socket);
#ifdef DEBUG
		fprintf (stderr,
			 "Can't find woking GPSd... we'll have fun without it ;-)\n");
#endif
	}
}


/* checks connection status */
/* this will not work in ad-hoc mode */
static int
net_check_connection()
{
	FILE *f;
	int n = 0;
	char s[256], t[32], u[32];

	snprintf(t,32,"%s:",cfg.device);
	f = fopen ("/proc/net/wireless", "r");
	if (f == NULL)
		return FALSE;
	while (fgets (s, 255, f) != NULL)
	{
		if (sscanf (s, "%s %*d %d. %*d. %*d. %*d %*d %*d %*d %*d", u, &n)>1);
			if (strcmp (t, u) == 0)
				break;
	}
	fclose (f);
	if (strcmp (u, t) == 0)
	{
		if (!QUIET_MODE) printf("Link level dev %s %d\n",u,n);
		return(n);
	}
	else
		return FALSE;
}


/* bring down the configured interface 
 * useful helper function for frontends 
 */
void
do_interface_down(void)
{
	char syscmd[128];
	FILE *idf;
	
	snprintf (syscmd, 127, "/sbin/ifconfig %s down", cfg.device);
	system(syscmd);
	/* check if we are on a HP h5xxx device*/
	if (!access("/proc/hal/model",R_OK))
	{
		if ((idf = fopen("/proc/hal/model","r")))
		{
			fscanf(idf,"%10s",syscmd);
			fclose(idf);
			if (strstr(syscmd,"5400"))
			{
				snprintf (syscmd, 127, "/sbin/rmmod at76c503-rfmd usbvnetr");
				system(syscmd);
			}
		}
	}
}


/* try to connect to a given wireless lan */
static int
net_associate (int seqnr, int sock)
{
	char syscmd[128];
	if ((seqnr != SEQ_USERNET) && (seqnr >= netcount))
		return FALSE;	// illegal seqnr

	cfg.scan = FALSE;
	if (RawSock >= 0)
	{
		closePacket ();
		shutCard ();
	}

	sprintf (syscmd, "/sbin/ifconfig %s up", cfg.device);
#ifdef DEBUG
	printf ("exec: %s\n", syscmd);
#endif
	system (syscmd);

	if (seqnr == SEQ_USERNET)
	{
#ifdef DEBUG
	printf ("supplied net selected");
#endif
		if (usernet.mode) // ad-hoc
		{
			sprintf (syscmd, "/sbin/iwconfig %s mode ad-hoc",
				 cfg.device);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s channel %d", cfg.device,
				 usernet.channel);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s essid %s", cfg.device,
				 usernet.ssid);
			system (syscmd);
		}
		else		// managed
		{
			sprintf (syscmd, "/sbin/iwconfig %s mode managed",
				 cfg.device);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s essid %s", cfg.device,
				 usernet.ssid);
			system (syscmd);
		}
		if (usernet.wep)
		{
			sprintf (syscmd, "/sbin/iwconfig %s key %s", cfg.device, usernet.wep_key);
			system (syscmd);
		}
		/* in the case that we have no ip set, we use dhcp too */
		if ((usernet.dhcp) || !usernet.ip[0]) 
		{
			sprintf (syscmd, dhcpcommands[cfg.dhcpcommand], cfg.device);
			system (syscmd);
		}
		else
		{
			sprintf (syscmd, "/sbin/ifconfig %s %d.%d.%d.%d netmask %d.%d.%d.%d", cfg.device, usernet.ip[0],usernet.ip[1],usernet.ip[2],usernet.ip[3],
			usernet.netmask[0],usernet.netmask[1],usernet.netmask[2],usernet.netmask[3]);
			system (syscmd);
			sprintf (syscmd, "route del default");
			system (syscmd);
			if (usernet.gateway[0])
			{
				sprintf (syscmd, "route add default gw %d.%d.%d.%d",usernet.gateway[0],usernet.gateway[1],usernet.gateway[2],usernet.gateway[3]);
				system (syscmd);
			}
		}
		if (usernet.nameserver[0])
		{
			sprintf (syscmd, "/usr/bin/gpe-conf task_nameserver %d.%d.%d.%d",usernet.nameserver[0],usernet.nameserver[1],usernet.nameserver[2],usernet.nameserver[3]);
			system (syscmd);
		}
		
	}
	else
	{
		if (netlist[seqnr]->isadhoc)  // ad-hoc mode
		{
			sprintf (syscmd, "/sbin/iwconfig %s mode ad-hoc",
				 cfg.device);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s channel %d", cfg.device,
				 netlist[seqnr]->channel);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s essid %s", cfg.device,
				 netlist[seqnr]->ssid);
			system (syscmd);
		}
		else		// managed
		{
			sprintf (syscmd, "/sbin/iwconfig %s mode managed",
				 cfg.device);
			system (syscmd);
			sprintf (syscmd, "/sbin/iwconfig %s essid %s", cfg.device,
				 netlist[seqnr]->ssid);
			system (syscmd);
		}
		if (netlist[seqnr]->dhcp)
		{
			sprintf (syscmd, dhcpcommands[cfg.dhcpcommand], cfg.device);
			system (syscmd);
		}
	}
	
	sleep (1);

	/* check connection and return success here */
	if (net_check_connection())
		send_info(I_SUCCESS,"Successfully connected to net.",sock);
	else
		send_info(I_FAILED,"Connecting failed.",sock);
	return TRUE;
}


static int
detect_driver (int sock)
{
	const char *stabfile;
	FILE *f;
	char drv[20];
	char device[20];
	int old_devtype, i, j;
	int known = FALSE;
	char s[80], *t, d[80], type[20];
	int have_driver = 0x00;
	char dev_hostap[20];
	char dev_orinoco[20];
	char dev_prism[20];
	char dev_cisco[20];
	
	sprintf (device, "none");
	old_devtype = cfg.devtype;
#ifdef DEBUG
	printf ("detecting driver...\n");
#endif
	/* get driver name */
	if (access ("/var/lib/pcmcia", R_OK) == 0)
	{
		stabfile = "/var/lib/pcmcia/stab";
	}
	else
	{
		stabfile = "/var/run/stab";
	}

	f = fopen (stabfile, "r");
	if (f == NULL)
		return FALSE;
	if (flock (fileno (f), LOCK_SH) != 0)
	{
		fprintf (stderr, "locking stabfile failed: %s",
			 strerror (errno));
		return FALSE;
	}

	fgetc (f);
	for (i = 0; i < 2; i++)
	{
		if (!fgets (s, 80, f))
			break;
		s[strlen (s) - 1] = '\0';
		*d = '\0';
		*type = '\0';
		snprintf (drv, 20, "none");
		for (;;)
		{
			int c = fgetc (f);
			if ((c == EOF) || (c == 'S'))
			{
#ifdef DEBUG
				printf ("driver loaded: %s conf: %s\n", d,
					cfg.device);
#endif				
				if (!strcmp (drv, "orinoco_cs") || !strcmp (drv,"spectrum_cs"))
				{
					have_driver += DT_ORINOCO;
					sprintf (dev_orinoco, "%s", d);
				}
				if (!strcmp (drv, "hostap_cs"))
				{
					have_driver += DT_HOSTAP;
					sprintf (dev_hostap, "%s", d);
				}
				if (!strcmp (drv, "prism2_cs"))
				{
					have_driver += DT_PRISM;
					sprintf (dev_prism, "%s", d);
				}
				if (strstr (drv, "airo"))
				{
					have_driver += DT_CISCO;
					sprintf (dev_cisco, "%s", d);
				}
				if (!strcmp (cfg.device, d))
				{
					i = 3;
				}
				break;
			}
			else
			{
				fgets (s, 80, f);
				t = s;
				t = strchr (t, '\t') + 1;
				snprintf (type,
					  strcspn (t, "\t\n") + 1, "%s", t);
				t = strchr (t, '\t') + 1;
				snprintf (drv,
					  strcspn (t, "\t\n") + 1, "%s", t);
				for (j = 0; j < 2; j++)
					t = strchr (t, '\t') + 1;
				t[strcspn (t, "\t\n")] = '\0';
				if (*d == '\0')
					strcpy (d, t);
				else
				{
					strcat (d, ", ");
					strcat (d, t);
				}
			}
		}
	}

	flock (fileno (f), LOCK_UN);
	fclose (f);

#ifdef DEBUG
	printf ("detected driver for configured card: %s\n", drv);
#endif
	/* check if we know the configured interface */

	if (!strcmp (drv, "orinoco_cs") || !strcmp (drv,"spectrum_cs"))
	{
		cfg.devtype = DT_ORINOCO;
		known = TRUE;
	}

	if (!strcmp (drv, "hostap_cs"))
	{
		cfg.devtype = DT_HOSTAP;
		known = TRUE;
	}

	if (!strcmp (drv, "prism2_cs"))
	{
		cfg.devtype = DT_PRISM;
		known = TRUE;
	}
	
	if (strstr (drv, "airo"))
	{
		cfg.devtype = DT_CISCO;
		known = TRUE;
	}
	
	if (known)
	{
		if (RawSock >= 0)
		{
			closePacket ();
			shutCard ();
		}
		if ((RawSock = openPacket ()) < 0)
		{
			fprintf (stderr,
				 "Detection failed: Can't open pcap device: %s",
				 strerror (errno));
			return FALSE;
		}
		else
		{
			closePacket ();
			shutCard ();
		}
		return TRUE;
	}
	else			/* try other detected interfaces */
	{
		if (have_driver > 0)
		{
			if (have_driver & DT_PRISM)
			{
				cfg.devtype = DT_PRISM;
				snprintf (cfg.device, 6, "%s", dev_prism);
			}
			if (have_driver & DT_ORINOCO)
			{
				cfg.devtype = DT_ORINOCO;
				snprintf (cfg.device, 6, "%s", dev_orinoco);
			}
			if (have_driver & DT_HOSTAP)
			{
				cfg.devtype = DT_HOSTAP;
				snprintf (cfg.device, 6, "%s", dev_hostap);
			}
			if (have_driver & DT_CISCO)
			{
				cfg.devtype = DT_CISCO;
				snprintf (cfg.device, 6, "%s", dev_cisco);
			}
#ifdef DEBUG
			printf ("autoselected interface: %s\n", cfg.device);
#endif			
			if (RawSock >= 0)
			{
				closePacket ();
				shutCard ();
			}
			if ((RawSock = openPacket ()) < 0)
			{
				fprintf (stderr,
					 "Detection failed: Can't open pcap device: %s",
					 strerror (errno));
				send_info(I_ERRCARD,"Error opening wireless LAN device.",sock);
				return FALSE;
			}
			else
			{
				closePacket ();
				shutCard ();
			}
			return TRUE;
		}

		else	/* probe scan on default devices */
		{
			int found_sc = FALSE;
			
			GSList *iter, *devs = find_wireless_cards();
			
			for (iter = devs; iter; iter = iter->next)
			{
				struct ps_card *c = (struct ps_card*)iter->data;
				printf("scanning %s\n",c->name);
				if (do_scan_interface(c->name, sock, 0.0, 0.0))
				{
					cfg.devtype = DT_SCAN;
					sprintf(cfg.device, c->name);
					found_sc = TRUE;
					break;
				}
			}
			for (iter = devs; iter; iter = iter->next)
			{
				struct ps_card *c = (struct ps_card*)iter->data;
				g_free(c->name);
				g_free(c->type);
			}
			g_slist_free(devs);
			if (found_sc) 
				return TRUE;
			else /* now we really lost... */
			{
				// tell client that no card was detected
				send_info(I_NOCARD,
					"No wireless LAN card detected... giving up.",sock);
				fprintf (stderr,
					 "No wireless LAN card detected... giving up.\n");
				sprintf (cfg.device, "%s", "none");
			}
			return TRUE;
		}
	}
	return FALSE;
}


int
waypointsAddNet (char* bssid, char* ssid, float lo, float la)
{
	if (WPFile)
	{
		if (strlen (ssid) > 0)
			fprintf (WPFile, "%s \t%f \t%f\n", ssid, lo, la);
		else
			fprintf (WPFile, "%s \t%f \t%f\n", bssid, lo, la);
	}
	return 0;
}

int
update_netlist (ScanResult_t * ares)
{
	int i = 0;
	int rpos = 0;
	int found_net = FALSE;
	psnetinfo_t *new_net;


	for (rpos = 0; rpos < netcount; rpos++)
	{
		// update network data
		if (!strncmp (netlist[rpos]->bssid, ares->BssId, 17))
		{

			// lets hack a data-transfer
			found_net = TRUE;
			new_net = netlist[rpos];
			new_net->isvalid = TRUE;

			// in some cases so we get a hidden network ssid
			if (check_str_content (ares->SSID))
			{
				if (!check_str_content (new_net->ssid))
					new_net->ishidden = TRUE;
				strncpy (new_net->ssid, ares->SSID, 32);
			}

			if (ares->isAp > (atoi (new_net->ap)))
				sprintf (new_net->ap, "%i", ares->isAp);
			new_net->last = ares->when;
			new_net->psum++;
			new_net->pdata += ares->isData;
			new_net->pint += ares->hasIntIV;
			new_net->dhcp += ares->dhcp;
			new_net->isadhoc += ares->isAdHoc;
			if (ares->speed > new_net->speed)
				new_net->speed = ares->speed;

			sprintf (new_net->type, "%.4X", ares->FrameType);

			if (ares->hasWep > new_net->wep)
				new_net->wep = ares->hasWep;
			new_net->channel = ares->Channel;

			if (ares->protocol != IP_PROTO_NONE)
			{
				for (i = 0; i < new_net->pcount; i++)
					if (new_net->pvec[i] == ares->protocol)
						goto out;
				new_net->pvec[(int)new_net->pcount] =
					ares->protocol;
				new_net->pcount++;
				new_net->pvec[(int)new_net->pcount] = 0;	
				if ((ares->protocol == IP_PROTO_ESP)
				    || (ares->protocol == IP_PROTO_AH))
					new_net->ipsec = TRUE;
			}
			out:

			new_net->cursiglevel = ares->Signal;

			if (new_net->maxsiglevel <= ares->Signal)	// save only strongest siglevel+position
			{
				new_net->longitude = ares->longitude;
				new_net->latitude = ares->latitude;
				new_net->maxsiglevel = ares->Signal;
			}
			else if (!(new_net->longitude) && (ares->longitude))	// update position if no saved pos and pos now available
			{
				new_net->longitude = ares->longitude;
				new_net->latitude = ares->latitude;
			}

			if ((ares->subnet[0]) != 0)
			{
				new_net->ip_range[0] = ares->subnet[0];
				new_net->ip_range[1] = ares->subnet[1];
				new_net->ip_range[2] = ares->subnet[2];
				new_net->ip_range[3] = ares->subnet[3];
			}

			return rpos;
		}
	}

	// we have a new contact
	if ((!found_net) && ((!cfg.filter) || ares->isData))
	{
		newnet_count++;
		netcount++;
		netlist =
			realloc (netlist, netcount * sizeof (psnetinfo_t *));
		netlist[netcount - 1] = malloc (sizeof (psnetinfo_t));

		// add data
		new_net = netlist[netcount - 1];
		new_net->seqnr = netcount - 1;

		memset (new_net, 0, sizeof (psnetinfo_t));
		new_net->ishidden = FALSE;
		new_net->isvalid = TRUE;
		strncpy (new_net->ssid, ares->SSID, 32);
		strncpy (new_net->bssid, ares->BssId, 17);

		new_net->last = ares->when;
		new_net->first = ares->when;
		new_net->pint = 0;
		new_net->dhcp = 0;
		new_net->psum = 1;
		if (ares->isData)
			new_net->pdata = 1;
		if (ares->hasIntIV)
			new_net->pint = 1;
		new_net->isadhoc += ares->isAdHoc;

		sprintf (new_net->wep_key, "<unknown>");

		sprintf (new_net->type, "%.4X", ares->FrameType);
		sprintf (new_net->ap, "%i", ares->isAp);

		new_net->speed = ares->speed;
		new_net->wep = ares->hasWep;
		new_net->channel = ares->Channel;

		new_net->longitude = ares->longitude;
		new_net->latitude = ares->latitude;
		new_net->maxsiglevel = ares->Signal;
		new_net->cursiglevel = ares->Signal;

		if (ares->protocol != IP_PROTO_NONE)
		{
			if ((ares->protocol == IP_PROTO_ESP)
			    || (ares->protocol == IP_PROTO_AH))
				new_net->ipsec = TRUE;
			new_net->pcount = 1;
			new_net->pvec[0] = ares->protocol;
			new_net->pvec[1] = 0;
		}
		else
		{
			new_net->pcount = 0;
			new_net->pvec[0] = 0;
		}

		if (WPFile)
		{
			waypointsAddNet(ares->SSID, ares->BssId, ares->longitude, ares->latitude);
		}
		if ((ares->subnet[3]) == 0)
		{
			memcpy (new_net->ip_range, ares->subnet, 3);
		}
		else
			memset (&(new_net->ip_range), ' ', 4);

#ifdef DEBUG
		fprintf (stderr, "Found new net: \n %s %s \n", new_net->ssid,
			 new_net->bssid);
#endif
		return (netcount - 1);
	}
	return -1;
}


void
clear_netlist ()
{
	int i;
	for (i = 0; i < netcount; i++)
		free (netlist[i]);
	free (netlist);
	netlist = NULL;
	netcount = 0;
}


static void
update_config (psconfig_t * ncfg, int ps_socket)
{
	if (!ncfg->scan)
	{
		if (RawSock >= 0)
		{
			closePacket ();
			shutCard ();
		}
	}

	/* scanner turned on, device changed, driver changed */
	if ((ncfg->scan) || (strcmp (ncfg->device, cfg.device))
	    || (ncfg->devtype != cfg.devtype))
	{
		if (RawSock >= 0)
		{
			closePacket ();
			shutCard ();
		}
		
		snprintf (cfg.device, 6, ncfg->device);
		
		cfg.devtype = ncfg->devtype;
		cfg.scan = ncfg->scan;
				
		if ((RawSock = openPacket ()) < 0)
		{
			fprintf (stderr,"Can't open pcap device: %s",
				 strerror (errno));
			cfg.scan = FALSE;
		}
		if (cfg.devtype == DT_SCAN)
		{
			/* check scan mode, do single scan */
			scan_avail = do_scan_interface(cfg.device, ps_socket, 0.0, 0.0);
			if (!scan_avail)
			{
				fprintf (stderr,"Can't initiate scan on device: %s",
					 strerror (errno));
				cfg.scan = FALSE;
			}
		}
		else
		{
			if ((RawSock = openPacket ()) < 0)
			{
				fprintf (stderr,"Can't open pcap device: %s",
					 strerror (errno));
				cfg.scan = FALSE;
			}
			scan_avail = TRUE;
		}
	}
	else
	{
		cfg.scan = ncfg->scan;
	}
	
	cfg.delay = ncfg->delay;

	if (cfg.scan && (RawSock<0) && (cfg.devtype != DT_SCAN))
	{
		RawSock = openPacket();
		scan_avail = (RawSock >= 0);
	}
	
	cfg.filter = ncfg->filter;
	cfg.singlechan = ncfg->singlechan;
	if (strncmp (cfg.wpfile, ncfg->wpfile, 255))
	{
#ifdef DEBUG
		printf ("opening new wp file\n");
#endif
		snprintf (cfg.wpfile, 255, ncfg->wpfile);
		if (WPFile) fclose(WPFile);
		if (cfg.wpfile[0])
		{
			if ((WPFile = fopen (cfg.wpfile, "w")) == NULL)
			{
				fprintf (stderr,
					 "Cant open waypoint file %s for writing\n",
					 cfg.wpfile);
			}
		}
	}

	cfg.autosend = ncfg->autosend;
	
	if (ncfg->dumptofile) pcap_start_dump();
		else pcap_stop_dump();
	cfg.dumptofile = ncfg->dumptofile;
	
	if (strncmp (cfg.dumpfile, ncfg->dumpfile, 255))
	{
#ifdef DEBUG
		printf ("opening new dump file\n");
#endif
		pcap_stop_dump();
		snprintf (cfg.dumpfile, 255, ncfg->dumpfile);
		if (cfg.dumpfile[0])
			pcap_start_dump();
	}
	cfg.dhcpcommand = ncfg->dhcpcommand;
}

int
detect_dhcp(void)
{
	int i;
	char binary[128] = {0};
	
	for (i=0; i < DHCP_CMDCOUNT; i++)
	{
		sscanf(dhcpcommands[i], "%128s", binary);
		if (!access(binary, X_OK))
		{
			cfg.dhcpcommand = i;
			return TRUE;
		}
	}
	return 0;
}

void
do_command (pscommand_t * cmd, int sock)
{
	switch (cmd->command)
	{
	case C_SENDLIST:
		send_netlist (sock, TRUE);
		break;
	case C_CLEARLIST:
		clear_netlist ();
		break;
	case C_DETECT_CARD:
		if (detect_driver (sock))
			send_config (sock);
		break;
	case C_ASSOCIATE:
		net_associate (cmd->par, sock);
		break;
	case C_GPSD:
		do_initgps(sock);
		break;	
	case C_IFDOWN:
		do_interface_down();
		break;	
	case C_DETECT_DHCP:
		if (detect_dhcp ())
			send_config (sock);
		break;
	default:
		fprintf (stderr, "Received unknown command.\n");
		break;
	}
}


int
check_messages (int sock)
{
	psmessage_t msg;
	struct pollfd pfd[1];
	static int len;
	
	pfd[0].fd = sock;
	pfd[0].events = (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI);
	while (poll (pfd, 1, 0) > 0)
	{
		if ((pfd[0].revents & POLLERR) || (pfd[0].revents & POLLHUP))
		{
#ifdef DEBUG			
			perror ("Err: connection lost: ");
#endif
			return -1;
		}
		if ((len = read (sock, (void *) &msg, sizeof (psmessage_t))) < 0)
		{
			perror ("err receiving message packet");
			close (sock);
			return -1;
		}
		else
		if (len == sizeof(psmessage_t))
		{
			switch (msg.type)
			{
			case msg_config:
				update_config (&msg.content.cfg, sock);
				break;
			case msg_command:
				do_command (&msg.content.command, sock);
				break;
			case msg_usernet:
				usernet = msg.content.usernet;
				break;
			default:
				break;
			}
		}
	}
	return 0;
}


void
update_all (ScanResult_t * aresult, int sock)
{
	update_netlist (aresult);
}


void
psmain (int ps_socket)
{
	int recvlen = MAX_BUFFER_SIZE;
	int q;
	unsigned char msgbuf[MAX_BUFFER_SIZE];
	int Channel = 0;
	int MaxPackPerChannel;
	time_t now, last_time;
	int Time_Add;
	static float Long, Lat, long_old, lat_old;
	int scount = 1;

	/* do some nasty things, dhcpcd will bring down our interface */
	system("/usr/bin/killall dhcpcd > /dev/null 2>&1");
	
	/* set usernet data invalid */
	usernet.inrange = FALSE;

	do_initgps(ps_socket);
	
	/* set max priority */
	setpriority (PRIO_PROCESS, 0, 0);	// -20
	
	RawSock = openPacket ();

	while ((RawSock < 0) && !stop_sniffing && (!scan_avail))
	{
#ifdef DEBUG
		perror ("Waiting for config\n");
#endif
		sleep (1);
		if (check_messages (ps_socket) == -1)
		{
			perror ("Can't read config :-(\n");
			return;
		}
	}

	time (&last_time);
	Time_Add = 0;

	while (stop_sniffing == 0)
	{

		if (!cfg.singlechan)
		{
			Channel++;
			if (Channel > 14)
				Channel = 1;
		}
		else
		{
			Channel = cfg.singlechan;
		}

		if (cfg.scan)
		{
			if (selectChannel (Channel) != 0)
				break;
		}

		/* in scan mode one result is enough */
		if (cfg.devtype == DT_SCAN)
			MaxPackPerChannel = 1;
		else
			MaxPackPerChannel = CHANNEL_READ_COUNT;
		do
		{
			/* wait for more data */
			usleep (cfg.delay);
			// do we still know _where_ we are?
			if (gps_active)
			{
				q = gps_fix_ctr;
				long_old = Long;
				lat_old = Lat;
				if (gps_get_xy (&Lat, &Long))
					gps_fix_ctr = 100;
				else if (gps_fix_ctr > 0)
					gps_fix_ctr--;
				if (gps_fix_ctr == 0)	// no gps fix
				{
					Lat = 0.0;
					Long = 0.0;
				}
				if ((q != gps_fix_ctr) || (Long != long_old)
				    || (Lat != lat_old))
					send_gpsdata (Long, Lat, gps_fix_ctr,
						      ps_socket);
			}
			
			/* receive next package if in monitor mode, run scan otherwise */
			memset (msgbuf, 0, recvlen);
			if (cfg.scan)
			{
				if (cfg.devtype != DT_SCAN)
					recvlen = getPacket (msgbuf, MAX_BUFFER_SIZE,0);
				else
				{
					/* make sure we stop break after this */
					recvlen = 0; 
					/* it will handle updating itself */
					do_scan_interface(cfg.device, ps_socket, Long, Lat);
				}
			}
			
			if (recvlen < 0)	/* Error */
			{
				recvlen = 0;
				break;
			}
			if (recvlen == 0)
				break;

			/* Got A valid inetersting ? */
			if ((cfg.scan)
			    && (processPacket (msgbuf, recvlen, Channel, ps_socket)))
			{

				if (Res.Channel < 1 || Res.Channel > 14)
				{
					continue;	/* Bogus pkt */
				}
				time (&now);
				if (now == last_time)
				{
					Time_Add++;
				}
				else
				{
					last_time = now;
					Time_Add = 0;
				}

				/*
				 * Now transfer collected data...
				 */
				Res.longitude = Long;
				Res.latitude = Lat;
				Res.when = now;
				update_all (&Res, ps_socket);
			}

		}
		while (--MaxPackPerChannel > 0);	/* Until we read enough */

		scount--;

		if ((!scount) && !stop_sniffing)
		{
			if (check_messages (ps_socket) < 0)
				stop_sniffing = TRUE;
			else
			{
				if (cfg.autosend)
					send_netlist (ps_socket, FALSE);
			}
			scount = 10;
		}
	}			/*  while  still sniffing */

	if (RawSock >= 0)
	{
		closePacket ();
		shutCard ();
	}
#ifdef DEBUG
	printf("leaving...\n");
#endif
}
