/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *           Copyright (C) 2005 SUSE Linux Products GmbH                   *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <string.h>

#include "config.h"
#include "powersave_dbus.h"
#include "powerlib.h"

DBusConnection *connection = NULL;

int dbusInit(void)
{
	DBusError error;
	dbus_error_init(&error);

	if (connection != NULL) {
		return 1;
	}

	connection = dbus_connection_open_private(DBUS_SYSTEM_BUS_SOCKET, &error);

	if (connection == NULL || dbus_error_is_set(&error)) {
		pDebug(DBG_INFO, "connection to system message bus failed: %s", error.message);
		dbus_error_free(&error);
		return 0;
	}

	dbus_bus_register(connection, &error);

	if (dbus_error_is_set(&error)) {
		pDebug(DBG_ERR, "registering connection with system message bus failed: %s\n", error.message);
		dbusFree();
		return 0;
	}

	return 1;
}

void dbusFree(void)
{
	if (connection != NULL) {
		dbus_connection_close(connection);
		dbus_connection_unref(connection);
		connection = NULL;
	}
}	

/* Shall we implement a type unspecific getMessageGeneric() 
 * that is not visible to others and use it for getMessageString(),
 * getMessageIntger(), getMessageErrorID() ?
 */
int dbusGetMessageString(DBusMessage * msg, char **s, int no)
{

	int current_type;
	DBusMessageIter iter;
	int x = 0;
	int ret = -1;
	int _no = 0;

	for (dbus_message_iter_init(msg, &iter); (current_type = dbus_message_iter_get_arg_type(&iter))
	     != DBUS_TYPE_INVALID; x++, dbus_message_iter_next(&iter)) {
		if (current_type == DBUS_TYPE_STRING) {
			if (_no < no) {
				_no++;
				continue;
			} else if (_no > no)
				break;
			else {
				dbus_message_iter_get_basic(&iter, s);
				ret = 0;
				break;
			}
		}
	}
	return ret;
}

int dbusGetMessageInteger(DBusMessage * msg, int *i, int no)
{

	int current_type;
	DBusMessageIter iter;
	int x = 0;
	int ret = -1;
	int _no = 0;

	dbus_int32_t i_32;
	for (dbus_message_iter_init(msg, &iter); (current_type = dbus_message_iter_get_arg_type(&iter))
	     != DBUS_TYPE_INVALID; x++, dbus_message_iter_next(&iter)) {
		if (current_type == DBUS_TYPE_INT32) {
			if (_no < no) {
				_no++;
				continue;
			} else if (_no > no)
				break;
			else {
				dbus_message_iter_get_basic(&iter, &i_32);
				*i = i_32;
				ret = 0;
				break;
			}
		}
	}
	return ret;
}

int dbusGetMessageErrorID(DBusMessage * msg, unsigned *error_id)
{

	int current_type;
	DBusMessageIter iter;
	int x = 0;
	int ret = -1;

	/* Only one DBUS_TYPE_UINT16 value is allowed per message -> this
	   is our error ID.
	 */
	dbus_uint16_t err_id;
	for (dbus_message_iter_init(msg, &iter); (current_type = dbus_message_iter_get_arg_type(&iter))
	     != DBUS_TYPE_INVALID; x++, dbus_message_iter_next(&iter)) {
		if (current_type == DBUS_TYPE_UINT16) {
			dbus_message_iter_get_basic(&iter, &err_id);
			*error_id = err_id;
			ret = 0;
			break;
		}
	}
	return ret;
}

unsigned dbusSendMessageGeneric(int msg_type,
				DBusMessage ** reply, const char *method, int first_arg_type, va_list var_args)
{
	if (!dbusInit())
		return REPLY_DBUS_ERROR;

	// msg_type is a POWERSAVE_MESSAGE_TYPE (enum)
	int retval;
	unsigned ret = REPLY_GENERAL_ERROR;

	DBusError error;
	dbus_error_init(&error);
	DBusMessage *message;

	const char *interface;

	if (msg_type == ACTION_MESSAGE)
		interface = PS_DBUS_ACTION_INTERFACE;
	else if (msg_type == REQUEST_MESSAGE)
		interface = PS_DBUS_REQUEST_INTERFACE;
	else if (msg_type == MANAGER_MESSAGE)
		interface = PS_DBUS_MANAGER_INTERFACE;
	else if (msg_type == SCRIPTS_MESSAGE)
		interface = PS_DBUS_SCRIPTS_INTERFACE;
	else if (msg_type == ADMIN_MESSAGE)
		interface = PS_DBUS_ADMIN_INTERFACE;
	else {
		pDebug(DBG_ERR, "Invalid interface");
		return REPLY_INVALID_INTERFACE;
	}

	message = dbus_message_new_method_call(NULL, "/com/novell/powersave", interface, method);

	if (message == NULL) {
		pDebug(DBG_ERR, "Couldn't allocate D-BUS message");
		dbusFree();
		return REPLY_DBUS_ERROR;
	}
	if (!dbus_message_set_destination(message, PS_DBUS_SERVICE)) {
		pDebug(DBG_ERR, "Not enough memory");
		dbus_message_unref(message);
		dbusFree();
		return REPLY_DBUS_ERROR;
	}

	retval = dbus_message_append_args_valist(message, first_arg_type, var_args);
	if (!retval) {
		pDebug(DBG_ERR, "Could not generate message");
		dbus_message_unref(message);
		dbusFree();
		return REPLY_DBUS_ERROR;
	}
	/* reply timeout: 2sec */
	*reply = dbus_connection_send_with_reply_and_block(connection, message, 2000, &error);
	if (dbus_error_is_set(&error) || reply == NULL) {
		pDebug(DBG_ERR, "Error: %s", error.message);
		dbus_message_unref(message);
		dbusFree();
		if (!strcmp(error.name, DBUS_ERROR_ACCESS_DENIED))
			return REPLY_NO_RIGHTS;
		return REPLY_DBUS_ERROR;
	}

	/* get_args cannot handle order of appended types, we could not say for
	 * sure that the first appended value is really INT16 ... use iterators 
	 * in helper function ...
	 */
	dbusGetMessageErrorID(*reply, &ret);

	if (dbus_error_is_set(&error)) {
		pDebug(DBG_ERR, "DBus Error: %s", error.name);
		dbus_message_unref(message);
		dbusFree();
		return REPLY_DBUS_ERROR;
	}
	/* You must not access the reply struct in any kind if the return value of this function
	 * is not REPLY_SUCCESS !!!
	 * always test like:
	 *      if (ret != REPLY_SUCCESS)
	 *              dbus_message_unref(reply);
	 *      else {
	 *             // no need for dbus_message_unref
	 *      }
	 */

	dbus_message_unref(message);

	return ret;
}

unsigned dbusSendSimpleMessage(int msg_type, const char *method)
{
	int ret;
	ret = dbusSendMessage(msg_type, method, DBUS_TYPE_INVALID);
	return ret;
}

unsigned dbusSendSimpleMessageWithReply(int msg_type, DBusMessage ** reply, const char *method)
{

	return dbusSendMessageWithReply(msg_type, reply, method, DBUS_TYPE_INVALID);
}

unsigned dbusSendMessage(int msg_type, const char *method, int first_arg_type, ...)
{
	int ret;
	DBusMessage *reply;
	va_list var_args;
	va_start(var_args, first_arg_type);
	ret = dbusSendMessageGeneric(msg_type, &reply, method, first_arg_type, var_args);
	va_end(var_args);
	if (ret == REPLY_SUCCESS)
		dbus_message_unref(reply);
	return ret;
}

unsigned dbusSendMessageWithReply(int msg_type, DBusMessage ** reply, const char *method, int first_arg_type, ...)
{
	int ret;
	va_list var_args;
	va_start(var_args, first_arg_type);
	ret = dbusSendMessageGeneric(msg_type, reply, method, first_arg_type, var_args);
	va_end(var_args);
	return ret;
}

dbus_int32_t establishConnection(int capabilities, DBusConnection * connection)
{
	int retval;
	unsigned ret = REPLY_GENERAL_ERROR;
	DBusMessage *message, *reply;
	DBusError error;

	dbus_error_init(&error);

	dbus_int32_t cap_32 = capabilities;

	message = dbus_message_new_method_call(PS_DBUS_SERVICE, PS_DBUS_PATH, PS_DBUS_MANAGER_INTERFACE, "Connect");

	if (message == NULL) {
		pDebug(DBG_ERR, "Couldn't allocate D-BUS message");
		return REPLY_DBUS_ERROR;
	}

	retval = dbus_message_append_args(message, DBUS_TYPE_INT32, &cap_32, DBUS_TYPE_INVALID);

	if (!retval) {
		pDebug(DBG_ERR, "Could not generate message");
		dbus_message_unref(message);
		return REPLY_DBUS_ERROR;
	}

	/* reply timeout: 2sec */
	reply = dbus_connection_send_with_reply_and_block(connection, message, 2000, &error);
	if (dbus_error_is_set(&error) || reply == NULL) {
		pDebug(DBG_ERR, "Error: %s", error.message);
		dbus_error_free(&error);
		return REPLY_DBUS_ERROR;
	}

	/* get_args cannot handle order of appended types, we could not say for
	 * sure that the first appended value is really INT16 ... use iterators 
	 * in helper function ...
	 */
	dbusGetMessageErrorID(reply, &ret);

	dbus_message_unref(message);
	return ret;

}

char *DBus_Error_Array[REPLY_ERROR_MAX] = {
	"success",
	"no connection",
	"no rights",
	"invalid param",
	"invalid method",
	"invalid interface",
	"not supported",
	"disabled",
	"already set",
	"dbus error",
	"general error",
	"dbus invalid message type",
};
