/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config_pm.h"

extern void strtointErr(char *str, int line, const char *file);

/* SchemeConfig must be intitialised with general settings */
SchemeConfig::SchemeConfig(const string &file_name, const GeneralConfig & gc)
{

	pDebug(DBG_DEBUG, "SchemeConfig Constructor");
	this->file_name = file_name;
	this->file_path = gc.scheme_dir + "/" + file_name;

	/* Initialise with general configurations ************** */
	CPU_IDLE_LIMIT = gc.CPU_IDLE_LIMIT;
	MAX_CPUS_ONLINE = gc.MAX_CPUS_ONLINE;
	CPU_IDLE_TIMEOUT = gc.CPU_IDLE_TIMEOUT;
	CPU_HIGH_LIMIT = gc.CPU_HIGH_LIMIT;
	JUMP_CPU_FREQ_MAX_LIMIT = gc.JUMP_CPU_FREQ_MAX_LIMIT;
	CPUFREQ_ENABLED = gc.CPUFREQ_ENABLED;
	CPUFREQ_HYSTERESIS = gc.CPUFREQ_HYSTERESIS;
	CONSIDER_NICE = gc.CONSIDER_NICE;
	POLL_INTERVAL = gc.POLL_INTERVAL;

	BAT_WARN_LIMIT = gc.BAT_WARN_LIMIT;
	BAT_LOW_LIMIT = gc.BAT_LOW_LIMIT;
	BAT_CRIT_LIMIT = gc.BAT_CRIT_LIMIT;

	ENABLE_THERMAL_MANAGEMENT = gc.ENABLE_THERMAL_MANAGEMENT;

	DISPLAY_BRIGHTNESS = gc.DISPLAY_BRIGHTNESS;

	USER_STANDBY_DISABLED = gc.USER_STANDBY_DISABLED;
	USER_SUSPEND2DISK_DISABLED = gc.USER_SUSPEND2DISK_DISABLED;
	USER_SUSPEND2RAM_DISABLED = gc.USER_SUSPEND2RAM_DISABLED;
	DPM_ENABLED = gc.DPM_ENABLED;
	POWER_BUTTON_DELAY = gc.POWER_BUTTON_DELAY;

	ALLOW_THROTTLING = gc.ALLOW_THROTTLING;
	MAX_CPU_THROTTLING = gc.MAX_CPU_THROTTLING;
	ALWAYS_THROTTLE = gc.ALWAYS_THROTTLE;
	CPUFREQUENCY = gc.CPUFREQUENCY;
	COOLING_POLICY = gc.COOLING_POLICY;
	SCHEME_NAME = gc.SCHEME_NAME;
	SCHEME_DESCRIPTION = gc.SCHEME_DESCRIPTION;
	CPUFREQ_CONTROL = gc.CPUFREQ_CONTROL;

	dpmClasses = gc.dpmClasses;

	eventMap = gc.eventMap;
	next = NULL;

	/* duplicate thermal settings from BIOS (always stay in general conf object) */
	for (int x = 0; gc.thermal_zones[x].present && x < MAX_THERMAL_ZONES; x++) {
		/* only trip points need to be copied, rest will be obtained and modified
		   from/by kernel */
		thermal_zones[x].critical = gc.thermal_zones[x].critical;
		thermal_zones[x].hot = gc.thermal_zones[x].hot;
		thermal_zones[x].passive = gc.thermal_zones[x].passive;
		for (int y = 0; y < ACPI_THERMAL_MAX_ACTIVE; y++) {
			thermal_zones[x].active[y] = gc.thermal_zones[x].active[y];
		}
	}
}

SchemeConfig::~SchemeConfig()
{
}

int SchemeConfig::readConfigFile(const string &file)
{
	return PS_Config::readConfigFile(file_path);
}

#ifdef CREATE_DEBUG_OUTPUT
#include <iostream>
ostream & operator<<(ostream & os, SchemeConfig & sc)
{
	string cpufreq_st;
	string thermal_management;

	os << endl << "------------------------------------------";
	os << endl << "Scheme: " << sc.SCHEME_NAME << endl << endl << "Scheme config file:" << sc.file_path << endl;

	switch (sc.CPUFREQUENCY){
		case _DYNAMIC:
			cpufreq_st = "dynamic";
			break;
		case _PERFORMANCE:
			cpufreq_st = "performance";
			break;
		case _POWERSAVE:
			cpufreq_st = "powersave";
			break;
		default:
			cpufreq_st = "invalid value";
	}

	switch (sc.ENABLE_THERMAL_MANAGEMENT == OFF){
		case OFF:
			thermal_management = "off";
			break;
		case KERNEL:
			thermal_management = "kernel";
			break;
		case KERNEL_PASSIVE:
			thermal_management = "kernel_passive";
			break;
		default:
			thermal_management = "invalid value";
			break;
	}

	os << endl << "Event settings:" << endl;

	for (EventIter i = sc.eventMap.begin(); i != sc.eventMap.end(); i++)
		os << i->first << "\t" << i->second << endl;

	os << endl << "CPU settings:"
	    << endl << "CPU_IDLE_LIMIT: " << sc.CPU_IDLE_LIMIT << "%"
	    << endl << "CPU_IDLE_TIMEOUT: " << sc.CPU_IDLE_TIMEOUT
	    << endl << "CPU_HIGH_LIMIT: " << sc.CPU_HIGH_LIMIT << "%"
	    << endl << "JUMP_CPU_FREQ_MAX_LIMIT: " << sc.JUMP_CPU_FREQ_MAX_LIMIT << "%"
	    << endl << "CPUFREQ_HYSTERESIS: " << sc.CPUFREQ_HYSTERESIS << "%"
	    << endl << "CONSIDER_NICE: " << (sc.CONSIDER_NICE == 1 ? "yes" : "no")
	    << endl << "POLL_INTERVAL: " << sc.POLL_INTERVAL
	    << endl << endl << "Battery settings:"
	    << endl << "BAT_WARN_LIMIT: " << sc.BAT_WARN_LIMIT
	    << endl << "BAT_LOW_LIMIT: " << sc.BAT_LOW_LIMIT
	    << endl << "BAT_CRIT_LIMIT: " << sc.BAT_CRIT_LIMIT
	    << endl << "FORCE_BATTERY_POLLING: " << (sc.FORCE_BATTERY_POLLING == 1 ? "yes" : "no")
	    << endl << endl << "Temperature settings:"
	    << endl << "POWERSAVE_ENABLE_THERMAL_MANAGEMENT: " << thermal_management
	    << endl << endl << "Other settings:" << endl << "USER_SUSPEND2DISK_DISABLED: " << (sc.
											       USER_SUSPEND2DISK_DISABLED
											       == 1 ? "yes" : "no")
	    << endl << "USER_SUSPEND2RAM_DISABLED: " << (sc.USER_SUSPEND2RAM_DISABLED == 1 ? "yes" : "no")
	    << endl << "USER_STANDBY_DISABLED: " << (sc.USER_STANDBY_DISABLED == 1 ? "yes" : "no")
	    << endl << "CONSIDER_NICE: " << (sc.CONSIDER_NICE ? "yes" : "no")
	    << endl << "POWER_BUTTON_DELAY: " << sc.POWER_BUTTON_DELAY;

	os << endl << endl << "POWERSAVE_ALLOW_THROTTLING: " << ((sc.ALLOW_THROTTLING == 0) ? "no" : "yes")
	    << endl << "MAX_CPU_THROTTLING: " << sc.MAX_CPU_THROTTLING
	    << endl << "POWERSAVE_CPUFREQUENCY: " << cpufreq_st
	    << endl << "POWERSAVE_CPUFREQU_CONTROL: " << (sc.CPUFREQ_CONTROL == CPUFREQ_USERSPACE ?
							  "userspace" : "kernel")
	    << endl << "POWERSAVE_USER_INPUT_TIMEOUT1: " << sc.USER_INPUT_TIMEOUT1
	    << endl << "POWERSAVE_USER_INPUT_TIMEOUT2: " << sc.USER_INPUT_TIMEOUT2
	    << endl << "POWERSAVE_COOLING_POLICY: " << ((sc.COOLING_POLICY == COOLING_MODE_PASSIVE) ? "passive" :
							"active");

	if (sc.ENABLE_THERMAL_MANAGEMENT == KERNEL 
	    || sc.ENABLE_THERMAL_MANAGEMENT == KERNEL_PASSIVE) {
		    
		int x = 0;
		for (; sc.thermal_zones[x].present && x < MAX_THERMAL_ZONES; x++) {
			os << endl << endl << "Thermal Zone no. " << x
			    << endl << "Temperature: " << sc.thermal_zones[x].temperature << " degree Celcius";
			if (sc.ENABLE_THERMAL_MANAGEMENT == KERNEL
			    // only show thermal zone for kernel_passive when there is a passive
			    // trip point defined
			    || sc.ENABLE_THERMAL_MANAGEMENT == KERNEL_PASSIVE && 
			    sc.thermal_zones[x].passive > 0)) {) {
				if (sc.thermal_zones[x].critical > 0)
					os << endl << "Critical: " << sc.thermal_zones[x].critical << " degree Celcius";
				else
					os << endl << "Critical: not supported";
				if (sc.thermal_zones[x].hot > 0)
					os << endl << "Hot: " << sc.thermal_zones[x].hot << " degree Celcius";
				else
					os << endl << "Hot: not supported";
			}
			if (sc.thermal_zones[x].passive > 0)
				os << endl << "Passive: " << sc.thermal_zones[x].passive << " degree Celcius"
				    << endl << "   Polling freq in passive mode (secs/10): " << sc.thermal_zones[x].tsp
				    << endl << "   TC1: " << sc.thermal_zones[x].tc1
				    << endl << "   TC2: " << sc.thermal_zones[x].tc2;
			else
				os << endl << "Passive: not supported";
			if (sc.ENABLE_THERMAL_MANAGEMENT == KERNEL) {
				for (int y = 0; y < ACPI_THERMAL_MAX_ACTIVE; y++) {
					if (sc.thermal_zones[x].active[y] > 0)
						os << endl << "Active[" << y << "]: " << sc.thermal_zones[x].active[y];
				}
				
				os << endl << "Cooling Mode: ";
				switch (sc.thermal_zones[x].cooling_mode) {
					case UNKNOWN:
						os << "Not supported";
						break;
					case ACTIVE:
						os << "Active";
						break;
					case PASSIVE:
						os << "Passive";
						break;
					default:
						os << "Undefined -> this is a bug.";
				}
			}
		}
		if (x == 0)
			os << endl << "Thermal control not activated, or no trip point support by BIOS";
	}

	return os;
}

#endif
