/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxcalc-globals.h"

#include "polyxcalc-init.h"
#include "polyxcalc-ui-calculator.h"
#include "polyxcalc-ui-chempad.h"
#include "polyxcalc-ui-recorder.h"
#include "polyxcalc-calcmass.h"
#include "polyxmass-ui-mz-ratio.h"



GtkWidget *
polyxcalc_ui_calculator_wnd_setup_window (PxmCalcCtxt *calcctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *combo = NULL;
  GladeXML *xml = NULL;
  
  gchar *gui_file = NULL;

 
  
  /* We are setting up the calculator window for the calcctxt
     (polyxcalc context). That calculator window is referenced in the calcctxt
     with the calculator_wnd member...
  */
  g_assert (calcctxt != NULL);
  
  g_assert (calcctxt->calculator_wnd == NULL);
  


  gui_file = 
    g_strdup_printf ("%s/polyxcalc.glade", userspec->gladedir);

  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "polyxcalc_calculator_wnd", 
		       PACKAGE);
  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_critical (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  calcctxt->calculator_wnd = 
    glade_xml_get_widget (xml, "polyxcalc_calculator_wnd");

  if (calcctxt->calculator_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to create the polymer mass calculator window\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  /* We will have to access the calcctxt from the pointer to _this_
     window.
   */
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "calcctxt", calcctxt);
 
  /* The polymer chemistry definition entry which tells the user
     what definition is currently loaded.
  */
  widget = glade_xml_get_widget (xml, "polymer_definition_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polymer_definition_entry", widget);
  
  if (calcctxt->polchemdefctxt != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			calcctxt->polchemdefctxt->polchemdef->type);
  
  
  /* INITIAL MASSES
   */
  widget = glade_xml_get_widget (xml, "initial_mono_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "initial_mono_entry", widget);
  
  widget = glade_xml_get_widget (xml, "initial_avg_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "initial_avg_entry", widget);

  widget = glade_xml_get_widget (xml, "init_masses_add_to_res_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "init_masses_add_to_res_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_init_masses_add_to_res_button),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "init_masses_rem_from_res_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "init_masses_rem_from_res_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_init_masses_rem_from_res_button),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "init_masses_clear_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "init_masses_clear_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_init_masses_clear_button),
     calcctxt);

  widget = glade_xml_get_widget (xml, "init_masses_send_to_res_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "init_masses_send_to_res_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_init_masses_send_to_res_button),
     calcctxt);

  
  /* UNIVERSAL CHEMICAL ENTITIES
   */
  widget = glade_xml_get_widget (xml, "univ_chem_entities_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "univ_chem_entities_checkbutton", widget);
  g_signal_connect
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_univ_chem_entities_checkbutton),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "univ_chem_entities_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "univ_chem_entities_vbox", widget);
  
  
  /* ATOMS
   */
  widget = glade_xml_get_widget (xml, "atoms_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_checkbutton", widget);
  
  g_signal_connect 
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_atoms_checkbutton),
     calcctxt);
  
  vbox = glade_xml_get_widget (xml, "atoms_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_vbox", vbox);
  
  widget = glade_xml_get_widget (xml, "atoms_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_frame", widget);
  
  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, "atoms_combo");
    g_assert (widget != NULL);
    g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
    "atoms_combo", widget);
  */

  /* Momentaneous manual creation of the GtkComboBoxEntry due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_entry_new_text ();
  g_assert (combo != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_combo", combo);
  
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);
  
  /* The combo has to be before the "count" label and GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);
  
  if (-1 == polyxcalc_ui_calculator_wnd_fill_atoms_combobox (calcctxt))
    g_critical (_("%s@%d: failed to set the atoms to the combobox\n"),
	   __FILE__, __LINE__);
  
  widget = GTK_WIDGET (GTK_ENTRY (GTK_BIN (combo)->child));
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_combo_entry", widget);
  
  widget = glade_xml_get_widget (xml, "atoms_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "atoms_count_entry", widget);

  /* FORMULA
   */  
  widget = glade_xml_get_widget (xml, "formula_actform_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "formula_actform_checkbutton", widget);

  g_signal_connect 
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_formula_actform_checkbutton),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "actform_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "actform_vbox", widget);

  widget = glade_xml_get_widget (xml, "actform_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "actform_frame", widget);

  widget = glade_xml_get_widget (xml, "actform_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "actform_entry", widget);
  
  widget = glade_xml_get_widget (xml, "actform_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "actform_count_entry", widget);


  /* POLYMER CHEMISTRY DEFINITION-RELATED ENTITIES: ONLY HANDLE THEM
     IF THE valid_polchemdef VARIABLE IS SET TO TRUE.
  */
  widget = glade_xml_get_widget (xml, 
				 "polchemdef_spec_entities_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polchemdef_spec_entities_checkbutton", widget);

  g_signal_connect
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_polchemdefspec_entities_checkbutton),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "polchemdef_spec_entities_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polchemdef_spec_entities_vbox", widget);
  
  
  /* MONOMERS
   */
  widget = glade_xml_get_widget (xml, "monomers_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_checkbutton", widget);
  
  g_signal_connect 
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_monomers_checkbutton),
     calcctxt);
  
  vbox = glade_xml_get_widget (xml, "monomers_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_vbox", vbox);
  
  widget = glade_xml_get_widget (xml, "monomers_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_frame", widget);

  /* Set the frame to inactive if 
     calcctxt->polchemdefctxt == NULL.
  */
  if (calcctxt->polchemdefctxt == NULL)
    gtk_widget_set_sensitive (widget, FALSE);
  
  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, "monomers_combo");
    g_assert (widget != NULL);
    g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
    "monomers_combo", widget);
  */
    
  /* Momentaneous manual creation of the GtkComboBoxEntry due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_entry_new_text ();
  g_assert (combo != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_combo", combo);
  
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the "count" label and GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  if (calcctxt->polchemdefctxt != NULL)
    {
      if (-1 == 
	  polyxcalc_ui_calculator_wnd_fill_monomers_combobox (calcctxt))
	g_critical (_("%s@%d: failed to set the monomers to the combo list\n"),
	       __FILE__, __LINE__);
    }
  
  widget = GTK_WIDGET (GTK_ENTRY (GTK_BIN (combo)->child));
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_combo_entry", widget);
  
  
  widget = glade_xml_get_widget (xml, "monomers_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "monomers_count_entry", widget);

  
  /* MODIFICATIONS
   */
  widget = glade_xml_get_widget (xml, "modifications_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_checkbutton", widget);

  g_signal_connect 
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_modifications_checkbutton),
     calcctxt);
  
  vbox = glade_xml_get_widget (xml, "modifications_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_vbox", vbox);

  widget = glade_xml_get_widget (xml, "modifications_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_frame", widget);

  /* Set the frame to inactive if 
     calcctxt->polchemdefctxt == NULL.
  */
  if (calcctxt->polchemdefctxt == NULL)
    gtk_widget_set_sensitive (widget, FALSE);

  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, "modifications_combo");
    g_assert (widget != NULL);
    g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
    "modifications_combo", widget);
  */

  /* Momentaneous manual creation of the GtkComboBoxEntry due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_entry_new_text ();
  g_assert (combo != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_combo", combo);
  
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the "count" label and GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  if (calcctxt->polchemdefctxt != NULL)
    {
      if (-1 == polyxcalc_ui_calculator_wnd_fill_modifs_combobox (calcctxt))
	g_critical (_("%s@%d: failed to set the modifs to the combo list\n"),
	       __FILE__, __LINE__);
    }
  
  widget = GTK_WIDGET (GTK_ENTRY (GTK_BIN (combo)->child));
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_combo_entry", widget);
  
  widget = glade_xml_get_widget (xml, "modifications_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "modifications_count_entry", widget);
  
  /* SEQUENCE
   */
  widget = glade_xml_get_widget (xml, "polymer_sequence_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polymer_sequence_checkbutton", widget);

  g_signal_connect 
    (GTK_OBJECT (widget),
     "toggled",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_polymer_sequence_checkbutton),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "sequence_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "sequence_vbox", widget);

  widget = glade_xml_get_widget (xml, "sequence_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "sequence_frame", widget);

  /* Set the frame to inactive if 
     calcctxt->polchemdefctxt == NULL.
  */
  if (calcctxt->polchemdefctxt == NULL)
    gtk_widget_set_sensitive (widget, FALSE);

  widget = glade_xml_get_widget (xml, "sequence_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "sequence_entry", widget);
  
  widget = glade_xml_get_widget (xml, "sequence_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "sequence_count_entry", widget);
  

  /* The helper windows, chempad and recorder... should they 
     be visible. This is what these two buttons govern.
  */
  widget = glade_xml_get_widget (xml, "toggle_chempad_window_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "toggle_chempad_window_button", widget);
  
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_toggle_chempad_window_button),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "toggle_recorder_window_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "toggle_recorder_window_button", widget);
  
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_toggle_recorder_window_button),
     calcctxt);
  
  /* RESULT MASSES
   */
  widget = glade_xml_get_widget (xml, "result_mono_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "result_mono_entry", widget);
  
  widget = glade_xml_get_widget (xml, "result_avg_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "result_avg_entry", widget);

  widget = glade_xml_get_widget (xml, "res_masses_add_to_init_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "res_masses_add_to_init_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_res_masses_add_to_init_button),
     calcctxt);

  widget = glade_xml_get_widget (xml, "res_masses_rem_from_init_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "res_masses_rem_from_init_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_res_masses_rem_from_init_button),
     calcctxt);

  widget = glade_xml_get_widget (xml, "res_masses_clear_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "res_masses_clear_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_res_masses_clear_button),
     calcctxt);

  widget = glade_xml_get_widget (xml, "res_masses_send_to_init_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "res_masses_send_to_init_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxcalc_ui_calculator_wnd_res_masses_send_to_init_button),
     calcctxt);


  
  /* THE OPERATIONS: BUTTONS' CALLBACKS
   */
  widget = glade_xml_get_widget (xml, "polyxcalc_apply_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polyxcalc_apply_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxcalc_ui_calculator_wnd_apply_button),
		    calcctxt);

  widget = glade_xml_get_widget (xml, "polyxcalc_clear_all_chemistry_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polyxcalc_clear_all_chemistry_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_clear_all_chemistry_button),
     calcctxt);
  
  widget = glade_xml_get_widget (xml, "polyxcalc_mz_ratio_calculator_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->calculator_wnd),
		     "polyxcalc_mz_ratio_calculator_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxcalc_ui_calculator_wnd_mz_ratio_calculator_button),
     calcctxt);
  

  /* We don't need the GladeXML object any more, so unref it
   * to save some memory.
   */
  g_object_unref (G_OBJECT (xml));


  /* Connect some important signals for the calcctxt->calculator_wnd proper.
   */
  g_signal_connect (G_OBJECT (calcctxt->calculator_wnd),
		    "delete_event",
		    G_CALLBACK (polyxcalc_ui_calculator_wnd_delete_event),
		    calcctxt);
  
  g_signal_connect (G_OBJECT (calcctxt->calculator_wnd),
		    "destroy_event",
		    G_CALLBACK (polyxcalc_ui_calculator_wnd_destroy_event),
		    calcctxt);

  gtk_widget_show_all (GTK_WIDGET (calcctxt->calculator_wnd));
    
  return calcctxt->calculator_wnd;
}




/* COMBO FILLING FUNCTION FOR ATOMS, MONOMERS AND MODIFS
 */
gint
polyxcalc_ui_calculator_wnd_fill_atoms_combobox (PxmCalcCtxt *calcctxt)
{
  GtkWidget *combo = NULL;

  PxmAtom *atom = NULL;
  
  gint iter = 0;
  gint count = 0;

  
  g_assert (calcctxt != NULL);
  g_assert (calcctxt->calculator_wnd != NULL);
  
  /* Get a pointer to the combo of interest.
   */
  combo = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "atoms_combo");
  g_assert (combo != NULL);

  /* In the calcctxt->calculator_wnd setting up function we did load
     the atom definition file that corresponded to the polymer
     chemistry definition pointer passed to it. If that pointer was
     NULL, then we did loaded the "basic" atom definition that is
     default in the system.

     We did load the atom definition by making PxmAtom instances that
     are now stored in the atomGPA array whose pointer is set in the
     calcctxt object.
  */


  /* We want that in the list there be an empty item (the first one,
     which is displayed by default) for the user to select if no atom
     is involved in the molecular calculation.
  */
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), "");

  for (iter = 0 ; iter < calcctxt->atomGPA->len ; iter++)
    {
      atom = g_ptr_array_index (calcctxt->atomGPA, iter);
      g_assert (atom != NULL);
      
      /* Apparently this form leaks memory due to the g_strdup ()
	 call.
       *
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
				 g_strdup (atom->symbol));
      */
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
				 atom->symbol);

      count++;
    }
  
  gtk_widget_show_all (GTK_WIDGET (combo));

  return count;
}  

gint
polyxcalc_ui_calculator_wnd_fill_monomers_combobox (PxmCalcCtxt* calcctxt)
{
  GtkWidget *combo = NULL;

  PxmPolchemdef *polchemdef = NULL;
  
  PxmMonomer *monomer = NULL;
  
  gint iter = 0;
  gint count = 0;
  
  g_assert (calcctxt != NULL);
  
  g_assert (calcctxt->calculator_wnd != NULL);

  g_assert (calcctxt->polchemdefctxt != NULL);
  polchemdef = calcctxt->polchemdefctxt->polchemdef;

  g_assert (polchemdef != NULL);
    
  /* Get a pointer to the combo of interest.
   */
  combo = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd),
			     "monomers_combo");
  g_assert (combo != NULL);
  
  
  /* We want that in the list there be an empty item (the first one,
   * which is displayed by default) for the user to select if no 
   * atom is involved in the molecular calculation.
   */
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), "");

  for (iter = 0 ; iter < polchemdef->monomerGPA->len ; iter++)
    {
      monomer = g_ptr_array_index (polchemdef->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
				 g_strdup (monomer->name));
      count++;
    }
  
  gtk_widget_show_all (GTK_WIDGET (combo));

  return count;
}  


gint
polyxcalc_ui_calculator_wnd_fill_modifs_combobox (PxmCalcCtxt* calcctxt)
{
  /* Contrary to the atoms case, here we only can show the modifs if
   * the window passed as parameter was opened for a given polymer
   * definition, from which we can take up the modifs...
   */
  GtkWidget *combo = NULL;

  PxmPolchemdef *polchemdef = NULL;
  
  PxmModif *modif = NULL;
  
  gint iter = 0;
  gint count = 0;
  
  g_assert (calcctxt != NULL);
  
  g_assert (calcctxt->calculator_wnd != NULL);

  g_assert (calcctxt->polchemdefctxt != NULL);
  polchemdef = calcctxt->polchemdefctxt->polchemdef;

  g_assert (polchemdef != NULL);
  
  /* Get a pointer to the combo of interest.
   */
  combo = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "modifications_combo");
  g_assert (combo != NULL);
  
  /* We want that in the list there be an empty item (the first one,
   * which is displayed by default) for the user to select if no 
   * atom is involved in the molecular calculation.
   */
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), "");

  for (iter = 0 ; iter < polchemdef->modifGPA->len ; iter++)
    {
      modif = g_ptr_array_index (polchemdef->modifGPA, iter);
      g_assert (modif != NULL);
      
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
				 g_strdup (modif->name));
      count++;
    }
  
  gtk_widget_show_all (GTK_WIDGET (combo));

  return count;
}  


/* OPERATIONS BUTTONS' HANDLERS.
 */
void
polyxcalc_ui_calculator_wnd_apply_button (GtkWidget *button,
					  gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  gchar *help = 0;

  PxmPolchemdef *polchemdef = NULL;
      
  /* We are asked to make a calculation by simple adding the
   * contributions of any currently accountable item, to have a mass
   * for it and to add it to the initial masses (if any) and to
   * display the new masses in the result masses entries.
   *
   * Each of the accounting functions below return 1 upon success or 0
   * upon failure.
   */
  g_assert (calcctxt != NULL);

  if (calcctxt->polchemdefctxt != NULL)
    polchemdef = calcctxt->polchemdefctxt->polchemdef;

  
  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  

  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  
  /* We only take into account the initial masses if the
   * result masses are empty. Othewise we take into account
   * the result masses, because, like a calculator we want
   * that the operations be incremental.
   */
  /* MONO
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we do not take
       * into account the initial mono mass, but the result mono mass
       * instead.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  return ;
	}
      
      /* Since we are going to use the result avg mass, we want to
       * store it so that we can later put it to the init avg mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 
				      calcctxt->masspair_temp->mono); 
    }
  else
    {
      /* The mono result mass entry is not filled with anything,
       * so consider taking into account the initial mono mass.
       */
      entry = g_object_get_data (G_OBJECT (calculator_wnd), 
				 "initial_mono_entry");
      g_assert (entry != NULL);
      
      help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
      
      if (help != NULL && strlen (help) > 0)
	{
	  if (FALSE == 
	      libpolyxmass_globals_strtod ((gchar *) help,
					   &(calcctxt->masspair_temp->mono)))
	    {
	      g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		     __FILE__, __LINE__, help);

	      return ;
	    }
	}
      
      /* Since we are going to use the init mono mass, we want to
       * store it so that we can later put it to the init mono mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 
				      calcctxt->masspair_temp->mono); 
    }

  /* AVG
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we do not take
       * into account the initial avg mass, but the result avg mass
       * instead.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  return ;
	}
      
      /* Since we are going to use the result avg mass, we want to
       * store it so that we can later put it to the init avg mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 
				     calcctxt->masspair_temp->avg); 
    }
  else
    {
      /* The avg result mass entry is not filled with anything,
       * so consider taking into account the initial avg mass.
       */
      entry = g_object_get_data (G_OBJECT (calculator_wnd), 
				 "initial_avg_entry");
      g_assert (entry != NULL);
      
      help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
      
      if (help != NULL && strlen (help) > 0)
	{
	  if (FALSE == 
	      libpolyxmass_globals_strtod ((gchar *) help,
					   &(calcctxt->masspair_temp->avg)))
	    {
	      g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		     __FILE__, __LINE__, help);

	      return ;
	    }
	}
      
      /* Since we are going to use the init avg mass, we want to
       * store it so that we can later put it to the init avg mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 
				     calcctxt->masspair_temp->avg); 
    }


  /* ATOMS
   */
  polxyxcalc_calcmass_account_atoms (calcctxt);
  
  /* FORMULA
   */
  polxyxcalc_calcmass_account_actform (calcctxt);
  
  if (polchemdef != NULL)
    {
      /* MONOMERS: ONLY IF A POLYMER DEFINITION EXISTS!
       */
      polxyxcalc_calcmass_account_monomers (calcctxt);
      
      /* MODIFS: ONLY IF A POLYMER DEFINITION EXISTS!
       */
      polxyxcalc_calcmass_account_modifs (calcctxt);
      
      /* SEQUENCE: ONLY IF A POLYMER DEFINITION EXISTS!
       */
      polxyxcalc_calcmass_account_sequence (calcctxt);
    }
  
  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  /* Set the old values to the initial mass entries.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_univ_chem_entities_checkbutton (GtkWidget *widget,
							gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "univ_chem_entities_vbox");
  g_assert (box != NULL);
  
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "univ_chem_entities_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    {
      gtk_widget_show_all (GTK_WIDGET (box));
      polyxcalc_ui_calculator_wnd_atoms_checkbutton (NULL, calcctxt);
      polyxcalc_ui_calculator_wnd_formula_actform_checkbutton 
	(NULL, calcctxt);
    }
  else
    gtk_widget_hide_all (GTK_WIDGET (box));

  return;
}


void 
polyxcalc_ui_calculator_wnd_atoms_checkbutton (GtkWidget *widget,
					       gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "atoms_vbox");
  g_assert (box != NULL);
  
  /* This function might be called by another function, like
     polyxcalc_ui_calculator_wnd_universal_chem_entities_checkbutton
     (), for example, with a NULL first argument. But since we know
     that we are working on a known checkbutton that is of no concern,
     we just get the checkbutton from the object data.
  */
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "atoms_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    gtk_widget_show_all (GTK_WIDGET (box));
  else
    gtk_widget_hide_all (GTK_WIDGET (box));
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_formula_actform_checkbutton (GtkWidget *widget,
							 gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "actform_vbox");
  g_assert (box != NULL);
  
  /* This function might be called by another function, like
     polyxcalc_ui_calculator_wnd_universal_chem_entities_checkbutton
     (), for example, with a NULL first argument. But since we know
     that we are working on a known checkbutton that is of no concern,
     we just get the checkbutton from the object data.
  */
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "formula_actform_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    gtk_widget_show_all (GTK_WIDGET (box));
  else
    gtk_widget_hide_all (GTK_WIDGET (box));
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_polchemdefspec_entities_checkbutton (GtkWidget *widget,
							     gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), 
			   "polchemdef_spec_entities_vbox");
  g_assert (box != NULL);
  
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "polchemdef_spec_entities_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    {
      gtk_widget_show_all (GTK_WIDGET (box));
      polyxcalc_ui_calculator_wnd_monomers_checkbutton (NULL, calcctxt);
      polyxcalc_ui_calculator_wnd_modifications_checkbutton 
	(NULL, calcctxt);
      polyxcalc_ui_calculator_wnd_polymer_sequence_checkbutton 
	(NULL, calcctxt);
    }
  else
    gtk_widget_hide_all (GTK_WIDGET (box));

  return;
}


void 
polyxcalc_ui_calculator_wnd_monomers_checkbutton (GtkWidget *widget,
						  gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "monomers_vbox");
  g_assert (box != NULL);
  
  /* This function might be called by another function, like
     polyxcalc_ui_calculator_wnd_polchemdef_spec_entities_checkbutton
     (), for example, with a NULL first argument. But since we know
     that we are working on a known checkbutton that is of no concern,
     we just get the checkbutton from the object data.
  */
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "monomers_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    gtk_widget_show_all (GTK_WIDGET (box));
  else
    gtk_widget_hide_all (GTK_WIDGET (box));

  return;
}


void 
polyxcalc_ui_calculator_wnd_modifications_checkbutton (GtkWidget *widget,
						       gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "modifications_vbox");
  g_assert (box != NULL);
  
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "modifications_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    gtk_widget_show_all (GTK_WIDGET (box));
  else
    gtk_widget_hide_all (GTK_WIDGET (box));

  return;
}


void 
polyxcalc_ui_calculator_wnd_polymer_sequence_checkbutton (GtkWidget *widget,
							  gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *checkbutton = NULL;
  GtkWidget *box = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  box = g_object_get_data (G_OBJECT (calculator_wnd), "sequence_vbox");
  g_assert (box != NULL);
  
  checkbutton = g_object_get_data (G_OBJECT (calculator_wnd), 
				   "polymer_sequence_checkbutton");
  g_assert (checkbutton != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton)))
    gtk_widget_show_all (GTK_WIDGET (box));
  else
    gtk_widget_hide_all (GTK_WIDGET (box));

  return;
}


void 
polyxcalc_ui_calculator_wnd_toggle_chempad_window_button (GtkWidget *button,
							  gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *chempad_wnd = NULL;

  g_assert (calcctxt != NULL);

  chempad_wnd = calcctxt->chempad_wnd;
  
  if (chempad_wnd == NULL)
    return;
  
  if (TRUE == GTK_WIDGET_VISIBLE (GTK_WIDGET (chempad_wnd)))
    gtk_widget_hide_all (GTK_WIDGET (chempad_wnd));
  else
    gtk_widget_show_all (GTK_WIDGET (chempad_wnd));

  return;
}


void 
polyxcalc_ui_calculator_wnd_toggle_recorder_window_button (GtkWidget *button,
							   gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *recorder_wnd = NULL;

  g_assert (calcctxt != NULL);

  recorder_wnd = calcctxt->recorder_wnd;
  
  if (recorder_wnd == NULL)
    return;
  
  if (TRUE == GTK_WIDGET_VISIBLE (GTK_WIDGET (recorder_wnd)))
    gtk_widget_hide_all (GTK_WIDGET (recorder_wnd));
  else
    gtk_widget_show_all (GTK_WIDGET (recorder_wnd));

  return;
}



/* MASS-related FUNCTIONS
 */

void 
polyxcalc_ui_calculator_wnd_init_masses_add_to_res_button (GtkWidget *button,
							   gpointer data)
{
  PxmCalcCtxt *calcctxt = data;
  
  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;
  
  GString *gs = NULL;
  
  gchar *help = 0;
    

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);
  


  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  

  /* MONO
   */

  /* First try to store the present result mono mass, so that 
   * we can redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  /* Second, get the initial mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);
  
  
  /* AVG
   */

  /* First try to store the present result avg mass, so that 
   * we can redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  /* Second, get the initial mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 0);
  
  
  g_string_append_printf (gs, 
			  _("Added initial (mono: %f ; avg: %f) \n"
			    "to result (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg,
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);


  /* Make the MONO calculation per se:
   */
  libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 
				  calcctxt->masspair_init->mono 
				  + calcctxt->masspair_temp->mono);
  /* Make the AVG calculation per se:
   */
  libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 
				 calcctxt->masspair_init->avg 
				 + calcctxt->masspair_temp->avg);

  
  /* Finally display the masses in their respective entry widget:
   */

  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  /* Set the old values to the initial mass entries.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  g_string_append_printf (gs, 
			  _("\t\t==> mono: %f ; avg: %f\n"),
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg);

  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_init_masses_rem_from_res_button (GtkWidget *button,
							     gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;
  
  GString *gs = NULL;
  
  gchar *help = 0;


  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);



  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  

  /* MONO
   */

  /* First try to store the present result mono mass, so that 
   * we can redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  /* Second, get the initial mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);
  
  /* AVG
   */

  /* First try to store the present result avg mass, so that 
   * we can redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  /* Second, get the initial mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 0);
  
  /* Start making the log for the initial conditions, prior to
     actually calculating the difference.
  */
  g_string_append_printf (gs, 
			  _("Removed initial (mono: %f ; avg: %f)\n"
			    "from result (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg,
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);

  /* Now we can perform the calculation.
   */

  /* Make the MONO calculation per se:
   */
  libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 
				  calcctxt->masspair_init->mono 
				  - calcctxt->masspair_temp->mono);

  /* Make the AVG calculation per se:
   */
  libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 
				 calcctxt->masspair_init->avg 
				 - calcctxt->masspair_temp->avg);

  /* Finally display the masses in their respective entry widget:
   */

  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  /* Set the old values to the initial mass entries.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  /* Finally conclude the operation's log.
   */
  g_string_append_printf (gs, 
			  "\t\t==> mono: %f ; avg: %f\n", 
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_init_masses_send_to_res_button(GtkWidget *button,
							   gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry_init = NULL;
  GtkWidget *entry_res = NULL;

  GString *gs = NULL;

  gchar *help = NULL;
  

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  
  
  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");

  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);


  /* MONO
   */

  /* First try to store the present result mono mass, so that we can
     use its value later in the recorder.
  */
  entry_res = g_object_get_data (G_OBJECT (calculator_wnd), 
				 "result_mono_entry");
  g_assert (entry_res != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
	 into masspair_init, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);

  /* Now that we have stored the previous value of the result, we can
     copy init value into the result entry widget.
  */
  entry_init = g_object_get_data (G_OBJECT (calculator_wnd), 
				  "initial_mono_entry");
  g_assert (entry_init != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  g_assert (help != NULL);

  gtk_entry_set_text (GTK_ENTRY (entry_res), help);

  /* Finally, make 100% sure that the new value in the result masses
     widget is actually a proper number, and store its value for the
     recorder.
  */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
	 into masspair_temp, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);

  
  /* AVG
   */

  /* First try to store the present result avg mass, so that we can
     use its value later in the recorder.
  */
  entry_res = g_object_get_data (G_OBJECT (calculator_wnd), 
				 "result_avg_entry");
  g_assert (entry_res != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
	 into masspair_init, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);

  /* Now that we have stored the previous value of the result, we can
     copy init value into the result entry widget.
  */
  entry_init = g_object_get_data (G_OBJECT (calculator_wnd), 
				  "initial_avg_entry");
  g_assert (entry_init != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  g_assert (help != NULL);

  gtk_entry_set_text (GTK_ENTRY (entry_res), help);

  /* Finally, make 100% sure that the new value in the result masses
     widget is actually a proper number, and store its value for the
     recorder.
  */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
	 into masspair_temp, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);

  /* Finally, we can ensure that the modification gets recorded.
   */    
  g_string_append_printf 
    (gs, 
     _("Sent initial (mono: %f ; avg: %f) \n"
       "to result (was mono: %f ; avg: %f):\n"
       "\t\t==> mono: %f ; avg: %f\n"),
     calcctxt->masspair_temp->mono, calcctxt->masspair_temp->avg,
     calcctxt->masspair_init->mono, calcctxt->masspair_init->avg,
     calcctxt->masspair_temp->mono, calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);
  
  return ;
}


void 
polyxcalc_ui_calculator_wnd_init_masses_clear_button (GtkWidget *button,
						      gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;

  gchar *help = NULL;
  
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);



  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");

  libpolyxmass_masspair_reset (calcctxt->masspair_init);

  /* MONO
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));

  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);
	  
	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  gtk_entry_set_text (GTK_ENTRY (entry), "");

  /* AVG
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));

  if (help != NULL && strlen (help) > 0)
    {
      /* The avg initial mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);
	  
	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  gtk_entry_set_text (GTK_ENTRY (entry), "");

  g_string_append_printf (gs, 
			  _("Cleared initial (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);
    
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);

  return;
}


void 
polyxcalc_ui_calculator_wnd_res_masses_add_to_init_button (GtkWidget *button,
							   gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;
  
  GString *gs = NULL;
  
  gchar *help = 0;
    

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);



  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  

  /* MONO
   */

  /* First try to store the present init mono mass, so that we can
   * redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the
       * mass into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  /* Second, get the result mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);
  
  /* AVG
   */

  /* First try to store the present initial avg mass, so that we can
   * redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg initial mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value to
       * the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  /* Second, get the result mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 0);
  
  /* Write the operation to the log.
   */
  g_string_append_printf (gs, 
			  _("Added result (mono: %f ; avg: %f)\n"
			    "to initial (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg,
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);


  /* Now we can perform the calculation.
   */

  /* Make the MONO calculation per se:
   */
  libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 
				  calcctxt->masspair_init->mono 
				  + calcctxt->masspair_temp->mono);
  
  /* Make the AVG calculation per se:
   */
  libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 
				 calcctxt->masspair_init->avg 
				 + calcctxt->masspair_temp->avg);
  
  /* Finally display the masses in their respective entry widget:
   */

  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);

  /* Finally conclude the operation's log.
   */
  g_string_append_printf (gs, 
			  _("\t\t==> mono: %f ; avg: %f\n"), 
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg);

  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_res_masses_rem_from_init_button (GtkWidget *button,
							     gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;
  
  GString *gs = NULL;
  
  gchar *help = 0;
    

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);



  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  

  /* MONO
   */

  /* First try to store the present init mono mass, so that we can
   * redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the
       * mass into masspair_init, so that later we can set this value
       * to the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  /* Second, get the result mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);
  
  /* AVG
   */

  /* First try to store the present initial avg mass, so that we can
   * redisplay it later in the init mass entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg initial mass contains something, so we store the mass
       * into masspair_init, so that later we can set this value to
       * the init mass entry.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  /* Second, get the result mass into masspair_temp:
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 0);
  
  /* Write the operation to the log.
   */
  g_string_append_printf (gs, 
			  _("Removed result (mono: %f ; avg: %f)\n"
			    "from initial (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg,
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);


  /* Now we can perform the calculation.
   */

  /* Make the MONO calculation per se:
   */
  libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 
				  calcctxt->masspair_init->mono 
				  - calcctxt->masspair_temp->mono);
  
  /* Make the AVG calculation per se:
   */
  libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 
				 calcctxt->masspair_init->avg 
				 - calcctxt->masspair_temp->avg);
  
  /* Finally display the masses in their respective entry widget:
   */

  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);

  /* Finally conclude the operation's log.
   */
  g_string_append_printf (gs, 
			  _("\t\t==> mono: %f ; avg: %f\n"), 
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg);

  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_res_masses_send_to_init_button (GtkWidget *button,
							    gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry_init = NULL;
  GtkWidget *entry_res = NULL;

  GString *gs = NULL;

  gchar *help = NULL;
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  
  
  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");

  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  libpolyxmass_masspair_reset (calcctxt->masspair_init);


  /* MONO
   */

  /* First try to store the present init mono mass, so that we can
     use its value later in the recorder.
  */
  entry_init = g_object_get_data (G_OBJECT (calculator_wnd), 
				  "initial_mono_entry");
  g_assert (entry_init != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono initial mass contains something, so we store the
	 mass into masspair_init, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);

  /* Now that we have stored the previous value of initial mass, we can
     copy result value into the init entry widget.
  */
  entry_res = g_object_get_data (G_OBJECT (calculator_wnd),
				 "result_mono_entry");
  g_assert (entry_res != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  g_assert (help != NULL);

  gtk_entry_set_text (GTK_ENTRY (entry_init), help);

  /* Finally, make 100% sure that the new value in the init masses
     widget is actually a proper number, and store its value for the
     recorder.
  */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono init mass contains something, so we store the mass
	 into masspair_temp, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_temp, 0);


  /* AVG
   */

  /* First try to store the present init avg mass, so that we can
     use its value later in the recorder.
  */
  entry_init = g_object_get_data (G_OBJECT (calculator_wnd), 
				  "initial_avg_entry");
  g_assert (entry_init != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
	 into masspair_init, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);

  /* Now that we have stored the previous value of init mass, we can
     copy result value into the init entry widget.
  */
  entry_res = g_object_get_data (G_OBJECT (calculator_wnd), "result_avg_entry");
  g_assert (entry_res != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_res));
  g_assert (help != NULL);

  gtk_entry_set_text (GTK_ENTRY (entry_init), help);

  /* Finally, make 100% sure that the new value in the init masses
     widget is actually a proper number, and store its value for the
     recorder.
  */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry_init));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg init mass contains something, so we store the mass
	 into masspair_temp, so that later we can use this value.
      */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  g_string_free (gs, TRUE);

	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_temp, 0);


  /* Finally, we can ensure that the modification gets recorded.
   */    
  g_string_append_printf 
    (gs, 
     _("Sent result (mono: %f ; avg: %f) \n"
       "to initial (was mono: %f ; avg: %f):\n"
       "\t\t==> mono: %f ; avg: %f\n"),
     calcctxt->masspair_temp->mono, calcctxt->masspair_temp->avg,
     calcctxt->masspair_init->mono, calcctxt->masspair_init->avg,
     calcctxt->masspair_temp->mono, calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);

  return ;
}


void 
polyxcalc_ui_calculator_wnd_res_masses_clear_button (GtkWidget *button,
						     gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;

  gchar *help = NULL;
  

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);


  
  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");

  libpolyxmass_masspair_reset (calcctxt->masspair_init);

  /* MONO
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));

  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);
	  
	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 0);
  
  gtk_entry_set_text (GTK_ENTRY (entry), "");

  /* AVG
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));

  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something, so we store the mass
       * into masspair_temp.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_init->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  g_string_free (gs, TRUE);
	  
	  return ;
	}
    }
  else
    libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 0);
  
  gtk_entry_set_text (GTK_ENTRY (entry), "");

  g_string_append_printf (gs, 
			  _("Cleared result (was mono: %f ; avg: %f)\n"),
			  calcctxt->masspair_init->mono, 
			  calcctxt->masspair_init->avg);
    
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);

  g_string_free (gs, TRUE);

  return;
}


void 
polyxcalc_ui_calculator_wnd_clear_all_chemistry_button (GtkWidget *button,
							gpointer data)
{
  PxmCalcCtxt *calcctxt = data;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;
  
  
  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);


  /* All the monomers/modifications/formula/atoms - related
   * entries must be cleared.
   */


  /* ATOMS
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), "atoms_combo_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd), "atoms_count_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  
  /* FORMULA
   */  
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "actform_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "actform_count_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  
  /* MONOMERS
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "monomers_combo_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "monomers_count_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  
  /* MODIFICATIONS
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "modifications_combo_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "modifications_count_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  
  /* SEQUENCE
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "sequence_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "sequence_count_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  
  return;
}


void 
polyxcalc_ui_calculator_wnd_mz_ratio_calculator_button (GtkWidget *button,
							gpointer data)
{
  GtkWidget *parent_wnd = NULL;
  GtkWidget *window = NULL;
  
  PxmCalcCtxt *calcctxt = data;
  
  GPtrArray *atomGPA = NULL;
  

  PxmIonizerule *ionizerule = NULL;
  

  g_assert (calcctxt != NULL);


  atomGPA = calcctxt->atomGPA;
  g_assert (atomGPA != NULL);

  parent_wnd = calcctxt->calculator_wnd;
  g_assert (parent_wnd != NULL);
  
  ionizerule = pxmchem_ionizerule_new ();
  
  pxmchem_ionizerule_set_actform (ionizerule, "+H1");

  pxmchem_ionizerule_set_charge (ionizerule, 1);

  pxmchem_ionizerule_set_level (ionizerule, 1);


  /* We call the mz ratio calculator window with a number of
     parameters and not simply the calcctxt pointer, because that same
     window might be called through the polyxedit module.
  */
  window = polyxmass_mz_ratio_wnd_setup (atomGPA, 
					 ionizerule,
					 parent_wnd,
					 "POLYXCALC");
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to open the m/z ratio calculator window\n"),
		  __FILE__, __LINE__);
      
      return;
    }
  
  /* Register this results' window !
   */
  if (NULL == polyxmass_winmngmt_register_window 
      ("POLYXCALC",
       window,
       parent_wnd,
       NULL,
       _("Results of m/z Calculations"),
       NULL,
       TRUE,
       polyxmass_mz_ratio_wnd_make_report))
    g_critical (_("%s@%d: failed to register the m/z calculations "
		  "results' window\n"),
		__FILE__, __LINE__);
  
  return;
}




/* WINDOW LIFE CYCLE-RELATED FUNCTIONS
 */
gboolean
polyxcalc_ui_calculator_wnd_delete_event (GtkWidget *widget,
					  GdkEvent *event,
					  gpointer data)
{
  /* We should remove the polyxcalc context from the array of
     currently existent contexts.
  */
  if (FALSE == g_ptr_array_remove (polyxcalc_calcctxtGPA, 
				   (PxmCalcCtxt *) data))
    {
      g_critical (_("%s@%d: failed to remove context from array.\n"),
	     __FILE__, __LINE__);
    }
  
  polyxcalc_calcctxt_free ((PxmCalcCtxt *) data);
  
  return TRUE;
}


gboolean
polyxcalc_ui_calculator_wnd_destroy_event (GtkWidget *widget,
					   GdkEvent *event,
					   gpointer data)
{
  return FALSE;
}


