//
// Utility.cpp
//
// $Id: //poco/1.2/NetSSL_OpenSSL/src/Utility.cpp#4 $
//
// Library: NetSSL_OpenSSL
// Package: SSLCore
// Module:  Utility
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Net/Utility.h"
#include "Poco/String.h"
#include "Poco/Util/OptionException.h"
#include <openssl/err.h>


namespace Poco {
namespace Net {


int Utility::HTTPS_PORT = 443;


Context::VerificationMode Utility::convertVerificationMode(const std::string& vMode)
{
	std::string mode = Poco::toLower(vMode);
	Context::VerificationMode verMode = Context::VERIFY_STRICT;

	if (mode == "none")
		verMode = Context::VERIFY_NONE;
	else if (mode == "relaxed")
		verMode = Context::VERIFY_RELAXED;
	else if (mode == "strict")
		verMode = Context::VERIFY_STRICT;
	else if (mode == "once")
		verMode = Context::VERIFY_ONCE;
	else
		throw Poco::Util::OptionException(std::string("Wrong value >") + vMode + std::string("< for a verificationMode. Can only be none, relaxed, strict or once."));

	return verMode;
}


std::string Utility::convertCertificateError(long errCode)
{
	std::string errMsg(X509_verify_cert_error_string(errCode));
	return errMsg;
}


std::string Utility::convertSSLError(SSL* pSSL, int errCode)
{
	
	std::string errMsg;
	if (errCode > 0) return "no error";

	int connectErr = SSL_get_error(pSSL, errCode);
	long lErr = 0;
	char buf[512];
	
	switch (connectErr)
	{
	case SSL_ERROR_ZERO_RETURN:
		// connection closed
		errMsg = "connection closed by server";
		break;
	case SSL_ERROR_WANT_READ:
		errMsg = "want read";
		break;
	case SSL_ERROR_WANT_WRITE:
		errMsg = "want write";
		break;
	case SSL_ERROR_WANT_CONNECT: 
		errMsg = "want connect";
		break;
	case SSL_ERROR_WANT_ACCEPT:
		errMsg = "want accept";
		break;
	case SSL_ERROR_WANT_X509_LOOKUP:
		errMsg = "want lookup";
		break;
	case SSL_ERROR_SYSCALL:
		errMsg = "syscall";
		break;
	case SSL_ERROR_SSL:
		lErr = ERR_get_error();
		if (errCode == 0)
		{
			errMsg = "EOF was observed";
		}
		else if (errCode == -1)
		{
			errMsg = "The underlying BIO reported an I/O error";
		}
		else
		{
			ERR_error_string_n(lErr, buf, 512);
			errMsg = buf;
		}
		break;
	default:
		errMsg = "none";
	}

	return errMsg;
}


} } // namespace Poco::Net
