//
// Context.cpp
//
// $Id: //poco/1.2/NetSSL_OpenSSL/src/Context.cpp#4 $
//
// Library: NetSSL_OpenSSL
// Package: SSLCore
// Module:  Context
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Net/Context.h"
#include "Poco/Net/SSLManager.h"
#include "Poco/Net/SSLException.h"
#include "Poco/File.h"
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/ssl.h>
#include <openssl/x509v3.h>


using Poco::File;


namespace Poco {
namespace Net {


Context::Context(
		const std::string& privateKeyFile, 
		const std::string& caLocation, 
		bool isServerContext,
		VerificationMode verMode,
		int verificationDepth,
		bool loadCAFromDefaultPath,
		const std::string& cypherList):_pSSLContext(0), _mode(verMode), _server(isServerContext)
{
	
	_pSSLContext = SSL_CTX_new(SSLv23_method());
	SSL_CTX_set_default_passwd_cb(_pSSLContext, &SSLManager::privateKeyPasswdCallback);

	File aFile(caLocation);
	int errCode = 0;
	if (aFile.isDirectory())
		errCode = SSL_CTX_load_verify_locations(_pSSLContext, 0, caLocation.c_str());
	else
		errCode = SSL_CTX_load_verify_locations(_pSSLContext, caLocation.c_str(), 0);
	if (errCode != 1)
	{
		SSL_CTX_free(_pSSLContext);
		_pSSLContext = 0;
		throw SSLContextException(std::string("Failed to load CA file/directory from ") + caLocation);
	}

	if (loadCAFromDefaultPath)
	{
		errCode = SSL_CTX_set_default_verify_paths(_pSSLContext);
		if (errCode != 1)
		{
			SSL_CTX_free(_pSSLContext);
			_pSSLContext = 0;
			throw SSLContextException(std::string("Failed to load CA file/directory from default location"));
		}
	}

	errCode = SSL_CTX_use_certificate_chain_file(_pSSLContext, privateKeyFile.c_str());
	if (errCode != 1)
	{
		SSL_CTX_free(_pSSLContext);
		_pSSLContext = 0;
		throw SSLContextException(std::string("Error loading certificate from file ") + privateKeyFile);
	}
	File tmp(privateKeyFile);
	poco_assert (tmp.exists());
	errCode = SSL_CTX_use_PrivateKey_file(_pSSLContext, privateKeyFile.c_str(), SSL_FILETYPE_PEM);
	if (errCode != 1)
	{
		SSL_CTX_free(_pSSLContext);
		_pSSLContext = 0;
		throw SSLContextException(std::string("Error loading private key from file ") + privateKeyFile);
	}
	int flags = (int)verMode;
	if (verMode == VERIFY_STRICT || verMode == VERIFY_ONCE)
		flags |= SSL_VERIFY_PEER;
	if (serverContext())
		SSL_CTX_set_verify(_pSSLContext, flags, &SSLManager::verifyServerCallback);
	else
		SSL_CTX_set_verify(_pSSLContext, flags, &SSLManager::verifyClientCallback);
	SSL_CTX_set_verify_depth(_pSSLContext, verificationDepth);
	SSL_CTX_set_mode(_pSSLContext, SSL_MODE_AUTO_RETRY);
}


Context::~Context()
{
	if (_pSSLContext)
	{
		SSL_CTX_free(_pSSLContext);
		_pSSLContext = 0;
	}
}


} } // namespace Poco::Net
