/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.instrument.ClassFileTransformer;
import java.lang.instrument.IllegalClassFormatException;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.HashSet;
import java.util.Set;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.Files;

/**
 * バイトコード変換を行うための抽象クラスです。
 * このクラスは状態を保持しますが、同期化されていません。
 * <p>
 * @author cypher256
 */
abstract public class AbstractTransformer implements ClassFileTransformer {

	/** ロガー */
	private static final Logger log = Logger.getLogger(AbstractTransformer.class);

	/** Javassist クラス・プール */
	private static final ClassPool classPool = new ClassPool(true);

	/** クラス・プールのクラスパスに追加済みロケーション・パス・セット */
	private static final Set<String> appendedLocationPathSet = new HashSet<String>();

	//-------------------------------------------------------------------------

	/**
	 * バイトコードを変換します。
	 * <p>
	 * @see ClassFileTransformer#transform(ClassLoader, String, Class,
	 *          java.security.ProtectionDomain, byte[])
	 */
	public byte[] transform(
			ClassLoader loader,
			String internalName,
			Class<?> classBeingRedefined,
			ProtectionDomain protectionDomain,
			byte[] bytecode) throws IllegalClassFormatException {

		long start = System.nanoTime();

		try {
			// JDK はバイトコード変換しない
			if (
				internalName.startsWith("java/") ||
				internalName.startsWith("javax/") ||
				internalName.startsWith("sun/") ||
				internalName.startsWith("com/sun/") ||
				internalName.startsWith("org/apache/commons")) {

				return null;
			}
			// トランスフォーマー内で使用するクラスはロード処理がループするため除外
			if (internalName.startsWith("jp/sourceforge/mergedoc/pleiades/aspect/resource/")) {
				if (
					internalName.endsWith("/AbstractClassCache") ||
					internalName.endsWith("/TransformedClassCache") ||
					internalName.endsWith("/ExcludesClassNameCache")) {

					return null;
				}
			}

			if (protectionDomain != null) {

				URL location = protectionDomain.getCodeSource().getLocation();
				if (location != null) {

					String locationPath = location.getPath();
					if (locationPath.contains("org.apache.jasper_")) {

						// Eclipse ヘルプ HTTP 500 エラー回避
						// jasper プラグインはバイトコード変換しない
						return null;
					}

					// バイトコード変換に必要な関連クラスパスをクラス・プールに追加。
					// パフォーマンスのために先に追加済みか検査する。
					if (!appendedLocationPathSet.contains(locationPath)) {

						if (locationPath.contains("org.eclipse.")) {
							if (
								locationPath.contains("org.eclipse.core.resources_") ||
								locationPath.contains("org.eclipse.core.runtime_") ||
								locationPath.contains("org.eclipse.debug.core_") ||
								locationPath.contains("org.eclipse.dltk.core_") ||
								locationPath.contains("org.eclipse.equinox.common_") ||
								locationPath.contains("org.eclipse.jdt.core_") ||
								locationPath.contains("org.eclipse.jdt.ui_") ||
								locationPath.contains("org.eclipse.jface_") ||
								locationPath.contains("org.eclipse.osgi_") ||
								locationPath.contains("org.eclipse.swt.") ||
								locationPath.contains("org.eclipse.ui.workbench_")) {

								addClassPath(locationPath);
							}
						}
						if (locationPath.contains("mergedoc.jstyle.swt")) {

							addClassPath(locationPath);
						}
					}
				}
			}

		} catch (Throwable e) {

			String msg = "バイトコード変換事前処理に失敗しました。" + internalName;
			log.error(e, msg);
			throw new IllegalClassFormatException(msg + " 原因:" + e);

		} finally {

			Analyses.end(AbstractTransformer.class, "transform", start);
		}

		try {
			// バイトコード変換テンプレート・メソッドの呼び出し
			String className = internalName.replace('/', '.');
			return transform(loader, className, protectionDomain, bytecode);

		} catch (Throwable e) {

			String msg = "バイトコード変換に失敗しました。" + internalName;

			// プラグイン開発で ViewPart などを使用している発生する場合あり
			//log.error(e, msg);
			//throw new IllegalClassFormatException(msg + " 原因:" + e);
			log.debug(msg + " " + e);
			return null;
		}
	}

	/**
	 * クラスパスを追加します。
	 * <p>
	 * @param locationPath ロケーション・パス
	 * @throws NotFoundException 指定されたパスが見つからない場合
	 */
	private void addClassPath(String locationPath) throws NotFoundException {

		appendedLocationPathSet.add(locationPath);
		String classPath = Files.decodePath(locationPath);
		classPool.appendClassPath(classPath);
	}

	/**
	 * バイトコードを変換するためのテンプレート・メソッドです。
	 * <p>
	 * @param loader クラス・ローダー
	 * @param className クラス名
	 * @param protectionDomain プロテクション・ドメイン
	 * @param bytecode バイトコード
	 * @return 変換後のバイトコード。変換する必要が無い場合は null。
	 * @throws CannotCompileException クラスがコンパイル出来ない場合
	 * @throws NotFoundException クラスが見つからない場合
	 * @throws IOException 入出力例外が発生した場合
	 */
	abstract protected byte[] transform(
			ClassLoader loader,
			String className,
			ProtectionDomain protectionDomain,
			byte[] bytecode
	) throws CannotCompileException, NotFoundException, IOException;

	/**
	 * バイトコードを指定して CtClass オブジェクトを作成します。
	 * <p>
	 * @param bytecode バイトコード
	 * @param protectionDomain プロテクション・ドメイン
	 * @return CtClass オブジェクト
	 * @throws IOException 入出力例外が発生した場合
	 * @throws NotFoundException クラスが見つからない場合
	 */
	protected CtClass createCtClass(
			byte[] bytecode,
			ProtectionDomain protectionDomain) throws IOException, NotFoundException {

		long start = System.nanoTime();

		CtClass ctClass = classPool.makeClass(new ByteArrayInputStream(bytecode));
		ctClass.detach();

		Analyses.end(AbstractTransformer.class, "createCtClass", start);
		return ctClass;
	}

	/**
	 * 指定された CtClass オブジェクトが持つすべてのコンストラクタおよび
	 * メソッド記述子をログに出力します。
	 * <p>
	 * @param ctClass CtClass オブジェクト
	 */
	protected void logDescriptor(CtClass ctClass) {
		for (CtBehavior behavior : ctClass.getDeclaredBehaviors()) {
			log.info("DESCRIPTOR: " +
					ctClass.getName() + "#" +
					behavior.getName() + " " +
					behavior.getMethodInfo().getDescriptor());
		}
	}
}
