/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
\*************************************************************************/

//--- LOCAL ---
#include "countrymap.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QApplication>
#include <QFile>
#include <QIcon>
#include <QMutex>
#include <QTextStream>
#include <QVariant>

//--- KDE4 ---
#include <kglobal.h>
#include <klocale.h>
#include <kstandarddirs.h>

#include <QTime>


QMutex CountryMap::singeltonMutex;
QMutex UsStatesMap::singeltonMutex;

QPointer<CountryMap>    CountryMap::m_pInstance = NULL;
QPointer<UsStatesMap>   UsStatesMap::m_pInstance = NULL;

const QString CountryMap::sFlagTemplate("l10n/%1/flag.png");


CountryMap::CountryMap *
CountryMap::instance()
{
	QMutexLocker locker( &singeltonMutex );
	if( !m_pInstance )
		m_pInstance = new CountryMap( QApplication::instance() );
	return m_pInstance;
}

void
CountryMap::clear()
{
	QMutexLocker locker( &singeltonMutex );
	if( m_pInstance )
		delete m_pInstance;
}

CountryMap::CountryMap( QObject * parent )
	: QAbstractListModel( parent ),
	  m_vPixmapCache( 75 )
{
	QFile resource(QLatin1String(":/countries.list") );
	if( resource.open( QIODevice::ReadOnly | QIODevice::Text ) )
	{
		QTextStream stream(&resource);
		QString sLine = stream.readLine();
		QString sCountryCode;
		QString sCountry;
		while( !sLine.isNull() )
		{
			sCountryCode = sLine.left(2);
			sCountry = sLine.right( sLine.length() - 2 ).simplified();
			m_vCountryMap.insert( sCountry, sCountryCode );
	
			sLine = stream.readLine();
		}
	}
}

CountryMap::~CountryMap()
{
	m_pInstance = NULL;
	qDebug() << "CountryMap will be removed...";
}

int
CountryMap::rowCount ( const QModelIndex & parent ) const
{
	Q_UNUSED( parent )
	return m_vCountryMap.count();
}

QVariant
CountryMap::headerData ( int section, Qt::Orientation orientation, int role ) const
{
	if( orientation == Qt::Horizontal && role == Qt::DisplayRole && section == 0 )
		return i18n("Country");
	else
		return QAbstractListModel::headerData(section, orientation, role);
}

QVariant
CountryMap::data( const QModelIndex & index, int role ) const
{
	QMutexLocker locker( &singeltonMutex );

	QVariant var;
	if( !index.isValid() || index.row() >= m_vCountryMap.count() )
		return var;

	QMap<QString, QString>::const_iterator it = m_vCountryMap.begin() + index.row();
	if( Qt::DisplayRole == role )
		var = it.key();
	else if( Qt::DecorationRole == role )
	{
		QPixmap * pm = private_getPixmapForCountryCode( it.value() );
		if( pm )
			var = QIcon(*pm);
	}
	else if( CountryCodeRole == role )
		var = it.value();
	return var;
}

QPixmap
CountryMap::getPixmapForCountry( const QString & country ) const
{
	return getPixmapForCountryCode( m_vCountryMap.value(country) );
}

QPixmap
CountryMap::getPixmapForCountryCode( const QString & countryCode ) const
{
	QMutexLocker locker( &singeltonMutex );
	QPixmap * pm = private_getPixmapForCountryCode( countryCode );
	if( pm )
		return *pm;
	return QPixmap();
}

QString
CountryMap::countryCode( const QString & country ) const
{
	QMutexLocker locker( &singeltonMutex );
	return m_vCountryMap.value( country );
}

int
CountryMap::countryIndex( const QString & country ) const
{
	QMutexLocker locker( &singeltonMutex );

	QMap<QString, QString>::const_iterator it = m_vCountryMap.constBegin();
	int idx = 0;
	while( it != m_vCountryMap.constEnd() )
	{
		if( country.compare( it.key(), Qt::CaseInsensitive ) == 0 )
			return idx;
		it++;
		idx += 1;
	}
	return -1;
}
int
CountryMap::countryCodeIndex( const QString & countryCode )
{
	QMutexLocker locker( &singeltonMutex );

	QMap<QString, QString>::const_iterator it = m_vCountryMap.constBegin();
	int idx = 0;
	while( it != m_vCountryMap.constEnd() )
	{
		if( countryCode.compare( it.value(), Qt::CaseInsensitive ) == 0 )
			return idx;
		it++;
		idx += 1;
	}
	return -1;
}

QPixmap *
CountryMap::private_getPixmapForCountryCode( const QString & countryCode ) const
{
	if( countryCode.isEmpty() )
		return NULL;

	QPixmap * pm = NULL;
	pm = m_vPixmapCache.object(countryCode);
	if( !pm )
	{
		QString sFlag;
		sFlag = KStandardDirs::locate("locale", sFlagTemplate.arg(countryCode));
		if( !sFlag.isEmpty() )
		{
			pm = new QPixmap(sFlag);
			m_vPixmapCache.insert( countryCode, pm );
		}
	}
	return pm;
}

/******************************************************************************/

UsStatesMap::UsStatesMap *
UsStatesMap::instance()
{
	QMutexLocker locker( &singeltonMutex );
	if( !m_pInstance )
		m_pInstance = new UsStatesMap( QApplication::instance() );
	return m_pInstance;
}

void
UsStatesMap::clear()
{
	QMutexLocker locker( &singeltonMutex );
	if( m_pInstance )
		delete m_pInstance;
}
UsStatesMap::UsStatesMap( QObject * parent )
	: QAbstractTableModel( parent )
{
	QFile resource( QLatin1String(":/us_states.list") );
	if( resource.open( QIODevice::ReadOnly | QIODevice::Text ) )
	{
		QTextStream stream(&resource);
		QString sLine = stream.readLine();
		QString sStateCode;
		QString sState;
		while( !sLine.isNull() )
		{
			sStateCode = sLine.left(2);
			sState = sLine.right( sLine.length() - 2 ).simplified();
			m_vStates.insert( sStateCode, sState );
	
			sLine = stream.readLine();
		}
	}
}

UsStatesMap::~UsStatesMap()
{
	m_pInstance = NULL;
	qDebug() << "USStatesMap will be removed...";
}

int
UsStatesMap::columnCount( const QModelIndex & parent ) const
{
	Q_UNUSED( parent )
	return 2;
}

int
UsStatesMap::rowCount ( const QModelIndex & parent ) const
{
	Q_UNUSED( parent )
	return m_vStates.count();
}

QVariant
UsStatesMap::headerData ( int section, Qt::Orientation orientation, int role ) const
{
	QVariant var;
	if( orientation == Qt::Horizontal && role == Qt::DisplayRole )
	{
		if( section == 0 )
			var = i18n("State");
		else
			var = i18n("State Code");
		return var;
	}
	return QAbstractTableModel::headerData( section, orientation, role );
}

QVariant
UsStatesMap::data(const QModelIndex & index, int role ) const
{
	QMutexLocker locker( &singeltonMutex );

	QVariant var;
	if( !index.isValid() || index.row() >= m_vStates.count() )
		return var;

	QMap<QString, QString>::const_iterator it = m_vStates.begin() + index.row();
	if( Qt::DisplayRole == role )
	{
		if( index.column() == 0 )
			var = it.value();
		else if( index.column() == 1 )
			var = it.key();
	}
	return var;
}

QString
UsStatesMap::state( const QString & stateCode ) const
{
	QMutexLocker locker( &singeltonMutex );
	return m_vStates.value( stateCode );
}

int
UsStatesMap::stateCodeIndex( const QString & stateCode ) const
{
	QMutexLocker locker( &singeltonMutex );

	QMap<QString, QString>::const_iterator it = m_vStates.constBegin();
	int idx = 0;
	while( it != m_vStates.constEnd() )
	{
		if( stateCode.compare( it.key(), Qt::CaseInsensitive ) == 0 )
			return idx;
		it++;
		idx += 1;
	}
	return -1;
}

int
UsStatesMap::stateIndex( const QString & state ) const
{
	QMutexLocker locker( &singeltonMutex );

	QMap<QString, QString>::const_iterator it = m_vStates.constBegin();
	int idx = 0;
	while( it != m_vStates.constEnd() )
	{
		if( state.compare( it.value(), Qt::CaseInsensitive ) == 0 )
			return idx;
		it++;
		idx += 1;
	}
	return -1;
}
