/*
 * effectselector.c - provides a list of available effects.
 *
 * Copyright (c) 2007 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 */

#include <iostream>

#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklist.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkscrolledwindow.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include "../support/generaldialogs.h"

#include "ppeffect.h"
#include "ppeffect_desaturate.h"
#include "ppeffect_temperature.h"

#include "effectlist.h"

#include "effectselector.h"

#include "../config.h"
#include "../gettext.h"
#define _(x) gettext(x)
#define N_(x) gettext_noop(x)

using namespace std;

enum {
	CHANGED_SIGNAL,
	LAST_SIGNAL
};

static guint effectselector_signals[LAST_SIGNAL] = { 0 };

static void effectselector_class_init (EffectSelectorClass *klass);
static void effectselector_init (EffectSelector *sel);


static EffectListItem *findbyname(EffectSelector *il,const char *name)
{
	GList *iter=il->imagelist;
	while(iter)
	{
		EffectListItem *ii=(EffectListItem *)iter->data;
		if(ii && (ii->name==NULL) && (name==NULL))
			return(ii);
		if(ii && ii->name && strcmp(ii->name,name)==0)
			return(ii);
		iter=g_list_next(iter);
	}
	cerr << name << " Not found" << endl;
	return(NULL);
}


static EffectListItem *findbypixbuf(EffectSelector *il,GdkPixbuf *pb)
{
	GList *iter=il->imagelist;
	while(iter)
	{
		EffectListItem *ii=(EffectListItem *)iter->data;
		if(ii && ii->GetIcon()==pb)
			return(ii);
		iter=g_list_next(iter);
	}
	cerr << " Not found" << endl;
	return(NULL);
}


static gint is_cmpfunc(const void *p1,const void *p2)
{
	EffectListItem *i1=(EffectListItem *)p1;
	EffectListItem *i2=(EffectListItem *)p2;
	return(strcmp(i1->name,i2->name));
}


static void clear_list(EffectSelector *il)
{
	gtk_list_store_clear(il->liststore);

	GList *iter=il->imagelist;
	while(iter)
	{
		EffectListItem *ii=(EffectListItem *)iter->data;
		GList *niter=g_list_next(iter);
		il->imagelist=g_list_delete_link(il->imagelist,iter);
		delete ii;
		iter=niter;
	}
	il->imagelist=NULL;
}


static void populate_list(EffectSelector *il)
{
	GtkTreeIter iter1;

	cerr << "Populating list..." << endl;
	for(int i=0;i<il->els->EffectCount();++i)
	{
		EffectListItem *ii=il->els->GetEffect(i);
		cerr << "Got effect..." << endl;
		il->imagelist=g_list_append(il->imagelist,ii);
		cerr << "Added..." << endl;
	}

//	cerr << "Sorting list..." << endl;
	
//	il->imagelist=g_list_sort(il->imagelist,is_cmpfunc);


	GList *liter=il->imagelist;
	while(liter)
	{
		EffectListItem *ii=(EffectListItem *)liter->data;

		gtk_list_store_append(il->liststore,&iter1);
		gtk_list_store_set(il->liststore,&iter1,0,ii->GetIcon(),-1);

		while(gtk_events_pending())
			gtk_main_iteration_do(false);

		liter=g_list_next(liter);
	}
}


static void rebuild_liststore(EffectSelector *c)
{
	if(!c->imagelist)
		populate_list(c);
	else
	{
		// Rebuild list view from EffectSelector
		GList *iter=c->imagelist;
	
		GtkTreeIter iter1;
	
		gtk_list_store_clear(c->liststore);
	
		while(iter)
		{
			EffectListItem *ii=(EffectListItem *)iter->data;
			gtk_list_store_append(c->liststore,&iter1);
			gtk_list_store_set(c->liststore,&iter1,0,ii->GetIcon(),-1);
			iter=g_list_next(iter);
		}	
	}
}


static void selection_changed(GtkTreeSelection *select,gpointer user_data)
{
	EffectSelector *pe=EFFECTSELECTOR(user_data);
	if(const EffectListItem *ii=effectselector_get_selected(pe))
	{
		g_signal_emit(G_OBJECT (pe),effectselector_signals[CHANGED_SIGNAL], 0);
	}
}


GtkWidget*
effectselector_new (EffectListSource *els)
{
	EffectSelector *c=EFFECTSELECTOR(g_object_new (effectselector_get_type (), NULL));

	c->liststore=gtk_list_store_new(1,GDK_TYPE_PIXBUF);
	c->els=els;

	GtkWidget *sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (c), sw, TRUE, TRUE, 0);
	gtk_widget_show(sw);

	c->treeview=gtk_tree_view_new_with_model(GTK_TREE_MODEL(c->liststore));
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	
	renderer=gtk_cell_renderer_pixbuf_new();
	column=gtk_tree_view_column_new_with_attributes("Effect",renderer,"pixbuf",0,NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(c->treeview),column);

	GtkTreeSelection *select;
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (c->treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
		G_CALLBACK (selection_changed),c);

	gtk_container_add(GTK_CONTAINER(sw),c->treeview);
	gtk_widget_show(c->treeview);
	
	GtkWidget *hbox=gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(c),hbox,FALSE,FALSE,0);
	gtk_widget_show(hbox);

	populate_list(c);
	
	return(GTK_WIDGET(c));
}


GType
effectselector_get_type (void)
{
	static GType stpuic_type = 0;

	if (!stpuic_type)
	{
		static const GTypeInfo effectselector_info =
		{
			sizeof (EffectSelectorClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) effectselector_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (EffectSelector),
			0,
			(GInstanceInitFunc) effectselector_init,
		};
		stpuic_type = g_type_register_static (GTK_TYPE_VBOX, "EffectSelector", &effectselector_info, GTypeFlags(0));
	}
	return stpuic_type;
}


static void *parent_class=NULL;

static void effectselector_destroy(GtkObject *object)
{
	EffectSelector *il=(EffectSelector *)object;
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);

	clear_list(il);
}


static void
effectselector_class_init (EffectSelectorClass *cls)
{
	GtkObjectClass *object_class=(GtkObjectClass *)cls;
//	GtkWidgetClass *widget_class=(GtkWidgetClass *)cls;

	parent_class = gtk_type_class (gtk_widget_get_type ());

	object_class->destroy = effectselector_destroy;

	effectselector_signals[CHANGED_SIGNAL] =
	g_signal_new ("changed",
		G_TYPE_FROM_CLASS (cls),
		GSignalFlags(G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION),
		G_STRUCT_OFFSET (EffectSelectorClass, changed),
		NULL, NULL,
		g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}


static void
effectselector_init (EffectSelector *c)
{
	c->imagelist=NULL;
}


gboolean effectselector_refresh(EffectSelector *c)
{
	clear_list(c);
	rebuild_liststore(c);
	return(true);
}


EffectListItem *effectselector_get_selected(EffectSelector *pe)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	
	GtkTreeSelection *select;
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (pe->treeview));

	if (gtk_tree_selection_get_selected (select,&model, &iter))
	{
		GdkPixbuf *pb;

		gtk_tree_model_get (model, &iter, 0, &pb, -1);
		if(pb)
		{
			EffectListItem *ii=findbypixbuf(pe,pb);
			return(ii);
		}
	}
	return(NULL);
}

#if 0

void effectselector_set_filename(EffectSelector *c,const char *filename)
{
	ImageEntry *ii=find_filename(c,filename);

	if(!ii)
		ii=add_node(c,filename);

	if(ii)
	{
		if(c->filename)
			free(c->filename);
		c->filename=NULL;
	
		if(filename)
			c->filename=strdup(filename);
		
		GtkTreeIter iter;
		GtkTreePath *path;
	
		if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(c->liststore),&iter))
		{
			do
			{
				GdkPixbuf *pb;
				gtk_tree_model_get(GTK_TREE_MODEL(c->liststore),&iter,0,&pb,-1);
				if(pb==ii->pixbuf)
				{
					path=gtk_tree_model_get_path(GTK_TREE_MODEL(c->liststore),&iter);
					gtk_tree_view_set_cursor(GTK_TREE_VIEW(c->treeview),path,NULL,false);
					gtk_tree_path_free(path);
					break;
				}
			} while(gtk_tree_model_iter_next(GTK_TREE_MODEL(c->liststore),&iter));
		}
	}
}
#endif
