/*

Copyright (C) 2007 Robert S. Edmonds 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

#include <getopt.h>
#include <stdio.h>
#include <string.h>

#include "cfgopt.h"
#include "util.h"

/* declarations */

static void cfgopt_load_value(cfgopt_t *cfg, char *value);

/* functions */

bool cfgopt_load(cfgopt_t *cfg, char *fname){
	FILE *fp;
	size_t len;
	char *line = NULL;
	bool success = true;

	if(!cfg || !fname)
		return false;
	if(!(fp = fopen(fname, "r")))
		ERROR("fopen failed: %s", strerror(errno));
	while(getline(&line, &len, fp) != -1){
		char *tok_key;
		char *tok_val;
		tok_key = strtok(line, "=\"\t\n");
		if(!tok_key) continue;
		tok_val = strtok(NULL, "=\"\t\n");
		if(!tok_val){
			DEBUG("null value for key='%s'", tok_key);
			continue;
		}

		cfgopt_t *cur = cfgopt_get(cfg, tok_key);
		if(!cur){
			DEBUG("unknown configuration key '%s'", tok_key);
			success = false;
			continue;
		}else{
			cfgopt_load_value(cur, tok_val);
		}

		FREE(line);
	}
	for(; cfg->type != CONFIG_END ; ++cfg)
		if(!cfg->val.str)
			cfgopt_load_value(cfg, cfg->default_value);
	FREE(line);
	fclose(fp);
	return success;
}

static void cfgopt_load_value(cfgopt_t *cfg, char *value){
	if(!value) return;
	switch(cfg->type){
	case CONFIG_STR: {
		cfg->val.str = strdup(value);
		break;
	}
	case CONFIG_DEC:
	case CONFIG_OCT: {
		int base = cfg->type == CONFIG_DEC ? 0 : 8;
		if(!cfg->val.num) NEW0(cfg->val.num);
		*cfg->val.num = strtoul(value, NULL, base);
		break;
	}
	case CONFIG_BOOL: {
		if(!cfg->val.boolean) NEW0(cfg->val.boolean);
		*cfg->val.boolean = strtoul(value, NULL, 0) == true;
		break;
	}
	default:
		ERROR("invalid configuration type %d", (int) cfg->type);
	}
}

cfgopt_t *cfgopt_get(cfgopt_t *cfg, char *key){
	for(; cfg->type != CONFIG_END ; ++cfg){
		if(strcasecmp(cfg->key, key) == 0)
			return cfg;
	}
	return NULL;
}

char *cfgopt_get_str(cfgopt_t *cfg, char *key){
	for(; cfg->type != CONFIG_END ; ++cfg){
		if(strcasecmp(cfg->key, key) == 0)
			return cfg->val.str;
	}
	return NULL;
}

char *cfgopt_get_str_dup(cfgopt_t *cfg, char *key){
	for(; cfg->type != CONFIG_END ; ++cfg){
		if(strcasecmp(cfg->key, key) == 0)
			if(cfg->val.str)
				return strdup(cfg->val.str);
	}
	return NULL;
}

long cfgopt_get_num(cfgopt_t *cfg, char *key){
	for(; cfg->type != CONFIG_END ; ++cfg){
		if(strcasecmp(cfg->key, key) == 0)
			if(cfg->val.num)
				return *cfg->val.num;
	}
	return 0;
}

void cfgopt_free(cfgopt_t *cfg){
	for(; cfg->type != CONFIG_END ; ++cfg){
		FREE(cfg->val.str);
	}
}

bool cfgopt_is_set(cfgopt_t *cfg, char *key){
	for(; cfg->type != CONFIG_END ; ++cfg){
		if((strcasecmp(cfg->key, key) == 0) && cfg->val.str)
			return true;
	}
	return false;
}

void cfgopt_print(cfgopt_t *cfg){
	for(; cfg->type != CONFIG_END ; ++cfg){
		switch(cfg->type){
		case CONFIG_STR:
			if(cfg->val.str)
				DEBUG("key='%s', val='%s'", cfg->key, cfg->val.str);
			break;
		case CONFIG_BOOL:
			if(cfg->val.boolean)
				DEBUG("key='%s', val=%.1d", cfg->key, *cfg->val.boolean);
			break;
		case CONFIG_DEC:
			if(cfg->val.num)
				DEBUG("key='%s', val=%ld", cfg->key, *cfg->val.num);
			break;
		case CONFIG_OCT:
			if(cfg->val.num)
				DEBUG("key='%s', val=%lo", cfg->key, *cfg->val.num);
			break;
		default:
			break;
		}
	}
}

void cfgopt_usage(cfgopt_t *cfg){
	for(; cfg->type != CONFIG_END ; ++cfg){
		switch(cfg->type){
		case CONFIG_STR:
		case CONFIG_DEC:
		case CONFIG_OCT:
			fprintf(stderr, "\t[ -%c <%s> %s %s%s%s]\n",
				cfg->cmd, cfg->key, cfg->help,
				cfg->default_value == NULL ? "" : "(default: ",
				cfg->default_value == NULL ? "" : cfg->default_value,
				cfg->default_value == NULL ? "" : ") "
			);
			break;
		case CONFIG_BOOL:
			fprintf(stderr, "\t[ -%c %s %s%s%s]\n",
				cfg->cmd, cfg->help,
				cfg->default_value == NULL ? "" : "(default: ",
				cfg->default_value == NULL ? "" : cfg->default_value,
				cfg->default_value == NULL ? "" : ") "
			);
			break;
		default:
			break;
		}
	}
}

void cfgopt_parse_args(cfgopt_t *cfg, int argc __unused, char **argv __unused){
	int c;
	char *options;
	cfgopt_t *cur;

	MALLOC(options, 2 * cfgopt_len(cfg));
	options[0] = '\0';

	for(cur = cfg ; cur->type != CONFIG_END ; ++cur){
		switch(cur->type){
		case CONFIG_STR:
		case CONFIG_DEC:
		case CONFIG_OCT:
			strcat(options, &cur->cmd);
			strcat(options, ":");
			break;
		case CONFIG_BOOL:
			strcat(options, &cur->cmd);
			break;
		default:
			break;
		}
	}

	while((c = getopt(argc, argv, options)) != EOF){
		for(cur = cfg ; cur->type != CONFIG_END ; ++cur){
			if(c == cur->cmd){
				if(cur->type == CONFIG_BOOL){
					if(cur->default_value && cur->default_value[0] == '0')
						cfgopt_load_value(cur, "1");
					else if(cur->default_value && cur->default_value[0] == '1')
						cfgopt_load_value(cur, "0");
					else
						cfgopt_load_value(cur, "1");
				}else
					cfgopt_load_value(cur, optarg);
			}
		}
	}

	for(cur = cfg ; cur->type != CONFIG_END ; ++cur)
		if(!cur->val.str)
			cfgopt_load_value(cur, cur->default_value);

	if((cur = cfgopt_get(cfg, "configfile")) && cur->val.str && !cfgopt_load(cfg, cur->val.str))
		ERROR("configuration error, exiting");

	FREE(options);
}

size_t cfgopt_len(cfgopt_t *cfg){
	size_t len = 0;
	for(; cfg->type != CONFIG_END ; ++cfg) ++len;
	return len;
}
