//=======================================================================
// mainwindow.cc
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#include "config.h"
#include "gconfig.h"
#include "mainwindow.h"
#include "maintreeview.h"
#include "preferences.h"
#include "about.h"
#include "pkgset.h"
#include "pkg.h"	// TAB_*
#include "find.h"
#include "util.h"
#include "paco/common.h"	// toString()
#include <gtkmm/stock.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/actiongroup.h>
#include <gtkmm/box.h>
#include <gtkmm/separator.h>
#include <gtkmm/scrolledwindow.h>
#include <sstream>

using Glib::ustring;
using std::string;
using namespace sigc;
using namespace Gpaco;


MainWindow::MainWindow()
:
	Gtk::Window(),
	mLabel("", 0.02, 0.5),
	mpTreeView(NULL),
	mProgressBar(),
	mpUIManager(Gtk::UIManager::create()),
	mpActionGroupMenuBar(Gtk::ActionGroup::create()),
	mpActionGroupPkg(Gtk::ActionGroup::create()),
	mpActionUpdate(Gtk::Action::create("UpdateDatabase", 
		Gtk::Stock::REFRESH, "_Update database")),
	mpActionViewFiles(Gtk::Action::create("ViewFiles", 
		Gtk::Stock::DIRECTORY, "View _files")),
	mpActionProperties(Gtk::Action::create("Properties", 
		Gtk::Stock::PROPERTIES)),
	mpActionRemovePkg(Gtk::Action::create("RemovePkg", 
		Gtk::Stock::DELETE, "_Remove...")),
	mpActionCreatePackage(Gtk::Action::create("CreatePackage", 
		"Create _package...")),
	mpActionUnlogPkg(Gtk::Action::create("UnlogPkg", 
		Gtk::Stock::REMOVE, "Remove from _database..."))
{
	mpTreeView = new MainTreeView(*this);

	set_default_size(GConfig::width(), GConfig::height());
	try { 
		set_icon_from_file(DATADIR "/pixmaps/gpaco.png"); 
		mpActionCreatePackage->set_gicon(Gio::Icon::create(
			DATADIR "/pixmaps/gpaco-package.png"));
	}
	catch (...) { }

	buildUIManager();

	for (guint i(0); i < GConfig::cols().size(); ++i)
		mpTreeView->get_column(GConfig::cols()[i])->set_visible();
	
	Gtk::ScrolledWindow* pScrolledWindow(Gtk::manage(new Gtk::ScrolledWindow()));
	pScrolledWindow->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	pScrolledWindow->add(*mpTreeView);

	mProgressBar.set_size_request(-1, 0);

	Gtk::HBox* pHBox(Gtk::manage(new Gtk::HBox()));
	pHBox->pack_start(mLabel, Gtk::PACK_SHRINK, 8);
	pHBox->pack_end(mProgressBar, Gtk::PACK_SHRINK);

	mpUIManager->get_widget("/MenuBar/MenuPackage")->signal_event().connect(mem_fun(*this, &MainWindow::onPkgMenu));

	Gtk::VBox* pVBox(Gtk::manage(new Gtk::VBox()));
	pVBox->pack_start(*(mpUIManager->get_widget("/MenuBar")), Gtk::PACK_SHRINK);
	pVBox->pack_start(*pScrolledWindow);
	pVBox->pack_start(*(Gtk::manage(new Gtk::HSeparator())), Gtk::PACK_SHRINK, 1);
	pVBox->pack_start(*pHBox, Gtk::PACK_SHRINK);

	add(*pVBox);
	show_all_children();
	mProgressBar.hide();
	move(GConfig::xPos(), GConfig::yPos());

	writeLabel();
	Glib::signal_timeout().connect(mem_fun(*this, &MainWindow::writeLabel), 500);
}


MainWindow::~MainWindow()
{
	delete mpTreeView;
}


bool MainWindow::onPkgMenu(GdkEvent*)
{
	int rows = mpTreeView->countSelected();
	bool onerow = rows == 1;
	
	mpActionViewFiles		->set_sensitive(onerow);
	mpActionProperties		->set_sensitive(onerow);
	mpActionRemovePkg		->set_sensitive(onerow && GConfig::logdirWritable());
	mpActionCreatePackage	->set_sensitive(onerow);
	mpActionUnlogPkg		->set_sensitive(rows && GConfig::logdirWritable());

	return false;
}


// [virtual]
bool MainWindow::on_delete_event(GdkEventAny*)
{
	quit();
	return true;
}


void MainWindow::quit()
{
	GConfig::saveSettings(*this);
	Glib::usleep(10000);
	hide();
}


void MainWindow::buildUIManager()
{
	g_assert(mpActionGroupMenuBar);
	g_assert(mpActionGroupPkg);
	g_assert(mpTreeView);

	mpActionGroupMenuBar->add(Gtk::Action::create("MenuFile", "_File"));
	
	mpActionGroupMenuBar->add(mpActionUpdate, 
		mem_fun(*mpTreeView, &MainTreeView::onUpdateDataBase));
	mpActionUpdate->set_sensitive(GConfig::logdirWritable());
	
	mpActionGroupMenuBar->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT),
		mem_fun(*this, &MainWindow::quit));
		
	mpActionGroupMenuBar->add(Gtk::Action::create("MenuEdit", "_Edit"));
	mpActionGroupMenuBar->add(Gtk::Action::create("FindFile", Gtk::Stock::FIND, 
		"_Find file"), mem_fun(*this, &MainWindow::find));
	mpActionGroupMenuBar->add(Gtk::Action::create("Preferences", 
		Gtk::Stock::PREFERENCES), mem_fun(*this, &MainWindow::preferences));
	
	mpActionGroupMenuBar->add(Gtk::Action::create("MenuPackage", "_Package"));
	
	mpActionGroupPkg->add(mpActionViewFiles, bind<int>
		(mem_fun(*mpTreeView, &MainTreeView::onPkgWindow), TAB_FILES));
	mpActionGroupPkg->add(mpActionProperties, bind<int>(mem_fun(*mpTreeView, &MainTreeView::onPkgWindow), TAB_INFO));
	mpActionGroupPkg->add(mpActionRemovePkg, bind<int>(mem_fun(*mpTreeView, &MainTreeView::onPkgWindow), TAB_REMOVE));
	mpActionGroupPkg->add(mpActionCreatePackage, bind<int>(mem_fun(*mpTreeView, &MainTreeView::onPkgWindow), TAB_PACKAGE));
	mpActionGroupPkg->add(mpActionUnlogPkg, mem_fun(*mpTreeView, &MainTreeView::onUnlog));

	mpActionGroupMenuBar->add(Gtk::Action::create("MenuHelp", "_Help"));
	mpActionGroupMenuBar->add(Gtk::Action::create("About", Gtk::Stock::ABOUT), 
		mem_fun(*this, &MainWindow::about));

	mpUIManager->insert_action_group(mpActionGroupMenuBar);
	mpUIManager->insert_action_group(mpActionGroupPkg);
	
	add_accel_group(mpUIManager->get_accel_group());

	mpUIManager->add_ui_from_string(
		"<ui>"
		"	<menubar name='MenuBar'>"
		"		<menu action='MenuFile'>"
		"			<menuitem action='UpdateDatabase'/>"
		"			<separator/>"
		"			<menuitem action='Quit'/>"
		"		</menu>"
		"		<menu action='MenuEdit'>"
		"			<menuitem action='FindFile'/>"
		"			<separator/>"
		"			<menuitem action='Preferences'/>"
		"		</menu>"
		"		<menu action='MenuPackage'>"
		"			<menuitem action='ViewFiles'/>"
		"			<menuitem action='Properties'/>"
		"			<menuitem action='RemovePkg'/>"
		"			<menuitem action='CreatePackage'/>"
		"			<separator/>"
		"			<menuitem action='UnlogPkg'/>"
		"		</menu>"
		"		<menu action='MenuHelp'>"
		"			<menuitem action='About'/>"
		"		</menu>"
		"	</menubar>"
		"</ui>");

	mpTreeView->build_uimanager(mpActionGroupPkg);
}


void MainWindow::find()
{
	Find::instance(*this);
}


void MainWindow::preferences()
{
	Preferences* p = new Preferences(*this);
	p->set_transient_for(*this);
	p->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
	p->run();
	delete p;
}


void MainWindow::about()
{
	About* a = new About();
	a->set_transient_for(*this);
	a->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
	a->run();
	delete a;
}


bool MainWindow::isColumnVisible(int colId)
{
	return mpTreeView->get_column(colId)->get_visible();
}


void MainWindow::setColumnVisible(int colId, bool visible /* = true */)
{
	mpTreeView->get_column(colId)->set_visible(visible);
}


bool MainWindow::writeLabel()
{
	if (mProgressBar.is_visible()) {
		// Updating the database
		return true;
	}

	std::ostringstream txt;
	txt << mpTreeView->pkgSet().size() << " Packages  |  "
		<< Paco::toString(mpTreeView->pkgSet().sizeInst());
	
	mLabel.set_text(txt.str());
	
	return true;
}

