//=======================================================================
// filestreeview.cc
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#include "config.h"
#include "gconfig.h"
#include "util.h"
#include "pkg.h"
#include "pkgwindow.h"
#include "filestab.h"
#include "filestreeview.h"
#include <sstream>
#include <algorithm>
#include <gtkmm/liststore.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/actiongroup.h>
#include <gtkmm/menu.h>
#include <gtkmm/stock.h>
#include <gtkmm/progressbar.h>
#include <glibmm/spawn.h>

using Glib::ustring;
using sigc::mem_fun;
using sigc::bind;
using sigc::slot;
using std::vector;
using std::string;
using std::remove_if;
using namespace Gpaco;

// Forward decls
static bool isMissing(File*);
static bool isBzip2(string const&);
static bool isGzip(string const&);
static bool isRegular(string const&);


//--------//
// public //
//--------//


FilesTreeView::FilesTreeView(Pkg& pkg)
:
	TreeView(),
	mPkg(pkg),
	mColumns(),
	mpModelAll(Gtk::ListStore::create(mColumns)),
	mpModelInst(Gtk::ListStore::create(mColumns)),
	mpModelMiss(Gtk::ListStore::create(mColumns)),
	mpModelEmpty(Gtk::ListStore::create(mColumns)),
	mppModel(NULL),
	mpActionStrip(Gtk::Action::create("Strip", Gtk::Stock::CLEAR, "_Strip")),
	mpActionCompress(Gtk::Action::create("Compress", "_Compress")),
	mpActionUncompress(Gtk::Action::create("Uncompress", "_Uncompress")),
	mpActionRemove(Gtk::Action::create("Remove", Gtk::Stock::DELETE, "_Remove..."))
{
	Glib::RefPtr<Gtk::ListStore>* pModel[3] = { &mpModelAll, &mpModelInst, &mpModelMiss };
	for (int i = 0; i < 3; ++i) {
		(*pModel[i])->set_sort_func(COL_STATUS, mem_fun(*this, &FilesTreeView::statusSortFunc));
		(*pModel[i])->set_sort_func(COL_NAME, mem_fun(*this, &FilesTreeView::nameSortFunc));
		(*pModel[i])->set_sort_func(COL_SIZE, mem_fun(*this, &FilesTreeView::sizeSortFunc));
	}
	setModel();

	// Status column
	int id = append_column("", mColumns.mStatus) - 1;
	g_assert(id == COL_STATUS);
	Gtk::TreeViewColumn* pCol = get_column(id);
	Gtk::CellRenderer* pCell = pCol->get_first_cell_renderer();
	pCol->set_sort_column(id);
	pCol->set_cell_data_func(*pCell, mem_fun(*this, &FilesTreeView::statusCellFunc));

	// Column "File"
	id = append_column("File", mColumns.mName) - 1;
	g_assert(id == COL_NAME);
	pCol = get_column(id);
	pCell = pCol->get_first_cell_renderer();
	pCol->set_sort_column(id);
	pCol->set_resizable();
	pCol->set_cell_data_func(*pCell, mem_fun(*this, &FilesTreeView::nameCellFunc));

	// Column "Size"
	id = append_column("Size", mColumns.mSize) - 1;
	g_assert(id == COL_SIZE);
	pCol = get_column(id);
	pCell = pCol->get_first_cell_renderer();
	pCol->set_sort_column(id);
	pCol->set_resizable();
	pCol->set_alignment(1.0);
	pCol->set_cell_data_func(*pCell, mem_fun(*this, &FilesTreeView::sizeCellFunc));
	pCell->property_xalign() = 1.0;

	// Build the popup menu
	mpActionGroup->add(mpActionStrip);
	mpActionGroup->add(Gtk::Action::create("StripAll", "_All"),
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), STRIP_ALL));
	mpActionGroup->add(Gtk::Action::create("StripDebug", "_Debug"),
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), STRIP_DEBUG));
	mpActionGroup->add(Gtk::Action::create("StripUnneeded", "_Unneeded"),
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), STRIP_UNNEEDED));
	mpActionGroup->add(mpActionCompress);
	mpActionGroup->add(Gtk::Action::create("CompressGzip", "_Gzip"),
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), GZIP));
	mpActionGroup->add(Gtk::Action::create("CompressBzip2", "_Bzip2"),
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), BZIP2));
	mpActionGroup->add(mpActionUncompress,
		bind<int>(mem_fun(*this, &FilesTreeView::onDo), UNCOMPRESS));
	mpActionGroup->add(mpActionRemove, mem_fun(*this, &FilesTreeView::onRemove));
	
	Glib::RefPtr<Gtk::Action>
		pActionSelectAll(Gtk::Action::create("SelectAll", "Select _all")),
		pActionUnselectAll(Gtk::Action::create("UnselectAll", "U_nselect all"));

	mpActionGroup->add(pActionSelectAll, mem_fun(*this, &TreeView::selectAll));
	mpActionGroup->add(pActionUnselectAll, mem_fun(*this, &TreeView::unselectAll));

	mpUIManager->insert_action_group(mpActionGroup);

	mpUIManager->add_ui_from_string(
		"<ui>"
		"	<popup name='PopupMenu'>"
		"		<menu action='Strip'>"
		"			<menuitem action='StripAll'/>"
		"			<menuitem action='StripDebug'/>"
		"			<menuitem action='StripUnneeded'/>"
		"		</menu>"
		"		<menu action='Compress'>"
		"			<menuitem action='CompressGzip'/>"
		"			<menuitem action='CompressBzip2'/>"
		"		</menu>"
		"		<menuitem action='Uncompress'/>"
		"		<menuitem action='Remove'/>"
		"		<separator/>"
		"		<menuitem action='SelectAll'/>"
		"		<menuitem action='UnselectAll'/>"
		"	</popup>"
		"</ui>");
	
	mpMenu = dynamic_cast<Gtk::Menu*>(mpUIManager->get_widget("/PopupMenu"));
	mpMenu->signal_event().connect(mem_fun(*this, &FilesTreeView::onPopupMenu));
}


// [virtual]
FilesTreeView::~FilesTreeView()
{ }


void FilesTreeView::writeLabel() const
{
	if ((*mppModel)->children().empty()) {
		setLabelText("");
		return;
	}

	vector<File*> selFiles;
	long size = 0, files = getSelected(selFiles, &size);

	if (!files) {
		g_assert(size == 0);
		if (mPkg.listFilesInst()) {
			size += mPkg.sizeInst();
			files += mPkg.filesInst();
		}
		if (mPkg.listFilesMiss()) {
			size += mPkg.sizeMiss();
			files += mPkg.filesMiss();
		}
	}

	std::ostringstream msg;
	msg << files << " file" << (files > 1 ? "s" : "")
		<< (selFiles.empty() ? "" : " selected") << " | " << Paco::toString(size);
	
	setLabelText(msg.str());
}


void FilesTreeView::resetModel()
{
	vector<File*> sel, visible;
	getSelected(sel);
	getVisible(visible);

	int sortColID;
	Gtk::SortType sortType;
	bool sorted = (*mppModel)->get_sort_column_id(sortColID, sortType);

	unset_model();

	mpModelAll->clear();
	mpModelInst->clear();
	mpModelMiss->clear();

	setModel();

	// Restore sorting
	if (sorted)
		(*mppModel)->set_sort_column_id(sortColID, sortType);

	// Restore scroll
	for (vector<File*>::iterator f = visible.begin(); f != visible.end(); ++f) {
		iterator it;
		if (getIter(*f, *mppModel, it)) {
			Gtk::TreePath path(it);
			scroll_to_row(path, 0.0);
			break;
		}
	}

	// Restore selection
	if (sel.empty())
		return;
	
	Gtk::TreeModel::Children rows = (*mppModel)->children();
	if (rows.empty())
		return;

	for (Gtk::TreeModel::Children::iterator r = rows.begin(); r != rows.end(); ++r) {
		File* file = (*r)[mColumns.mFile];
		for (Pkg::iterator f = sel.begin(); f != sel.end(); ++f) {
			if (file == *f)
				mpSelection->select(*r);
		}
	}
}


//---------//
// private //
//---------//


void FilesTreeView::setModel()
{
	for (Pkg::iterator f = mPkg.begin(); f != mPkg.end(); ++f) {
		(*mpModelAll->append())[mColumns.mFile] = *f;
		if ((*f)->missing())
			(*mpModelMiss->append())[mColumns.mFile] = *f;
		else
			(*mpModelInst->append())[mColumns.mFile] = *f;
	}

	if (mPkg.listFilesInst())
		mppModel = mPkg.listFilesMiss() ? &mpModelAll : &mpModelInst;
	else
		mppModel = mPkg.listFilesMiss() ? &mpModelMiss : &mpModelEmpty;
	
	set_model(*mppModel);
	writeLabel();
}


//
// Get the currently visible files
//
void FilesTreeView::getVisible(vector<File*>& files)
{
	g_assert(files.empty());

	Gtk::TreeModel::Path start, end;
	if (!get_visible_range(start, end))
		return;
	iterator endIt = (*mppModel)->get_iter(end);

	for (iterator it = (*mppModel)->get_iter(start); it != endIt; ++it)
		files.push_back((*it)[mColumns.mFile]);
}


//
// Get the currently selected files
//
long FilesTreeView::getSelected(vector<File*>& files, long* size /* = NULL */) const
{
	vector<Gtk::TreeModel::Path> rows = mpSelection->get_selected_rows();
	vector<Gtk::TreeModel::Path>::iterator r;

	for (r = rows.begin(); r != rows.end(); ++r) {
		File* file = (*((*mppModel)->get_iter(*r)))[mColumns.mFile];
		g_assert(file != NULL);
		files.push_back(file);
		if (size && file->size() > 0)
			*size += file->size();
	}

	return files.size();
}


void FilesTreeView::onDo(int action)
{
	Lock lock;
	g_assert(GConfig::logdirWritable());

	// get the selected files
	vector<File*> files;
	g_return_if_fail(getSelected(files) > 0L);

	// show the progress bar
	Gtk::ProgressBar& progressBar = mPkg.window()->filesTab()->progressBar();
	progressBar.show();
	progressBar.add_modal_grab();

	vector<string> argv(3);

	switch (action) {
		case STRIP_ALL:
			argv[0] = "strip";
			argv[1] = "--strip-all";
			setLabelText("Removing all symbols");
			break;
		case STRIP_DEBUG:
			argv[0] = "strip";
			argv[1] = "--strip-debug";
			setLabelText("Removing debug symbols");
			break;
		case STRIP_UNNEEDED:
			argv[0] = "strip";
			argv[1] = "--strip-unneeded";
			setLabelText("Removing unneeded symbols");
			break;
		case UNCOMPRESS:
			argv[1] = "--force";
			setLabelText("Uncompressing files");
			break;
		case GZIP:
			argv[0] = "gzip";
			argv[1] = "--force";
			setLabelText("Compressing files with gzip");
			break;
		case BZIP2:
			argv[0] = "bzip2";
			argv[1] = "--force";
			setLabelText("Compressing files with bzip2");
			break;
		default:
			g_assert_not_reached();
	}

	gfloat cnt = 0.0;
	int status;
	bool update = false;
	string err, cwd = Glib::get_current_dir();

	for (Pkg::iterator f = files.begin(); f != files.end(); ++f) {
		
		g_return_if_fail(mPkg.window());
		progressBar.set_fraction(++cnt / files.size());
		refreshMainLoop();

		// skip symlinks and missing files
		if (!isRegular((*f)->name()))
			continue;

		else if (action == UNCOMPRESS) {
			if (isBzip2((*f)->name()))
				argv[0] = "bunzip2";
			else if (isGzip((*f)->name()))
				argv[0] = "gunzip";
			else
				continue;
		}

		else if (isBzip2((*f)->name()) || isGzip((*f)->name()))
			continue;
		
		// do not gzip the gzip program itself
		else if (action == GZIP && Glib::str_has_suffix((*f)->name(), "/gzip"))
			continue;
		
		// do not bzip2 the bzip2 program itself
		else if (action == BZIP2 && Glib::str_has_suffix((*f)->name(), "/bzip2"))
			continue;

		argv[2] = (*f)->name();
		
		Glib::spawn_sync(cwd, argv, Glib::SPAWN_SEARCH_PATH, slot<void>(), NULL, &err, &status);

		if (!WIFEXITED(status) || WEXITSTATUS(status) != EXIT_SUCCESS) {
			if (mPkg.window()) {
				errorDialog(mPkg.window(), "Error while running the " + argv[0]
					+ " process:\n\n" + err);
			}
			break;
		}
		update = true;

		switch (action) {
			case UNCOMPRESS:
				if (isBzip2((*f)->name()))
					(*f)->name().erase((*f)->name().size() - 4);
				else if (isGzip((*f)->name()))
					(*f)->name().erase((*f)->name().size() - 3);
				break;
			case GZIP:
				(*f)->name() += ".gz";
				break;
			case BZIP2:
				(*f)->name() += ".bz2";
		}

		Pkg::getSize((*f)->size(), (*f)->name());

	}

	if (update) {
		(*mppModel)->foreach(mem_fun(*this, &FilesTreeView::rowChanged));
		writeLabel();
		mPkg.update(false);
	}

	progressBar.remove_modal_grab();
	progressBar.hide();
	writeLabel();
}


void FilesTreeView::setLabelText(ustring const& text) const
{
	if (mPkg.window())
		mPkg.window()->filesTab()->setLabelText(text);
}


void FilesTreeView::onRemove()
{
	g_assert(GConfig::logdirWritable());

	Lock lock;

	// get the selected files
	vector<File*> files;
	g_return_if_fail(getSelected(files) > 0L);

	// ask the user
	std::ostringstream msg;
	msg << "Remove " << files.size() << " file"
		<< (files.size() > 1 ? "s" : "") << " ?\n";
	if (!questionDialog(mPkg.window(), msg.str()))
		return;

	// discard missing files
	files.erase(remove_if(files.begin(), files.end(), isMissing), files.end());
	if (files.empty())
		return;

	// show the progress bar
	Gtk::ProgressBar& progressBar = mPkg.window()->filesTab()->progressBar();
	progressBar.show();
	progressBar.add_modal_grab();

	// start removing
	gfloat err = 0.0, cnt = 0.0;
	bool stopMsg = false;
	ustring errmsg("Error while removing files:\n\n");

	for (Pkg::iterator f = files.begin(); f != files.end(); ++f) {
		if (!unlink((*f)->name().c_str())) {
			(*f)->missing(true);
			iterator it;
			g_return_if_fail(mPkg.window());
			getIter(*f, mpModelInst, it);
			mpModelInst->erase(it);
			(*mpModelMiss->append())[mColumns.mFile] = *f;
			++cnt;
		}
		else if (++err < 16)
			errmsg += "' " + (*f)->name() + "': " + Glib::strerror(errno) + "\n";
		else if (!stopMsg) {
			errmsg += " ...";
			stopMsg = true;
		}

		g_return_if_fail(mPkg.window());
		progressBar.set_fraction((cnt + err) / files.size());
		refreshMainLoop();
	}

	if (err)
		errorDialog(mPkg.window(), errmsg);
	if (cnt)
		mPkg.update();
	
	progressBar.remove_modal_grab();
	progressBar.hide();
}


bool FilesTreeView::getIter(File const* file, Glib::RefPtr<Gtk::ListStore> model,
	iterator& it) const
{
	Gtk::TreeModel::Children child = model->children();

	for (it = child.begin(); it != child.end(); ++it) {
		if (file == (*it)[mColumns.mFile])
			return true;
	}

	return false;
}


bool FilesTreeView::onPopupMenu(GdkEvent*)
{
	g_assert(!(*mppModel)->children().empty());

	bool vis = countSelected();
	mpActionStrip		->set_visible(vis);
	mpActionCompress	->set_visible(vis);
	mpActionUncompress	->set_visible(vis);
	mpActionRemove		->set_visible(vis);

	bool sen = GConfig::logdirWritable();
	mpActionStrip		->set_sensitive(sen);
	mpActionCompress	->set_sensitive(sen);
	mpActionUncompress	->set_sensitive(sen);
	mpActionRemove		->set_sensitive(sen);
	
	return false;
}


// [virtual]
bool FilesTreeView::on_button_press_event(GdkEventButton* e)
{
	Gtk::TreeModel::Path path;
	Gtk::TreeViewColumn* col;
	int x, y;
	bool ret = true;

	if (!get_path_at_pos(e->x, e->y, path, col, x, y))
		unselectAll();

	// Single click on the right mouse button
	else if (e->type == GDK_BUTTON_PRESS && e->button == 3) {
		if (!mpSelection->is_selected(path))
			ret = Gtk::TreeView::on_button_press_event(e);
		mpMenu->popup(e->button, e->time);
	}
	else
		ret = Gtk::TreeView::on_button_press_event(e);
	
	writeLabel();
	return ret;
}


// [virtual]
bool FilesTreeView::on_key_press_event(GdkEventKey* e)
{
	bool done = false;

	switch (e->keyval) {
		case GDK_Delete:
		case GDK_BackSpace:
			if (countSelected()) {
				onRemove();
				done = true;
			}
			break;
		case GDK_Menu:
			if (!(*mppModel)->children().empty()) {
				mpMenu->popup(0, e->time);
				done = true;
			}
			break;
	}

	bool ret = done ? done : Gtk::TreeView::on_key_press_event(e);
	writeLabel();

	return ret;
}


bool FilesTreeView::rowChanged(Gtk::TreeModel::Path const& path, iterator const& iter)
{
	(*mppModel)->row_changed(path, iter);
	return false;
}


void FilesTreeView::statusCellFunc(Gtk::CellRenderer* pCell, iterator const& it)
{
	File* file = (*it)[mColumns.mFile];
	g_assert(file != NULL);

	Gtk::CellRendererPixbuf* pCellPixbuf = dynamic_cast<Gtk::CellRendererPixbuf*>(pCell);
	g_assert(pCellPixbuf != NULL);

	pCellPixbuf->property_pixbuf() = render_icon(file->missing() ?
		Gtk::Stock::NO : Gtk::Stock::YES, Gtk::ICON_SIZE_SMALL_TOOLBAR);
}


void FilesTreeView::nameCellFunc(Gtk::CellRenderer* pCell, iterator const& it)
{
	File* file = (*it)[mColumns.mFile];
	g_assert(file != NULL);

	Gtk::CellRendererText* pCellText = dynamic_cast<Gtk::CellRendererText*>(pCell);
	g_assert(pCellText != NULL);

	pCellText->property_text() = file->name();
}


void FilesTreeView::sizeCellFunc(Gtk::CellRenderer* pCell, iterator const& it)
{
	File* file = (*it)[mColumns.mFile];
	g_assert(file != NULL);

	Gtk::CellRendererText* pCellText = dynamic_cast<Gtk::CellRendererText*>(pCell);
	g_assert(pCellText != NULL);

	pCellText->property_text() = Paco::toString(file->size());
}


int FilesTreeView::statusSortFunc(iterator const& a, iterator const&)
{
	File* fa = (*a)[mColumns.mFile];
	g_assert(fa != NULL);
	return fa->missing() ? -1 : 1;
}


int FilesTreeView::sizeSortFunc(iterator const& a, iterator const& b)
{
	File* fa = (*a)[mColumns.mFile];
	File* fb = (*b)[mColumns.mFile];
	g_assert(fa != NULL && fb != NULL);
	return fa->size() < fb->size() ? -1 : 1;
}


int FilesTreeView::nameSortFunc(iterator const& a, iterator const& b)
{
	File* fa = (*a)[mColumns.mFile];
	File* fb = (*b)[mColumns.mFile];
	g_assert(fa != NULL && fb != NULL);
	return fa->name() < fb->name() ? -1 : 1;
}


//-----------------------------------//
// class FilesTreeView::ModelColumns //
//-----------------------------------//


FilesTreeView::ModelColumns::ModelColumns()
{
	add(mFile);
	add(mStatus);
	add(mName);
	add(mSize);
}


//-------------------//
// static free funcs //
//-------------------//


static bool isMissing(File* file)
{
	return !Glib::file_test(file->name(), Glib::FILE_TEST_EXISTS | Glib::FILE_TEST_IS_SYMLINK);
}


static bool isGzip(string const& path)
{
	return Glib::str_has_suffix(path, ".gz") || Glib::str_has_suffix(path, ".tgz");
}


static bool isBzip2(string const& path)
{
	return Glib::str_has_suffix(path, ".bz2");
}


static bool isRegular(string const& path)
{
	return Glib::file_test(path, Glib::FILE_TEST_IS_REGULAR) &&
		!Glib::file_test(path, Glib::FILE_TEST_IS_SYMLINK);
}

