
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Some of the code in this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: oxine.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 */

#include "config.h"

#include <assert.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "codeset.h"
#include "disc.h"
#include "environment.h"
#include "event.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playback_menu.h"
#include "playlist_m3u.h"
#include "playlist_menu.h"
#include "scheduler.h"
#include "settings_menu.h"
#include "vdr.h"
#include "weather.h"
#include "weather_menu.h"

// global variables
oxine_t *oxine;

#define OPTION_VERBOSE      (1000)

static const char *short_options = "?hvF"
#ifdef HAVE_DISC_POLLING
    "P"
#endif
#ifdef HAVE_LIRC
    "L"
#endif
#ifdef HAVE_JOYSTICK
    "J"
#endif
#ifdef HAVE_WEATHER
    "W"
#endif
#ifdef HAVE_XINERAMA
    "X"
#endif
    "V:A:l:";

static struct option long_options[] = {
    {"help", no_argument, 0, 'h'},
    {"version", no_argument, 0, 'v'},
    {"no-fullscreen", no_argument, 0, 'F'},
    {"video-driver", required_argument, 0, 'V'},
    {"audio-driver", required_argument, 0, 'A'},
    {"logfile", required_argument, 0, 'l'},
#ifdef HAVE_DISC_POLLING
    {"no-polling", no_argument, 0, 'P'},
#endif
#ifdef HAVE_LIRC
    {"no-lirc", no_argument, 0, 'L'},
#endif
#ifdef HAVE_JOYSTICK
    {"no-joystick", no_argument, 0, 'J'},
#endif
#ifdef HAVE_XINERAMA
    {"xineramascreen", required_argument, 0, 'X'},
#endif
#ifdef HAVE_WEATHER
    {"no-weather", no_argument, 0, 'W'},
#endif
    {"verbose", optional_argument, 0, OPTION_VERBOSE},
    {0, no_argument, 0, 0}
};


static void
show_usage (void)
{
    xine_t *xine = xine_new ();
    xine_init (xine);

    printf ("\n");
    printf (_("Usage: oxine [OPTIONS] [MRL]\n"));
    printf ("\n");
    printf (_("OPTIONS are:\n"));
    printf (_("  -v, --version                Display version.\n"));
    printf (_("      --verbose [=level]       "
              "Set verbosity level. Default is 1.\n"));
    printf (_("  -l, --logfile <filename>     Set the log filename.\n"));
    printf (_("  -V, --video-driver <drv>     Select video driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               auto ");

    const char *const *driver_ids;
    if ((driver_ids = xine_list_video_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            if (odk_is_supported_video_driver (driver_id))
                printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
    printf (_("  -A, --audio-driver <drv>     Select audio driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               ");
    if ((driver_ids = xine_list_audio_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
#ifdef HAVE_XINERAMA
    printf (_("  -X, --xineramascreen [=num]  "
              "Sets the screen on which the fullscreen window\n"
              "                               will be shown.\n"));
#endif
    printf (_("  -F, --no-fullscreen          "
              "Do no start in fullscreen mode.\n"));
#ifdef HAVE_DISC_POLLING
    printf (_("  -P, --no-polling             "
              "Do not use automatic polling for removable media.\n"));
#endif
#ifdef HAVE_LIRC
    printf (_("  -L, --no-lirc                "
              "Do not use remote control (LIRC).\n"));
#endif
#ifdef HAVE_JOYSTICK
    printf (_("  -J, --no-joystick            "
              "Do not use joystick support.\n"));
#endif
#ifdef HAVE_WEATHER
    printf (_("  -W, --no-weather             "
              "Do not automaticaly retrieve weather information.\n"));
#endif
    printf ("\n");
    printf (_("Examples for valid MRLs (media resource locator):\n"
              "  File:  'path/foo.vob'\n"
              "         '/path/foo.vob'\n"
              "         'file://path/foo.vob'\n"
              "         'fifo://[[mpeg1|mpeg2]:/]path/foo'\n"
              "         'stdin://[mpeg1|mpeg2]'\n"
              "  DVD:   'dvd://VTS_01_2.VOB'\n"
              "         'dvd://path/foo.iso'\n"
              "         'dvd://<track number>'\n"
              "  VCD:   'vcd://<track number>'\n"));
    printf ("\n");

    xine_exit (xine);
}


static inline void
oxine_config_register_all (void)
{
    /* Drivers */
    xine_config_register_string (oxine->xine, "video.driver", "auto",
                                 _("video driver to use"),
                                 _("video driver to use"), 10, NULL, NULL);

    xine_config_register_string (oxine->xine, "audio.driver", "auto",
                                 _("audio driver to use"),
                                 _("audio driver to use"), 10, NULL, NULL);

    /* GUI */
    xine_config_register_num (oxine->xine, "gui.x11.xinerama_screen", 0,
                              _("Xinerama screen for fullscreen window"),
                              _("Xinerama screen for fullscreen window"),
                              10, NULL, NULL);

    xine_config_register_num (oxine->xine, "gui.x11.window_width", OSD_WIDTH,
                              _("non-fullscreen window width"),
                              _("The width of the non-fullscreen window"), 10,
                              NULL, NULL);

    xine_config_register_num (oxine->xine, "gui.x11.window_height",
                              OSD_HEIGHT, _("non-fullscreen window height"),
                              _("The height of the non-fullscreen window"),
                              10, NULL, NULL);

    xine_config_register_num (oxine->xine, "gui.x11.monitor_pixel_aspect", 0,
                              _("aspect ratio of one monitor pixel"),
                              _("aspect ratio of one monitor pixel"),
                              10, NULL, NULL);

    /* Shutdown */
    xine_config_register_bool (oxine->xine, "shutdown.ask", true,
                               _("ask before leaving oxine"),
                               _("Are we to show the quit confirm dialog?"),
                               10, NULL, NULL);

    xine_config_register_num (oxine->xine, "shutdown.last", 0,
                              _("command that was selected the last time"),
                              _("command that was selected the last time"),
                              10, NULL, NULL);

    xine_config_register_string (oxine->xine, "shutdown.shutdown_command",
                                 "/sbin/poweroff",
                                 _("shell command (power off)"),
                                 _("shell command (power off)"),
                                 10, NULL, NULL);

    xine_config_register_string (oxine->xine, "shutdown.reboot_command",
                                 "/sbin/reboot",
                                 _("shell command (reboot)"),
                                 _("shell command (reboot)"), 10, NULL, NULL);

    xine_config_register_string (oxine->xine, "shutdown.standby_command",
                                 "",
                                 _("shell command (standby)"),
                                 _("shell command (standby)"),
                                 10, NULL, NULL);

    /* Weather */
    xine_config_register_string (oxine->xine, "weather.location", "EDDI",
                                 _("METAR ICAO location indicator"),
                                 _("The METAR ICAO Location Indicator "
                                   "defines for which location the weather "
                                   "is to be displayed."), 10, NULL, NULL);

    /* Playlist */
    xine_config_register_bool (oxine->xine, "playlist.auto_reload", true,
                               _("reload playlist on start"),
                               _("If this is enabled and if you don't "
                                 "specify any MRL on the command line, "
                                 "oxine will automatically load the "
                                 "previous playlist."), 10, NULL, NULL);

    xine_config_register_bool (oxine->xine, "playlist.auto_play", false,
                               _("play reloaded playlist on start"),
                               _("If this is enabled and "
                                 "gui.playlist_auto_reload is also enabled "
                                 "the reloaded playlist will automatically "
                                 "be played."), 10, NULL, NULL);

    /* Subtitles */
    xine_config_register_bool (oxine->xine, "subtitles.autoload", true,
                               _("subtitle autoloading"),
                               _("Automatically load subtitles "
                                 "if they exist."), 10, NULL, NULL);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "media.dvd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/dvd";
        xine_config_update_entry (oxine->xine, &centry);
    }

    xine_config_lookup_entry (oxine->xine, "media.audio_cd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &centry);
    }

    xine_config_lookup_entry (oxine->xine, "media.vcd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &centry);
    }
}


int
main (int argc, char **argv)
{
    xine_cfg_entry_t centry;

    int c;
    int option_index;

    char *video_driver_id = NULL;
    char *audio_driver_id = NULL;

    char *log_filename = NULL;
    int verbosity = XINE_VERBOSITY_NONE;
    int xinerama_screen = -1;

    bool fullscreen = true;
#ifdef HAVE_LIRC
    bool use_lirc = true;
#endif
#ifdef HAVE_JOYSTICK
    bool use_joystick = true;
#endif
#ifdef HAVE_DISC_POLLING
    bool use_polling = true;
#endif
#ifdef HAVE_WEATHER
    bool use_weather = true;
#endif

#ifdef DEBUG
    char *codeset = get_system_encoding ();
    char *current_skindir = resolve_softlink (OXINE_SKINDIR "/default");

    debug ("     PACKAGE_NAME: %s", PACKAGE_NAME);
    debug ("  PACKAGE_VERSION: %s", PACKAGE_VERSION);
    debug ("     XINE_VERSION: %s", xine_get_version_string ());
    debug ("             HOME: %s", get_dir_home ());
    debug ("             LANG: %s", getenv ("LANG"));
    debug ("          CODESET: %s", codeset);
    debug ("  OXINE_LOCALEDIR: %s", OXINE_LOCALEDIR);
    debug ("    OXINE_DATADIR: %s", OXINE_DATADIR);
    debug ("  OXINE_VISUALDIR: %s", OXINE_VISUALDIR);
    debug ("    OXINE_SKINDIR: %s", OXINE_SKINDIR);
    debug ("  CURRENT_SKINDIR: %s", current_skindir);

    if (codeset)
        ho_free (codeset);
    if (current_skindir)
        ho_free (current_skindir);
#endif

    /* Set up internationalization */
#ifdef HAVE_SETLOCALE
    if (!setlocale (LC_ALL, ""))
        error ("The current locale is not supported by the C library!");
#endif

#ifdef ENABLE_NLS
    bindtextdomain (PACKAGE, OXINE_LOCALEDIR);
    textdomain (PACKAGE);
#endif

    /* Parse command line arguments */
    while ((c = getopt_long (argc, argv, short_options,
                             long_options, &option_index)) != EOF) {
        switch (c) {
        case OPTION_VERBOSE:
            if (optarg != NULL)
                verbosity = strtol (optarg, &optarg, 10);
            else
                verbosity = 1;
            break;
#ifdef HAVE_XINERAMA
        case 'X':
            if (optarg != NULL)
                xinerama_screen = strtol (optarg, &optarg, 10);
            break;
#endif
        case 'A':
            audio_driver_id = ho_strdup (optarg);
            break;
        case 'V':
            video_driver_id = ho_strdup (optarg);
            break;
        case 'l':
            log_filename = ho_strdup (optarg);
            break;
        case 'F':
            fullscreen = false;
            break;
#ifdef HAVE_DISC_POLLING
        case 'P':
            use_polling = false;
            break;
#endif
#ifdef HAVE_LIRC
        case 'L':
            use_lirc = false;
            break;
#endif
#ifdef HAVE_JOYSTICK
        case 'J':
            use_joystick = false;
            break;
#endif
#ifdef HAVE_WEATHER
        case 'W':
            use_weather = false;
            break;
#endif
        case 'v':
            printf (_("This is %s - A free Video-Player v%s\n"),
                    PACKAGE_NAME, PACKAGE_VERSION);
            printf (_("(c) 2005 by the oxine project\n"));
            exit (1);
            break;
        case 'h':
        case '?':
            show_usage ();
            exit (0);
            break;
        default:
            error (_("Invalid argument %d!"));
            show_usage ();
            exit (1);
            break;
        }
    }

    logger_init (log_filename);
    ho_free (log_filename);

    oxine = (oxine_t *) ho_new (oxine_t);
    oxine->xine = xine_new ();
    oxine->otk_clear_job = 0;
    oxine->playback_menu_job = 0;
#ifdef HAVE_DISC_POLLING
    oxine->use_polling = use_polling;
    filelist_ref_set (&oxine->removable_discs,
                      filelist_new (NULL, NULL, NULL,
                                    ALLOW_FILES_MULTIMEDIA));
#endif
#ifdef HAVE_WEATHER
    oxine->weather = NULL;
    pthread_mutex_init (&oxine->weather_mutex, NULL);
#endif
    oxine->user_interface_is_visible = false;
    oxine->playback_controls_are_visible = false;
    oxine->stream_parameter_is_visible = false;
    oxine->rw_playlist = playlist_new (get_file_playlist_rw (),
                                       playlist_rw_change_cb);
    oxine->ro_playlist = playlist_new (NULL, NULL);
    oxine->current_playlist = oxine->ro_playlist;
    pthread_mutex_init (&oxine->download_mutex, NULL);
    oxine->current_dialog_data.background_mrl = NULL;
    oxine->current_dialog_data.msg = NULL;
    oxine->shutdown_command = NULL;

    /* Load and init the xine configuration */
    debug ("reading configuration from file %s", get_file_config ());
    xine_config_load (oxine->xine, get_file_config ());

    /* Init the xine engine */
    xine_engine_set_param (oxine->xine, XINE_ENGINE_PARAM_VERBOSITY,
                           verbosity);
    xine_init (oxine->xine);

    /* Register our xine config items */
    oxine_config_register_all ();
    environment_config_register_all ();
#ifdef HAVE_VDR
    vdr_config_register_all ();
#endif

    /* Init the filelist extensions */
    filelist_extensions_init ();

    /* Screen to use when in xinerama fullscreen mode. */
    if (xinerama_screen != -1) {
        xine_config_lookup_entry (oxine->xine, "gui.x11.xinerama_screen",
                                  &centry);
        centry.num_value = xinerama_screen;
        xine_config_update_entry (oxine->xine, &centry);
    }

    /* Video driver to use. */
    if (video_driver_id) {
        xine_config_lookup_entry (oxine->xine, "video.driver", &centry);
        centry.str_value = video_driver_id;
        xine_config_update_entry (oxine->xine, &centry);
        ho_free (video_driver_id);
    }

    /* Audio driver to use. */
    if (audio_driver_id) {
        xine_config_lookup_entry (oxine->xine, "audio.driver", &centry);
        centry.str_value = audio_driver_id;
        xine_config_update_entry (oxine->xine, &centry);
        ho_free (audio_driver_id);
    }

    /* Start our scheduler */
    start_scheduler ();

    /* Init ODK */
    oxine->odk = odk_init (oxine->xine);
    if (!oxine->odk) {
        fatal (_("Could not initialize ODK!"));
        abort ();
    }

    /* Register our event handler with ODK */
    odk_add_event_handler (oxine->odk, oxine_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_LOW);

    /* Init OTK */
    oxine->otk = otk_init (oxine->odk);
    if (!oxine->otk) {
        fatal (_("Could not inizialize OTK!"));
        abort ();
    }
#ifdef HAVE_LIRC
    if (use_lirc) {
        odk_listen_to (oxine->odk, "lirc");
    }
#endif
#ifdef HAVE_JOYSTICK
    if (use_joystick) {
        odk_listen_to (oxine->odk, "joystick");
    }
#endif

    /* Add anything that was not recognized to the playlist */
    int i = optind;
    filelist_t *arglist = NULL;
    filelist_ref_set (&arglist, filelist_new (NULL, NULL, "/",
                                              ALLOW_FILES_MULTIMEDIA));
    for (; i < argc; i++) {
        char *mrl = argv[i];
        char *title = create_title (mrl);
        filelist_add (arglist, title, mrl, FILE_TYPE_UNKNOWN);
        ho_free (title);
    }
    fileitem_t *fileitem = filelist_first (arglist);
    while (fileitem) {
        filelist_expand (fileitem);
        playlist_add_fileitem (oxine->ro_playlist, fileitem);
        fileitem = filelist_next (arglist, fileitem);
    }
    filelist_ref_set (&arglist, NULL);

    /* If the user wants this we reload the last playlist and start playing
     * the last song that was played before oxine exited. */
    xine_config_lookup_entry (oxine->xine, "playlist.auto_reload", &centry);
    int auto_reload = centry.num_value;

    xine_config_lookup_entry (oxine->xine, "playlist.auto_play", &centry);
    int auto_play = centry.num_value;

    /* Reload the music and the video playlists. */
    if (auto_reload) {
        playlist_load (oxine->rw_playlist);
    }

    /* Decide what to play (if anything). If there is anything in the general
     * playlist (the user added files via the command line) this is played. */
    playitem_t *item2play = NULL;
    if (playlist_length (oxine->ro_playlist) > 0) {
        oxine->current_playlist = oxine->ro_playlist;
        item2play = playlist_get_first (oxine->current_playlist);
    }

    /* If the general playlist is empty we have a look at the music and video
     * lists to see if there is something to be played in them. */
    else if (auto_play) {
        oxine->current_playlist = oxine->rw_playlist;
        item2play = playlist_get_current (oxine->current_playlist);
    }

    /* Load main menu. */
    main_menu_init ();

    /* Show main window. */
    odk_show_window (oxine->odk, fullscreen);

    oxine->backto_menu = hide_user_interface;
    oxine->playback_ended_menu = show_main_menu_cb;

    /* If we found an item to play we play it... */
    if (item2play) {
        playlist_play_item (oxine, show_playback_menu_cb,
                            oxine->current_playlist, item2play);
    } else {
        show_main_menu_cb (oxine);
    }

#ifdef HAVE_DISC_POLLING
    /* Start polling for discs. */
    if (oxine->use_polling)
        disc_start_polling (oxine);
#endif

#ifdef HAVE_WEATHER
    /* Start the weather thread. */
    if (use_weather)
        start_weather_thread (oxine);
#endif

    /* Start the event loop. */
    odk_run (oxine->odk);

    /* Stop the scheduler. */
    stop_scheduler ();

#ifdef HAVE_WEATHER
    /* Stop the weather thread. */
    stop_weather_thread ();
#endif

#ifdef HAVE_DISC_POLLING
    /* Stop polling for discs. */
    if (oxine->use_polling)
        disc_stop_polling (oxine);
#endif

    if (oxine->otk)
        otk_free (oxine->otk);
    if (oxine->odk)
        odk_free (oxine->odk);

#ifdef HAVE_DISC_POLLING
    /* Free the list containing the removable discs. */
    filelist_ref_set (&oxine->removable_discs, NULL);
#endif

#ifdef HAVE_WEATHER
    /* Free the weather data. */
    weather_free (oxine->weather);
    pthread_mutex_destroy (&oxine->weather_mutex);
#endif

    /* Free the menu items of the main menu. */
    if (oxine->main_menu_items)
        main_menu_free ();

    /* Free the playlists */
    if (oxine->rw_playlist)
        playlist_free (oxine->rw_playlist);
    if (oxine->ro_playlist)
        playlist_free (oxine->ro_playlist);

    /* Free the dialog data */
    ho_free (oxine->current_dialog_data.background_mrl);
    ho_free (oxine->current_dialog_data.msg);

    /* Destroy the download mutex */
    pthread_mutex_destroy (&oxine->download_mutex);

    /* Save xine config and free xine */
    xine_config_save (oxine->xine, get_file_config ());
    xine_exit (oxine->xine);

    /* We copy the shutdown because we gotta free oxine. */
    char shutdown_command[1024];
    if (oxine->shutdown_command)
        strncpy (shutdown_command, oxine->shutdown_command, 1024);
    else
        shutdown_command[0] = '\0';

    /* Free oxine. */
    ho_free (oxine->shutdown_command);
    ho_free (oxine);

    filelist_extensions_free ();

    filelist_menu_free ();
    settings_menu_free ();
    playlist_menu_free ();

    environment_free ();

    logger_free ();

    heapstat ();

    /* We run the shutdown command. */
    if (strlen (shutdown_command)) {
        debug ("executing shutdown command %s", shutdown_command);
        system (shutdown_command);
    }

    return 0;
}
