
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist_menu.c 1541 2006-11-13 16:25:16Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "filelist.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "mediamarks.h"
#include "mediamarks_favorites.h"
#include "oxine.h"
#include "playlist.h"
#include "playback_menu.h"
#include "playlist_menu.h"
#include "vdr.h"

extern oxine_t *oxine;

static otk_cb_t show_last_filelist_menu_cb = enter_media_menu_cb;

static l_list_t *prev_filelist = NULL;
static l_list_t *next_filelist = NULL;

static filelist_t *current_filelist = NULL;
static filelist_t *toplevel_filelist = NULL;
static filelist_t *extras_filelist = NULL;
static filelist_t *favorites_filelist = NULL;

static otk_widget_t *button_up = NULL;
static otk_widget_t *button_prev = NULL;
static otk_widget_t *button_next = NULL;
static otk_widget_t *button_home = NULL;

#define FILELIST_BUTTONS_NUM 7
static otk_widget_t *filelist_menu_buttons[FILELIST_BUTTONS_NUM];
static otk_widget_t *filelist_menu_list = NULL;
static otk_widget_t *filelist_menu_window = NULL;
static otk_widget_t *filelist_menu_title = NULL;

static char *current_thumbnail = NULL;
static odk_osd_image_t *filelist_menu_thumbnail = NULL;

/*
 * ***************************************************************************
 * Some prototypes of methods declared in this file
 * ***************************************************************************
 */
static void filelist_menu_update_list (void);
static void filelist_menu_update_buttons (void);
static void filelist_menu_update_thumbnail (const char *thumbnail_mrl);

static void filelist_menu_set_current_filelist (filelist_t * newlist);

static void filelist_focus_enter_cb (void *entry_cb_data);
static void filelist_focus_leave_cb (void *entry_cb_data);

static void filelist_select_cb (void *entry_cb_data);
static void filelist_activate_cb (void *entry_cb_data);
static void filelist_remove_cb (void *entry_cb_data);

static void filelist_playall_cb (void *entry_cb_data);
static void filelist_playall_recursive_cb (void *entry_cb_data);

static fileitem_t **filelist_get_selected_fileitems (int *num_selected);

/*
 * ***************************************************************************
 * Methods that are part of the filelist GUI.
 * ***************************************************************************
 */
static void
filelist_menu_update_buttons (void)
{
    bool enabled;
    bool has_prev = (l_list_length (prev_filelist) > 0);
    bool has_next = (l_list_length (next_filelist) > 0);
    bool is_toplevel = (current_filelist != toplevel_filelist);

    int num_selected;
    fileitem_t **items = filelist_get_selected_fileitems (&num_selected);

    otk_widget_set_enabled (button_up, is_toplevel);
    otk_widget_set_enabled (button_home, is_toplevel);
    otk_widget_set_enabled (button_prev, has_prev);
    otk_widget_set_enabled (button_next, has_next);

    enabled = (num_selected > 0);
    otk_widget_set_enabled (filelist_menu_buttons[0], enabled);
    otk_widget_set_enabled (filelist_menu_buttons[1], enabled);

    /* Remove/ Reset/ Eject */
    if (current_filelist == favorites_filelist) {
        otk_widget_set_enabled (filelist_menu_buttons[2], true);
        otk_button_set_text (filelist_menu_buttons[2], _("Reset"));
    } else if ((num_selected > 0)
               && (strcmp (current_filelist->mrl,
                           get_dir_oxine_playlists ()) == 0)) {
        otk_widget_set_enabled (filelist_menu_buttons[2], true);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }
#ifdef HAVE_VDR
    else if ((num_selected > 0)
             && (strcmp (current_filelist->mrl,
                         FILELIST_VDR_RECORDINGS_MRL) == 0)) {
        otk_widget_set_enabled (filelist_menu_buttons[2], true);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }
#endif
#ifdef HAVE_DISC_POLLING
    else if ((num_selected == 1)
             && (items[0]->parent == oxine->removable_discs)) {
        otk_widget_set_enabled (filelist_menu_buttons[2], true);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }
#endif
    else {
        otk_widget_set_enabled (filelist_menu_buttons[2], false);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }

    /* Current title */
    enabled = odk_current_is_playback_mode (oxine->odk);
    otk_widget_set_visible (filelist_menu_buttons[6], enabled);

    ho_free (items);
}


static void
filelist_menu_addto_list (fileitem_t * item)
{
    otk_widget_t *w;

    if (strncasecmp (item->mrl, "dvd://", 6) == 0) {
        char *device = NULL;
#ifdef HAVE_DISC_POLLING
        if (item->parent == oxine->removable_discs)
            device = ((device_entry_t *) item->user_data)->device;
#endif
        w = otk_listentry_new (filelist_menu_list, item->title,
                               play_dvd_cb, device,
                               filelist_select_cb, item, NULL, NULL);
    }

    else if (strncasecmp (item->mrl, "vcd://", 6) == 0) {
        char *device = NULL;
#ifdef HAVE_DISC_POLLING
        if (item->parent == oxine->removable_discs)
            device = ((device_entry_t *) item->user_data)->device;
#endif
        w = otk_listentry_new (filelist_menu_list, item->title,
                               play_vcd_cb, device,
                               filelist_select_cb, item, NULL, NULL);
    }

    else {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               filelist_activate_cb, item,
                               filelist_select_cb, item,
                               filelist_remove_cb, item);
    }

    otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                    filelist_focus_leave_cb, item);
}


static void
filelist_menu_update_list (void)
{
    if (!filelist_menu_list)
        return;

    /* If we're currently not in the toplevel filelist and the filelist does
     * NOT have a parent we're probably in a removable disc, that has been
     * removed (see disc.c). To rescue ourself of segmentation faults we jump
     * to the toplevel filelist and clear the prev and next lists. */
    if ((current_filelist != toplevel_filelist)
        && (current_filelist->parent == NULL)) {
        filelist_menu_set_current_filelist (toplevel_filelist);
        l_list_clear (prev_filelist, NULL);
        l_list_clear (next_filelist, NULL);
    }

    otk_clear_list (filelist_menu_list);

    mutex_lock (&current_filelist->mutex);
#ifdef HAVE_DISC_POLLING
    mutex_lock (&oxine->removable_discs->mutex);
#endif

    /* If we're in the music menu and the current filelist has regular
     * files in it, we add a play-all-entries entry to the top of the
     * list. */
    if ((current_filelist != toplevel_filelist)
        && (current_filelist != favorites_filelist)
#ifdef HAVE_VDR
        && (strcmp (current_filelist->mrl, FILELIST_VDR_RECORDINGS_MRL) != 0)
#endif
        && (strcmp (current_filelist->mrl, "/") != 0)
        && (strcmp (current_filelist->mrl, get_dir_home ()) != 0)
        && (strcmp (current_filelist->mrl, get_dir_oxine_playlists ()) != 0)) {
        bool has_regular_entries = false;
        bool has_sublist_entries = false;

        fileitem_t *item = filelist_first (current_filelist);
        while (item) {
            if (item->type == FILE_TYPE_REGULAR)
                has_regular_entries = true;
            if ((item->type != FILE_TYPE_REGULAR)
                && (item->type != FILE_TYPE_UNKNOWN))
                has_sublist_entries = true;
            item = filelist_next (current_filelist, item);
        }

        if (has_sublist_entries) {
            char *title = _("Play all titles in this folder "
                            "and all subfolders...");
            otk_listentry_new (filelist_menu_list, title,
                               filelist_playall_recursive_cb,
                               current_filelist, NULL, NULL, NULL, NULL);
        } else if (has_regular_entries) {
            char *title = _("Play all titles in this folder...");
            otk_listentry_new (filelist_menu_list, title,
                               filelist_playall_cb, current_filelist,
                               NULL, NULL, NULL, NULL);
        }
    }

    /* next we add all the entries from the extras filelist */
    if ((current_filelist == toplevel_filelist) && extras_filelist) {
        fileitem_t *item = filelist_first (extras_filelist);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (extras_filelist, item);
        }
    }

    /* next we add all the entries from the current filelist */
    {
        int i = 0;
        fileitem_t *item = filelist_first (current_filelist);
        while (item && ((current_filelist != favorites_filelist)
                        || (i < 20))) {
            filelist_menu_addto_list (item);
            item = filelist_next (current_filelist, item);
            i += 1;
        }
    }

#ifdef HAVE_DISC_POLLING
    /* last we add all removable drives currently available */
    if (current_filelist == toplevel_filelist) {
        fileitem_t *item = filelist_first (oxine->removable_discs);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (oxine->removable_discs, item);
        }
    }
#endif

#ifdef HAVE_DISC_POLLING
    mutex_unlock (&oxine->removable_discs->mutex);
#endif
    mutex_unlock (&current_filelist->mutex);

    otk_list_set_pos (filelist_menu_list, current_filelist->top_position);
    otk_list_set_focus (filelist_menu_list, current_filelist->cur_position);
}


static void
filelist_menu_update_thumbnail (const char *thumbnail_mrl)
{
    if (filelist_menu_thumbnail)
        odk_osd_hide_image (filelist_menu_thumbnail);
    filelist_menu_thumbnail = NULL;

    if (thumbnail_mrl) {
        char *mrl = NULL;

        if (is_downloadable (thumbnail_mrl))
            mrl = download_to_cache (thumbnail_mrl, NULL, 0);
        else
            mrl = ho_strdup (thumbnail_mrl);

        if (mrl && (access (mrl, R_OK) == 0)) {
            int x = 110;
            int y = 470;
            int w = 180;
            int h = 220;

            if (odk_current_is_playback_mode (oxine->odk)) {
                y += 20;
                h -= 40;
            }

            filelist_menu_thumbnail = odk_osd_show_image (oxine->odk, mrl,
                                                          x, y, w, h,
                                                          ODK_ALIGN_CENTER |
                                                          ODK_ALIGN_VCENTER,
                                                          true, 0x000000);
        }

        ho_free (mrl);
    }
}


static void
filelist_menu_set_current_filelist (filelist_t * newlist)
{
    filelist_t *prev = (filelist_t *) l_list_last (prev_filelist);
    filelist_t *next = (filelist_t *) l_list_last (next_filelist);

    /* We are returning to the directory on top of the prev stack. We 
     * remove it from the prev stack and add the current directory to the 
     * next stack. */
    if (newlist == prev) {
        l_list_remove (prev_filelist, newlist);
        l_list_append (next_filelist, current_filelist);
    }

    /* We are returning to the directory on top of the next stack. We 
     * remove it from the next stack and add the current directory to the 
     * prev stack. */
    else if (newlist == next) {
        l_list_remove (next_filelist, newlist);
        l_list_append (prev_filelist, current_filelist);
    }

    /* We are entering a directory that is neither on top of the prev stack
     * nor on top of the next stack. We clear the next stack and add the
     * current directory to the prev stack. */
    else {
        l_list_clear (next_filelist, NULL);
        l_list_append (prev_filelist, current_filelist);
    }

    current_filelist->top_position = otk_list_get_pos (filelist_menu_list);
    current_filelist->cur_position = otk_list_get_focus (filelist_menu_list);
    filelist_ref_set (&current_filelist, newlist);

    /* If we enter a directory that contains a thumbnail image we display
     * this. */
    if (current_thumbnail)
        ho_free (current_thumbnail);
    current_thumbnail = get_thumbnail (newlist->mrl);

    if (filelist_menu_title) {
        if (current_filelist == toplevel_filelist)
            otk_label_set_text (filelist_menu_title, _("Choose an entry..."));
        else {
            char *tmp = ho_strdup (current_filelist->title);
            char *title = tmp;

            if (title[0] == '[')
                title++;
            if (title[strlen (title) - 1] == ']')
                title[strlen (title) - 1] = '\0';
            otk_label_set_text (filelist_menu_title, title);

            ho_free (tmp);
        }
    }

    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
filelist_remove_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if (strcmp (current_filelist->mrl, get_dir_oxine_playlists ()) == 0) {
        if (unlink (fileitem->mrl) == 0) {
            filelist_remove (current_filelist, fileitem);
            filelist_menu_set_current_filelist (current_filelist);
        } else {
            error (_("Could not remove '%s': %s!"),
                   fileitem->mrl, strerror (errno));
        }
    }
#ifdef HAVE_VDR
    else if (strcmp (current_filelist->mrl, FILELIST_VDR_RECORDINGS_MRL) == 0) {
        if (vdr_remove_recording (fileitem->mrl)) {
            filelist_remove (current_filelist, fileitem);
            filelist_menu_set_current_filelist (current_filelist);
        } else {
            error (_("Could not remove '%s': %s!"),
                   fileitem->mrl, _("SVDRP error"));
        }
    }
#endif
#ifdef HAVE_DISC_POLLING
    else if (fileitem->parent == oxine->removable_discs) {
        device_entry_t *entry = (device_entry_t *) fileitem->user_data;
        disc_eject (entry->device);
    }
#endif
}


/*
 * This is the callback for a click on an item in the list.
 */
static void
filelist_select_cb (void *entry_cb_data)
{
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


/*
 * This is the callback for a doubleclick on an item in the list.
 */
static void
filelist_activate_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if ((fileitem->type == FILE_TYPE_MOUNTPOINT)
        && !disc_mount (fileitem->mrl)) {
        show_message_dialog (backto_menu_cb, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("Could not open disc!"));
        return;
    }

    filelist_expand (fileitem);

    if (fileitem->sublist) {
        fileitem_t *first = filelist_first (fileitem->sublist);
        if ((filelist_length (fileitem->sublist) == 1)
            && (first->type == FILE_TYPE_REGULAR)) {
            fileitem = first;
        } else {
            filelist_menu_set_current_filelist (fileitem->sublist);
        }
    }

    if (fileitem->type == FILE_TYPE_REGULAR) {
        playlist_clear (oxine->rw_playlist);
        playlist_add_fileitem (oxine->rw_playlist, fileitem);
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->rw_playlist);
    }
#ifdef HAVE_VDR
    if (fileitem->type == FILE_TYPE_VDR_RECORDING) {
        if (vdr_play_recording (fileitem->mrl)) {
            play_vdr_cb (oxine);
        }
    }
#endif

    filelist_menu_update_buttons ();
}


static void
filelist_focus_enter_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if (!fileitem)
        return;

    /* If the fileitem is a directory this would point to the standard thumbnail file. */
    char *thumbnail_mrl = get_thumbnail (fileitem->mrl);

    /* If the fileitem has its own thumbnail we show it. */
    if (fileitem->thumbnail_mrl) {
        filelist_menu_update_thumbnail (fileitem->thumbnail_mrl);
    }

    /* Else we show the thumbnail of the directory this fileitem points to. */
    else if (thumbnail_mrl) {
        filelist_menu_update_thumbnail (thumbnail_mrl);
    }

    /* Last we try to show the thumbnail of the current directory. */
    else {
        filelist_menu_update_thumbnail (current_thumbnail);
    }

    if (thumbnail_mrl)
        ho_free (thumbnail_mrl);
}


static void
filelist_focus_leave_cb (void *entry_cb_data)
{
    /* If available we show the thumbnail of the current directory. */
    filelist_menu_update_thumbnail (current_thumbnail);
}


static void
filelist_up_cb (void *oxine_p)
{
    if (current_filelist) {
#ifdef HAVE_DISC_POLLING
        if ((current_filelist->parent == oxine->removable_discs)
            || (current_filelist->parent == extras_filelist)) {
#else
        if (current_filelist->parent == extras_filelist) {
#endif
            filelist_menu_set_current_filelist (toplevel_filelist);
        } else {
            filelist_menu_set_current_filelist (current_filelist->parent);
        }
    }
}


static void
filelist_home_cb (void *oxine_p)
{
    if (current_filelist != toplevel_filelist) {
        filelist_menu_set_current_filelist (toplevel_filelist);
    }
}


static void
filelist_prev_cb (void *oxine_p)
{
    filelist_t *last = (filelist_t *) l_list_last (prev_filelist);
    if (last) {
        filelist_menu_set_current_filelist (last);
    }
}


static void
filelist_next_cb (void *oxine_p)
{
    filelist_t *last = (filelist_t *) l_list_last (next_filelist);
    if (last) {
        filelist_menu_set_current_filelist (last);
    }
}


static void
filelist_menu_show_list (void)
{
    int x = 780 - 4 * 46 + 6;
    int y = 100;

    filelist_menu_title =
        otk_label_new (oxine->otk, 220, 117, x - 240,
                       OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                       _("Choose an entry..."));
    otk_widget_set_font (filelist_menu_title, "sans", 32);

    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 140, 560, 440,
                      30, 30, true, true, OTK_LIST_MULTIPLE_SELECTION, oxine);

    button_prev =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_LEFT),
                               filelist_prev_cb, oxine);
    x += 46;
    button_next =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_RIGHT),
                               filelist_next_cb, oxine);
    x += 46;
    button_up =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_UP),
                               filelist_up_cb, oxine);
    x += 46;
    button_home =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_HOME),
                               filelist_home_cb, oxine);
}


static void
filelist_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (!filelist_menu_is_current_menu (oxine))
        return;

    if (!(event->type == OXINE_EVENT_KEY))
        return;

    switch (event->source.key) {
    case OXINE_KEY_HOME:
        filelist_home_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_BACK:
        if (current_filelist != toplevel_filelist)
            filelist_up_cb (oxine);
        else
            show_main_menu_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_PREV:
        filelist_prev_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_NEXT:
        filelist_next_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


static fileitem_t **
filelist_get_selected_fileitems (int *num_selected)
{
    return (fileitem_t **) otk_list_get_selected (filelist_menu_list,
                                                  num_selected);
}




/*
 * ***************************************************************************
 * Methods that are part the music and video menu
 * ***************************************************************************
 */
static int
playlist_add_selected_items (oxine_t * oxine)
{
    int num_selected;
    fileitem_t **fileitems = filelist_get_selected_fileitems (&num_selected);
    if (!fileitems)
        return 0;

    int i = 0;
    for (; i < num_selected; i++) {
        filelist_expand (fileitems[i]);
        playlist_add_fileitem (oxine->rw_playlist, fileitems[i]);
        if (favorites_filelist && (fileitems[i]->type == FILE_TYPE_DIRECTORY)) {
            favorites_add_directory (favorites_filelist,
                                     fileitems[i]->title, fileitems[i]->mrl);
        }
    }
    ho_free (fileitems);

    return num_selected;
}


static void
add_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (otk_list_get_selected_count (filelist_menu_list)) {
        playlist_add_selected_items (oxine);
        show_playlist_menu_cb (oxine);
    }
}


static void
play_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (otk_list_get_selected_count (filelist_menu_list)) {
        playlist_clear (oxine->rw_playlist);
        playlist_add_selected_items (oxine);
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->rw_playlist);
    }
}


static void
remove_selected_cb (void *oxine_p)
{
    if (current_filelist == favorites_filelist) {
        if (!unlink (get_file_favorites ())) {
            error (_("Could not remove '%s': %s!"),
                   get_file_favorites (), strerror (errno));
        }
        filelist_clear (favorites_filelist);
        filelist_menu_set_current_filelist (current_filelist);
        return;
    }

    if (!otk_list_get_selected_count (filelist_menu_list))
        return;

    int num_selected;
    fileitem_t **fileitems = filelist_get_selected_fileitems (&num_selected);
    if (!fileitems)
        return;

    int i = 0;
    for (; i < num_selected; i++) {
        filelist_remove_cb (fileitems[i]);
    }

    filelist_menu_set_current_filelist (current_filelist);
    ho_free (fileitems);
}


static void
filelist_playall_recursive_cb (void *entry_cb_data)
{
    filelist_t *filelist = (filelist_t *) entry_cb_data;

    playlist_clear (oxine->rw_playlist);

    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        filelist_expand (fileitem);
        playlist_add_fileitem (oxine->rw_playlist, fileitem);
        fileitem = filelist_next (filelist, fileitem);
    }

    if (favorites_filelist) {
        char *title = create_title (filelist->mrl);
        favorites_add_directory (favorites_filelist, title, filelist->mrl);
        ho_free (title);
    }

    playlist_play_first (oxine, show_playback_menu_cb, oxine->rw_playlist);
}


static void
filelist_playall_cb (void *entry_cb_data)
{
    filelist_t *filelist = (filelist_t *) entry_cb_data;

    playlist_clear (oxine->rw_playlist);
    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        if (fileitem->type == FILE_TYPE_REGULAR) {
            playlist_add_fileitem (oxine->rw_playlist, fileitem);
        }
        fileitem = filelist_next (filelist, fileitem);
    }

    if (favorites_filelist) {
        char *title = create_title (filelist->mrl);
        favorites_add_directory (favorites_filelist, title, filelist->mrl);
        ho_free (title);
    }

    playlist_play_first (oxine, show_playback_menu_cb, oxine->rw_playlist);
}


static void
filelist_menu_show_gui (char *background)
{
    show_menu_background (background);

    if (filelist_menu_window) {
        otk_window_set_current (oxine->otk, filelist_menu_window);

        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);

        filelist_menu_update_list ();
        filelist_menu_update_buttons ();

        return;
    }
    odk_add_event_handler (oxine->odk, filelist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    filelist_menu_window = create_new_window (true, true);

    int x = 20;
    int y = 100;
    filelist_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Play"),
                             play_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[1] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add"),
                             add_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[2] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Remove"),
                             remove_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[3] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Eject"),
                             eject_cb, oxine);
    y += 50;
    filelist_menu_buttons[4] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Playlist"),
                             show_playlist_menu_cb, oxine);
    y += 40;
    filelist_menu_buttons[5] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                             show_main_menu_cb, oxine);
    y += 40;
    filelist_menu_buttons[6] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Current title"),
                             show_playback_menu_cb, oxine);


    filelist_menu_show_list ();
    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
}


void
show_music_menu_cb (void *oxine_p)
{
    /* This is necessary as we have to make sure this method is not called
     * without first initializing the filelists. */
    if ((show_last_filelist_menu_cb != show_music_menu_cb)
        || !current_filelist) {
        show_last_filelist_menu_cb = show_music_menu_cb;
        enter_music_menu_cb (oxine);
        return;
    }

    hide_user_interface (oxine);

    filelist_menu_show_gui (OXINE_BACKGROUNDS "/musicmenu.png");

    show_last_filelist_menu_cb = show_music_menu_cb;
    oxine->backto_menu = show_music_menu_cb;
    oxine->current_menu = show_music_menu_cb;
    if (odk_current_is_logo_mode (oxine->odk))
        oxine->playback_ended_menu = show_music_menu_cb;

    show_user_interface (oxine);
}


void
show_video_menu_cb (void *oxine_p)
{
    /* This is necessary as we have to make sure this method is not called
     * without first initializing the filelists. */
    if ((show_last_filelist_menu_cb != show_video_menu_cb)
        || !current_filelist) {
        show_last_filelist_menu_cb = show_video_menu_cb;
        enter_video_menu_cb (oxine);
        return;
    }

    hide_user_interface (oxine);

    filelist_menu_show_gui (OXINE_BACKGROUNDS "/videomenu.png");

    show_last_filelist_menu_cb = show_video_menu_cb;
    oxine->backto_menu = show_video_menu_cb;
    oxine->current_menu = show_video_menu_cb;
    if (odk_current_is_logo_mode (oxine->odk))
        oxine->playback_ended_menu = show_video_menu_cb;

    show_user_interface (oxine);
}


void
show_media_menu_cb (void *oxine_p)
{
    /* This is necessary as we have to make sure this method is not called
     * without first initializing the filelists. */
    if ((show_last_filelist_menu_cb != show_media_menu_cb)
        || !current_filelist) {
        show_last_filelist_menu_cb = show_media_menu_cb;
        enter_media_menu_cb (oxine);
        return;
    }

    hide_user_interface (oxine);

    filelist_menu_show_gui (OXINE_BACKGROUNDS "/mediamenu.png");

    show_last_filelist_menu_cb = show_media_menu_cb;
    oxine->backto_menu = show_media_menu_cb;
    oxine->current_menu = show_media_menu_cb;
    if (odk_current_is_logo_mode (oxine->odk))
        oxine->playback_ended_menu = show_media_menu_cb;

    show_user_interface (oxine);
}


void
show_filelist_menu_cb (void *oxine_p)
{
    show_last_filelist_menu_cb (oxine);
}


static void
filelist_menu_enter (oxine_t * oxine, char *mediamarks, int allowed_filetypes)
{
    filelist_ref_set (&toplevel_filelist,
                      filelist_new (NULL, NULL, mediamarks,
                                    allowed_filetypes));
    filelist_ref_set (&current_filelist, toplevel_filelist);
    filelist_ref_set (&favorites_filelist, NULL);
    filelist_ref_set (&extras_filelist, NULL);

    if (prev_filelist)
        l_list_clear (prev_filelist, NULL);
    else
        prev_filelist = l_list_new ();
    if (next_filelist)
        l_list_clear (next_filelist, NULL);
    else
        next_filelist = l_list_new ();

    if (!mediamarks_read (toplevel_filelist)) {
        error (_("Could not load mediamarks."));
        info (_("You should create a file containing "
                "mediamarks at %s!"), mediamarks);
        {
            const char *mrl = get_dir_home ();
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        {
            const char *mrl = "/";
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        filelist_sort (toplevel_filelist, NULL);
    }

    if (current_thumbnail)
        ho_free (current_thumbnail);
    current_thumbnail = NULL;
}


void
enter_music_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    otk_window_keep (filelist_menu_window, 0);
    filelist_menu_window = NULL;
    odk_del_event_handler (oxine->odk, filelist_menu_event_handler);

    show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                         OXINE_BACKGROUNDS "/musicmenu.png",
                         _("Please wait..."));

    char *mediamarks = (char *) get_file_mediamarks_music ();
    filelist_menu_enter (oxine, mediamarks, ALLOW_FILES_MUSIC);

    filelist_ref_set (&extras_filelist,
                      filelist_new (NULL, NULL, NULL, ALLOW_FILES_MUSIC));
    {
        const char *mrl = get_dir_oxine_playlists ();
        char *title = create_title (mrl);
        filelist_add (extras_filelist, title, mrl, FILE_TYPE_DIRECTORY);
        ho_free (title);
    }
    {
        filelist_ref_set (&favorites_filelist,
                          favorites_new (extras_filelist));
    }

    show_music_menu_cb (oxine_p);
}


void
enter_video_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    otk_window_keep (filelist_menu_window, 0);
    filelist_menu_window = NULL;
    odk_del_event_handler (oxine->odk, filelist_menu_event_handler);

    show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                         OXINE_BACKGROUNDS "/videomenu.png",
                         _("Please wait..."));

    char *mediamarks = (char *) get_file_mediamarks_video ();
    filelist_menu_enter (oxine, mediamarks, ALLOW_FILES_VIDEO);

#ifdef HAVE_VDR
    filelist_ref_set (&extras_filelist,
                      filelist_new (NULL, NULL, NULL, ALLOW_FILES_VIDEO));
    {
        const char *mrl = FILELIST_VDR_RECORDINGS_MRL;
        char *title = create_title (mrl);
        filelist_add (extras_filelist, title, mrl, FILE_TYPE_MEDIAMARKS);
        ho_free (title);
    }
#endif

    show_video_menu_cb (oxine_p);
}


void
enter_media_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    otk_window_keep (filelist_menu_window, 0);
    filelist_menu_window = NULL;
    odk_del_event_handler (oxine->odk, filelist_menu_event_handler);

    show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                         OXINE_BACKGROUNDS "/mediamenu.png",
                         _("Please wait..."));

    char *mediamarks = (char *) get_file_mediamarks_media ();
    filelist_menu_enter (oxine, mediamarks, ALLOW_FILES_MULTIMEDIA);

    show_media_menu_cb (oxine_p);
}


int
filelist_menu_is_current_menu (oxine_t * oxine)
{
    int ret = 0;

    ret |= (oxine->current_menu == show_music_menu_cb);
    ret |= (oxine->current_menu == show_video_menu_cb);
    ret |= (oxine->current_menu == show_media_menu_cb);

    return ret;
}


void
filelist_menu_free (void)
{
    if (current_thumbnail)
        ho_free (current_thumbnail);

    if (next_filelist)
        l_list_free (next_filelist, NULL);
    next_filelist = NULL;
    if (prev_filelist)
        l_list_free (prev_filelist, NULL);
    prev_filelist = NULL;

    filelist_ref_set (&current_filelist, NULL);
    filelist_ref_set (&favorites_filelist, NULL);
    filelist_ref_set (&extras_filelist, NULL);
    filelist_ref_set (&toplevel_filelist, NULL);
}
