
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.c 1555 2006-11-15 09:04:35Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "filelist.h"
#include "disc.h"
#include "environment.h"
#include "i18n.h"
#include "heap.h"
#include "list.h"
#include "mutex.h"
#include "logger.h"
#include "playlist_m3u.h"
#include "mediamarks.h"
#include "vdr.h"

extern oxine_t *oxine;

static char *xine_extensions = NULL;

static l_list_t *allowed_xine_extensions;
static l_list_t *allowed_audio_extensions;
static l_list_t *allowed_video_extensions;
static l_list_t *allowed_subtitle_extensions;

static char *audio_extensions[] = {
    "16sv", "16sv", "669", "8svx", "8svx", "aac", "abs", "ac3", "aif", "aiff",
    "amf", "anx", "au", "aud", "axa", "axv", "cak", "cpk", "dat", "dps",
    "dts", "film", "flac", "iff", "ik2", "iki", "it", "m4a", "mdl", "med",
    "mka", "mod", "mp+", "mp2", "mp3", "mpa", "mpc", "mpega", "nsf", "ogg",
    "ra", "ram", "rpm", "s3m", "shn", "snd", "spx", "stm", "str", "svx",
    "voc", "vox", "wav", "wma", "wve", "xa", "xa1", "xa2", "xap", "xas", "xm",
    NULL
};

static char *video_extensions[] = {
    "4xm", "anim", "anim", "anim3", "anim5", "anim7", "anim8", "anx", "asf",
    "asp", "asx", "avi", "axa", "axv", "cak", "cin", "cpk", "cue", "dat",
    "dif", "divx", "dv", "film", "flc", "fli", "flv", "iff", "iki", "ik2",
    "dps", "m2p", "m2t", "m2v", "mjpg", "mkv", "mng", "mov", "mp4", "mpe",
    "mpeg", "mpg", "mpv", "mv8", "mve", "nsv", "ogm", "pes", "pva", "qt",
    "ram", "rm", "rmvb", "roq", "rpm", "rv", "spx", "str", "trp", "ts", "vmd",
    "vob", "vqa", "wax", "wma", "wmv", "wva", "wvx", "xa", "xa1", "xa2",
    "xap", "xas", "y4m",
    NULL
};

static char *image_extensions[] = {
    "gif", "ham", "ham6", "ham8", "iff", "ilbm", "jpeg", "jpg", "png",
    NULL
};

static char *subtitle_extensions[] = {
    "asc", "smi", "srt", "ssa", "sub", "txt",
    NULL
};

static char *title_prefixes[] = {
    "The ", "Der ", "Die ", "Das ", "Les ",
    NULL
};

static int
is_audio (const char *ext)
{
    int i = 0;
    for (; audio_extensions[i]; i++) {
        if (strcasecmp (audio_extensions[i], ext) == 0) {
            return 1;
        }
    }
    return 0;
}

static int
is_video (const char *ext)
{
    int i = 0;
    for (; video_extensions[i]; i++) {
        if (strcasecmp (video_extensions[i], ext) == 0) {
            return 1;
        }
    }
    return 0;
}

static int
is_image (const char *ext)
{
    int i = 0;
    for (; image_extensions[i]; i++) {
        if (strcasecmp (image_extensions[i], ext) == 0) {
            return 1;
        }
    }
    return 0;
}

static int
is_subtitle (const char *ext)
{
    int i = 0;
    for (; subtitle_extensions[i]; i++) {
        if (strcasecmp (subtitle_extensions[i], ext) == 0) {
            return 1;
        }
    }
    return 0;
}

/* 
 * ***************************************************************************
 * Name:            filelist_extensions_init
 * Access:          public
 *
 * Description:     Initializes the list of allowed fileextensions. This is
 *                  done by comparing a list of known audio, video and
 *                  subtitle extensions to the extensions xine-lib knows
 *                  about. Only entries that appear in both lists are added to
 *                  the allowed-lists.
 * ***************************************************************************
 */
void
filelist_extensions_init (void)
{
    allowed_xine_extensions = l_list_new ();
    allowed_audio_extensions = l_list_new ();
    allowed_video_extensions = l_list_new ();
    allowed_subtitle_extensions = l_list_new ();

    char *exts = xine_get_file_extensions (oxine->xine);
    xine_extensions = ho_strdup (exts);

    char *p = xine_extensions;
    while (p) {
        char *q = index (p, ' ');
        if (q) {
            q[0] = '\0';
            q++;
        }

        l_list_append (allowed_xine_extensions, p);

        int found = 0;
        if (is_audio (p)) {
            l_list_append (allowed_audio_extensions, p);
            found = 1;
        }
        if (is_video (p)) {
            l_list_append (allowed_video_extensions, p);
            found = 1;
        }
        if (is_image (p)) {
            // IGNORE for now
            found = 1;
        }
        if (is_subtitle (p)) {
            l_list_append (allowed_subtitle_extensions, p);
            found = 1;
        }
        if (!found) {
            debug ("Unknown file extension: '%s'.", p);
        }

        p = q;
    }

    free (exts);
}


/* 
 * ***************************************************************************
 * Name:            filelist_extensions_init
 * Access:          public
 *
 * Description:     Frees the allowed-extension-lists.
 * ***************************************************************************
 */
void
filelist_extensions_free (void)
{
    l_list_free (allowed_xine_extensions, NULL);
    l_list_free (allowed_audio_extensions, NULL);
    l_list_free (allowed_video_extensions, NULL);
    l_list_free (allowed_subtitle_extensions, NULL);
    ho_free (xine_extensions);
}


/* 
 * ***************************************************************************
 * Name:            is_file_allowed
 * Access:          public
 *
 * Description:     Determines if the file is to be added depending on the
 *                  currently allowed filetypes. The suffix of the file is
 *                  compared to all suffixes of the allowed type.
 * ***************************************************************************
 */
bool
is_file_allowed (const char *mrl, int allowed_filetypes)
{
    char *suffix = rindex (mrl, '.');

    if (!suffix)
        return false;

    suffix = (char *) (suffix + 1);

    switch (allowed_filetypes) {
    case ALLOW_FILES_ALL:
        {
            bool allow = false;
            char *base = get_basename (mrl);
            if (base[0] != '.')
                allow = true;
            ho_free (base);
            return allow;
        }
        break;
    case ALLOW_FILES_MULTIMEDIA:
        {
            char *ext = l_list_first (allowed_xine_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_xine_extensions, ext);
            }
        }
        break;
    case ALLOW_FILES_MUSIC:
        {
            char *ext = l_list_first (allowed_audio_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_audio_extensions, ext);
            }
        }
        break;
    case ALLOW_FILES_VIDEO:
        {
            int i = 0;
            for (; video_extensions[i]; i++) {
                if (strcasecmp (video_extensions[i], suffix) == 0) {
                    return true;
                }
            }
        }
        break;
    case ALLOW_FILES_SUBTITLE:
        {
            int i = 0;
            for (; subtitle_extensions[i]; i++) {
                if (strcasecmp (subtitle_extensions[i], suffix) == 0) {
                    return true;
                }
            }
        }
        break;
    }

    return false;
}


/* 
 * ***************************************************************************
 * Name:            has_suffix
 * Access:          public
 *
 * Description:     Determines if a file has the suffix passed.
 * ***************************************************************************
 */
static int
has_suffix (const char *mrl, const char *sfx)
{
    char *suffix = rindex (mrl, '.');

    if (!suffix)
        return 0;

    suffix = (char *) (suffix + 1);

    return (strcasecmp (suffix, sfx) == 0) ? 1 : 0;
}


/* 
 * ***************************************************************************
 * Name:            fileitem_free_cb
 * Access:          private
 *
 * Description:     This is the callback passed to l_list_free and
 *                  l_list_clear to free the memory of a fileitem. 
 * ***************************************************************************
 */
static void
fileitem_free_cb (void *data)
{
    fileitem_t *fileitem = (fileitem_t *) data;

    assert (fileitem);

    if (fileitem->sublist)
        fileitem->sublist->parent = NULL;
    filelist_ref_set (&fileitem->sublist, NULL);

    ho_free (fileitem->title);
    ho_free (fileitem->mrl);
    ho_free (fileitem->description);
    ho_free (fileitem->thumbnail_mrl);

    ho_free (fileitem);
}


/* 
 * ***************************************************************************
 * Name:            fileitem_swap_cb
 * Access:          private
 *
 * Description:     This is the callback passed to l_list_sort, to determin,
 *                  if two fileitems have to be reordered.
 * ***************************************************************************
 */
static int
fileitem_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    assert (f1);
    assert (f2);

    if ((f2->type == FILE_TYPE_AUTODISC)
        && (f1->type != FILE_TYPE_AUTODISC))
        return 1;
    if ((f1->type == FILE_TYPE_AUTODISC)
        && (f2->type != FILE_TYPE_AUTODISC))
        return 0;

    if (strcmp (f2->mrl, get_file_favorites ()) == 0)
        return 1;
    if (strcmp (f1->mrl, get_file_favorites ()) == 0)
        return 0;

    if (strcmp (f2->mrl, get_dir_oxine_playlists ()) == 0)
        return 1;
    if (strcmp (f1->mrl, get_dir_oxine_playlists ()) == 0)
        return 0;

#ifdef HAVE_VDR
    if (strcmp (f2->mrl, FILELIST_VDR_RECORDINGS_MRL) == 0)
        return 1;
    if (strcmp (f1->mrl, FILELIST_VDR_RECORDINGS_MRL) == 0)
        return 0;
#endif

    if (f2->type == FILE_TYPE_MEDIAMARKS)
        return 1;
    if (f1->type == FILE_TYPE_MEDIAMARKS)
        return 0;

    if ((f2->type == FILE_TYPE_M3U) && (f1->type != FILE_TYPE_M3U))
        return 1;
    if ((f1->type == FILE_TYPE_M3U) && (f2->type != FILE_TYPE_M3U))
        return 0;

    if (strcmp (f2->mrl, get_dir_home ()) == 0)
        return 1;
    if (strcmp (f1->mrl, get_dir_home ()) == 0)
        return 0;

    if (strcmp (f2->mrl, "/") == 0)
        return 1;
    if (strcmp (f1->mrl, "/") == 0)
        return 0;

    if ((f2->type == FILE_TYPE_DIRECTORY)
        && (f1->type != FILE_TYPE_DIRECTORY))
        return 1;
    if ((f1->type == FILE_TYPE_DIRECTORY)
        && (f2->type != FILE_TYPE_DIRECTORY))
        return 0;

    if ((f2->type == FILE_TYPE_MOUNTPOINT)
        && (f1->type != FILE_TYPE_MOUNTPOINT))
        return 0;
    if ((f1->type == FILE_TYPE_MOUNTPOINT)
        && (f2->type != FILE_TYPE_MOUNTPOINT))
        return 1;

    // sort the rest by name
    if (strcmp (f1->title, f2->title) > 0)
        return 1;

    return 0;
}


/* 
 * ***************************************************************************
 * Name:            filelist_new
 * Access:          public
 *
 * Description:     Creates a new filelist. Only files of the allowed type
 *                  will be added when expanding a directory.
 * ***************************************************************************
 */
filelist_t *
filelist_new (filelist_t * parent, const char *title, const char *mrl,
              fileitem_allowed_t allowed_filetypes)
{
    filelist_t *filelist = ho_new (filelist_t);

    filelist->list = l_list_new ();
    filelist->allowed_filetypes = allowed_filetypes;
    if (mrl)
        filelist->mrl = ho_strdup (mrl);
    else
        filelist->mrl = ho_strdup (FILELIST_TOPLEVEL_MRL);
    if (title)
        filelist->title = ho_strdup (title);
    else
        filelist->title = NULL;
    filelist->parent = parent;

    pthread_mutexattr_init (&filelist->mutex_attr);
    pthread_mutexattr_settype (&filelist->mutex_attr,
                               PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&filelist->mutex, &filelist->mutex_attr);

    filelist->top_position = 0;
    filelist->cur_position = 0;
    filelist->reference_count = 0;

    return filelist;
}


/* 
 * ***************************************************************************
 * Name:            filelist_clear
 * Access:          public
 *
 * Description:     Removes all fileitems from the list.
 * ***************************************************************************
 */
void
filelist_clear (filelist_t * filelist)
{
    assert (filelist);

    mutex_lock (&filelist->mutex);

    l_list_clear (filelist->list, fileitem_free_cb);
    filelist->top_position = 0;
    filelist->cur_position = 0;

    mutex_unlock (&filelist->mutex);
}


/* 
 * ***************************************************************************
 * Name:            filelist_free
 * Access:          private
 *
 * Description:     Removes all fileitems and frees the list.
 * ***************************************************************************
 */
static void
filelist_free (filelist_t * filelist)
{
    assert (filelist);

    l_list_free (filelist->list, fileitem_free_cb);
    pthread_mutex_destroy (&filelist->mutex);

    ho_free (filelist->title);
    ho_free (filelist->mrl);

    ho_free (filelist);
}


/* 
 * ***************************************************************************
 * Name:            filelist_ref_set
 * Access:          public
 *
 * Description:     This function is used to keep track of the number of
 *                  references that point at a filelist. The first parameter
 *                  is a pointer to a filelist pointer, the second the value
 *                  that is to be assigned to that pointer. As soon as the
 *                  reference count reaches 0 the filelist will be freed.
 * ***************************************************************************
 */
void
filelist_ref_set (filelist_t ** p, filelist_t * d)
{
    assert (p);

    if (*p) {
        if ((*p)->reference_count == 1) {
            filelist_free (*p);
        } else {
            (*p)->reference_count--;
        }
    }
    *p = d;
    if (*p) {
        (*p)->reference_count++;
    }
}


/* 
 * ***************************************************************************
 * Name:            filelist_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void
filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb)
{
    assert (filelist);

    mutex_lock (&filelist->mutex);

    if (swap_cb)
        l_list_sort (filelist->list, swap_cb);
    else
        l_list_sort (filelist->list, fileitem_swap_cb);

    mutex_unlock (&filelist->mutex);
}


/* 
 * ***************************************************************************
 * Name:            filelist_add
 * Access:          public
 *
 * Description:     Adds a new entry to the list.
 * ***************************************************************************
 */
fileitem_t *
filelist_add (filelist_t * filelist, const char *title, const char *mrl,
              fileitem_type_t type)
{
    assert (filelist);
    assert (title);
    assert (mrl);

    mutex_lock (&filelist->mutex);

    if (type == FILE_TYPE_UNKNOWN) {
        struct stat filestat;
        stat (mrl, &filestat);

        // it's a directory
        if (S_ISDIR (filestat.st_mode) && access (mrl, F_OK) == 0) {
            type = FILE_TYPE_DIRECTORY;
        }
        // it's a playlist
        else if (has_suffix (mrl, "m3u")) {
            type = FILE_TYPE_M3U;
        }
        // we assume its a regular file
        else {
            type = FILE_TYPE_REGULAR;
        }
    }

    fileitem_t *fileitem = ho_new (fileitem_t);

    fileitem->title = ho_strdup (title);
    fileitem->mrl = ho_strdup (mrl);
    fileitem->description = NULL;
    fileitem->thumbnail_mrl = NULL;
    fileitem->user_data = NULL;
    fileitem->type = type;
    filelist_ref_set (&fileitem->sublist, NULL);
    fileitem->parent = filelist;

    l_list_append (filelist->list, fileitem);

    mutex_unlock (&filelist->mutex);

    return fileitem;
}


/* 
 * ***************************************************************************
 * Name:            starts_with
 * Access:          private
 *
 * Description:     Returns true if str starts with start. The check is case
 *                  insensitive.
 * ***************************************************************************
 */
static int
starts_with (const char *str, const char *start)
{
    int i = 0;
    int ret = 1;
    for (; (i < strlen (start)) && ret; i++) {
        ret &= (toupper (str[i]) == toupper (start[i]));
    }
    return ret;
}


/* 
 * ***************************************************************************
 * Name:            rework_title
 * Access:          private
 *
 * Description:     If the title starts with one of the suffixes, this suffix
 *                  is cut from the beginning of the title and pastet to the
 *                  end.
 * ***************************************************************************
 */
static char *
rework_title (const char *title)
{
    int i = 0;

    for (; title_prefixes[i]; i++) {
        char *prefix = title_prefixes[i];
        if (starts_with (title, prefix)) {
            return ho_strdup_printf ("%s, %c%c%c", title + 4,
                                     title[0], title[1], title[2]);
        }
    }

    return ho_strdup (title);
}


/* 
 * ***************************************************************************
 * Name:            create_title
 * Access:          public
 *
 * Description:     Creates a title from the MRL. The returned string must be
 *                  freed when not used any more.
 * ***************************************************************************
 */
static char *
_create_title (const char *mrl, fileitem_type_t type)
{
    char *title = NULL;

    /* Get the basename of the filename. */
    char *filename_bn = get_basename (mrl);
    char *str_p = filename_bn;
    for (; str_p[0]; str_p++) {
        if (str_p[0] == '_') {
            str_p[0] = ' ';
        }
    }
    char *filename_bnw = rework_title (filename_bn);

    /* Remove the suffix of the filename. */
    char *filename_sf = ho_strdup (filename_bn);
    char *suffix = rindex (filename_sf, '.');
    if (suffix) {
        suffix[0] = '\0';
    }
    char *filename_sfw = rework_title (filename_sf);

    switch (type) {
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
        if (strcmp (mrl, get_dir_home ()) == 0)
            title = ho_strdup_printf ("[%s]", _("My Home"));
        else if (strcmp (mrl, "/") == 0)
            title = ho_strdup_printf ("[%s]", _("Filesystem"));
        else if (strcmp (mrl, get_dir_oxine_playlists ()) == 0)
            title = ho_strdup_printf ("[%s]", _("My Playlists"));
        else
            title = ho_strdup_printf ("[%s]", filename_bnw);
        break;
    case FILE_TYPE_M3U:
        title = ho_strdup_printf (_("[M3U-Playlist: %s]"), filename_sf);
        break;
    default:
        if (strncasecmp (mrl, "dvd:/", 5) == 0)
            title = ho_strdup_printf ("%s", _("DVD"));
        else if (strncasecmp (mrl, "vcd:/", 5) == 0)
            title = ho_strdup_printf ("%s", _("Video CD"));
        else if (strncasecmp (mrl, "cdda:/", 6) == 0)
            title = ho_strdup_printf ("%s", _("Audio CD"));
        else if (strcmp (mrl, get_file_mediamarks_music ()) == 0)
            title = ho_strdup_printf ("[%s]", _("My Music"));
        else if (strcmp (mrl, get_file_mediamarks_video ()) == 0)
            title = ho_strdup_printf ("[%s]", _("My Films"));
        else if (strcmp (mrl, get_file_favorites ()) == 0)
            title = ho_strdup_printf ("[%s]", _("My Favorites"));
#ifdef HAVE_VDR
        else if (strcmp (mrl, FILELIST_VDR_RECORDINGS_MRL) == 0)
            title = ho_strdup_printf ("[%s]", _("My VDR Recordings"));
#endif
        else
            title = ho_strdup_printf ("%s", filename_sfw);
        break;
    }

    ho_free (filename_bn);
    ho_free (filename_bnw);
    ho_free (filename_sf);
    ho_free (filename_sfw);

    return title;
}

char *
create_title (const char *mrl)
{
    char *title = NULL;

    if (mrl) {
        fileitem_type_t type = FILE_TYPE_UNKNOWN;

        if (access (mrl, R_OK) == 0) {
            struct stat filestat;
            stat (mrl, &filestat);
            if (S_ISDIR (filestat.st_mode)) {
                type = FILE_TYPE_DIRECTORY;
            } else if (S_ISREG (filestat.st_mode) && has_suffix (mrl, "m3u")) {
                type = FILE_TYPE_M3U;
            }
        }

        title = _create_title (mrl, type);
    }

    return title;
}


/* 
 * ***************************************************************************
 * Name:            directory_read
 * Access:          static
 *
 * Description:     Reads a directory and adds all entries to the list that
 *                  are of the currently allowed type.
 * ***************************************************************************
 */
static int
directory_read (filelist_t * filelist)
{
    DIR *dirp;
    struct dirent *entp;

    assert (filelist);
    assert (filelist->mrl);

    dirp = opendir (filelist->mrl);
    if (dirp == NULL) {
        error (_("Could not open '%s': %s!"), filelist->mrl,
               strerror (errno));
        return 0;
    }

    while ((entp = readdir (dirp))) {

        if (entp->d_name[0] == '.')
            continue;

        char sub_mrl[1024];
        snprintf (sub_mrl, 1024, "%s/%s", filelist->mrl, entp->d_name);

        struct stat filestat;
        stat (sub_mrl, &filestat);

        // it's a directory
        if (S_ISDIR (filestat.st_mode)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_DIRECTORY);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        // it's a playlist
        else if (S_ISREG (filestat.st_mode) && has_suffix (sub_mrl, "m3u")) {
            char *title = _create_title (sub_mrl, FILE_TYPE_M3U);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_M3U);
            ho_free (title);
        }
        // it's a regular file 
        else if (S_ISREG (filestat.st_mode)
                 && is_file_allowed (sub_mrl, filelist->allowed_filetypes)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_REGULAR);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_REGULAR);
            ho_free (title);
        }
    }
    closedir (dirp);

    filelist_sort (filelist, NULL);

    return 1;
}


/* 
 * ***************************************************************************
 * Name:            filelist_expand
 * Access:          public
 *
 * Description:     Depending on the type of the fileitem it is expanded.
 *                  Expanding means that any directory, playlist or
 *                  mediamarks-file is read and added to the sublist of the
 *                  fileitem.
 * ***************************************************************************
 */
void
filelist_expand (fileitem_t * fileitem)
{
    assert (fileitem);
    assert (fileitem->mrl);

    mutex_lock (&fileitem->parent->mutex);

    int top_position = 0;
    int cur_position = 0;

    if ((fileitem->type == FILE_TYPE_DIRECTORY)
        || (fileitem->type == FILE_TYPE_MOUNTPOINT)) {
        if (fileitem->sublist) {
            top_position = fileitem->sublist->top_position;
            cur_position = fileitem->sublist->cur_position;
            filelist_ref_set (&fileitem->sublist, NULL);
        }
    }

    if ((fileitem->type == FILE_TYPE_DIRECTORY)
        || (fileitem->type == FILE_TYPE_MOUNTPOINT)
        || (fileitem->type == FILE_TYPE_MEDIAMARKS)
        || (fileitem->type == FILE_TYPE_M3U)
        || (fileitem->type == FILE_TYPE_AUTODISC)) {

        if (!fileitem->sublist) {
            /* Create a new sublist. */
            filelist_t *sublist =
                filelist_new (fileitem->parent, fileitem->title,
                              fileitem->mrl,
                              fileitem->parent->allowed_filetypes);

            switch (fileitem->type) {
            case FILE_TYPE_MOUNTPOINT:
            case FILE_TYPE_DIRECTORY:
                directory_read (sublist);
                break;
            case FILE_TYPE_MEDIAMARKS:
#ifdef HAVE_VDR
                if (strcmp (fileitem->mrl, FILELIST_VDR_RECORDINGS_MRL) == 0)
                    vdr_recordings_read (sublist);
                else
#endif
                if (strcmp (fileitem->mrl, get_file_favorites ()) == 0)
                    favorites_read (sublist);
                else
                    mediamarks_read (sublist);
                break;
            case FILE_TYPE_M3U:
                playlist_m3u_read (sublist);
                break;
            case FILE_TYPE_AUTODISC:
                disc_autoscan_read (sublist);
            default:
                break;
            }

            /* Link the sublist to this fileitem. */
            filelist_ref_set (&fileitem->sublist, sublist);
        }
    }

    if ((fileitem->type == FILE_TYPE_DIRECTORY)
        || (fileitem->type == FILE_TYPE_MOUNTPOINT)) {
        if (fileitem->sublist) {
            fileitem->sublist->top_position = top_position;
            fileitem->sublist->cur_position = cur_position;
        }
    }

    mutex_unlock (&fileitem->parent->mutex);
}


/* 
 * ***************************************************************************
 * Name:            filelist_remove
 * Access:          public
 *
 * Description:     Removes the fileitem from the list.
 * ***************************************************************************
 */
void
filelist_remove (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);
    assert (filelist == fileitem->parent);

    mutex_lock (&filelist->mutex);

    l_list_remove (filelist->list, fileitem);
    fileitem_free_cb (fileitem);

    mutex_unlock (&filelist->mutex);
}


/* 
 * ***************************************************************************
 * Name:            filelist_first
 * Access:          public
 *
 * Description:     Returns the first fileitem.
 * ***************************************************************************
 */
fileitem_t *
filelist_first (filelist_t * filelist)
{
    assert (filelist);

    return (fileitem_t *) l_list_first (filelist->list);
}


/* 
 * ***************************************************************************
 * Name:            filelist_next
 * Access:          public
 *
 * Description:     Returns the next fileitem.
 * ***************************************************************************
 */
fileitem_t *
filelist_next (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);
    assert (filelist == fileitem->parent);

    return (fileitem_t *) l_list_next (filelist->list, fileitem);
}


/* 
 * ***************************************************************************
 * Name:            filelist_get_by_mrl
 * Access:          public
 *
 * Description:     Returns (if its there) the fileitem with the passed MRL.
 * ***************************************************************************
 */
fileitem_t *
filelist_get_by_mrl (filelist_t * filelist, const char *mrl)
{
    assert (filelist);

    mutex_lock (&filelist->mutex);

    fileitem_t *founditem = NULL;
    if (mrl) {
        fileitem_t *fileitem = filelist_first (filelist);
        while (fileitem) {
            if (strcasecmp (fileitem->mrl, mrl) == 0) {
                founditem = fileitem;
            }
            fileitem = filelist_next (filelist, fileitem);
        }
    }

    mutex_unlock (&filelist->mutex);

    return founditem;
}


/* 
 * ***************************************************************************
 * Name:            filelist_contains
 * Access:          public
 *
 * Description:     Determines, if the list contains an entry with the given
 *                  MRL.
 * ***************************************************************************
 */
bool
filelist_contains (filelist_t * filelist, const char *mrl)
{
    assert (filelist);

    mutex_lock (&filelist->mutex);

    bool contains = false;
    if (mrl) {
        fileitem_t *fileitem = filelist_first (filelist);
        while (fileitem) {
            if (strcasecmp (fileitem->mrl, mrl) == 0) {
                contains = true;
                break;
            }
            fileitem = filelist_next (filelist, fileitem);
        }
    }

    mutex_unlock (&filelist->mutex);

    return contains;
}


/* 
 * ***************************************************************************
 * Name:            filelist_length
 * Access:          public
 *
 * Description:     Returns the number of fileitems in this list.
 * ***************************************************************************
 */
int
filelist_length (filelist_t * filelist)
{
    assert (filelist);

    return l_list_length (filelist->list);
}
