
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc_iso9660.c 998 2006-05-05 09:41:45Z mschwerin $
 *
 * The two functions
 * - get_iso9660_volume_name 
 * - get_iso9660_volume_name_data_track_offset 
 * were taken from the source code (gnome-vfs-cdrom.c) of libgnomevfs.
 *
 */
#include "config.h"

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

#include "codeset.h"
#include "heap.h"
#include "i18n.h"
#include "disc_iso9660.h"

#define ISO_SECTOR_SIZE         2048
#define ISO_ROOT_START          (ISO_SECTOR_SIZE * (offset + 16))
#define ISO_VD_MAX              84

#define ISO_VD_BOOT             0
#define ISO_VD_PRIMARY          1
#define ISO_VD_SUPPLEMENTARY    2
#define ISO_VD_SET_TERMINATOR   255

#define ISODCL(from, to)        (to - from + 1)

typedef struct {
    char type[ISODCL (1, 1)];
    char id[ISODCL (2, 6)];
    char version[ISODCL (7, 7)];
    char unused1[ISODCL (8, 8)];
    char system_id[ISODCL (9, 40)];
    char volume_id[ISODCL (41, 72)];
    /* The following bytes are not really 
     * unused, but we don't need them here. */
    char unused2[ISODCL (73, 2048)];
} iso9660_primary_volume_descriptor_t;

#ifdef HAVE_LINUX_CDROM_H
static int
get_iso9660_volume_name_data_track_offset (int fd)
{
    struct cdrom_tocentry toc;
    char toc_header[2];
    int i;
    int offset;

    if (ioctl (fd, CDROMREADTOCHDR, &toc_header)) {
        return 0;
    }

    for (i = toc_header[0]; i <= toc_header[1]; i++) {
        memset (&toc, 0, sizeof (struct cdrom_tocentry));
        toc.cdte_track = i;
        toc.cdte_format = CDROM_MSF;
        if (ioctl (fd, CDROMREADTOCENTRY, &toc)) {
            return 0;
        }

        if (toc.cdte_ctrl & CDROM_DATA_TRACK) {
            offset = ((i == 1) ? 0 :
                      (int) toc.cdte_addr.msf.frame +
                      (int) toc.cdte_addr.msf.second * 75 +
                      (int) toc.cdte_addr.msf.minute * 75 * 60 - 150);
            return offset;
        }
    }

    return 0;
}
#endif // ifdef HAVE_LINUX_CDROM_H

static char *
get_iso9660_volume_name (int fd)
{
    iso9660_primary_volume_descriptor_t iso_buffer;
    int offset;
    int i;
    int vd_alt_offset;
    char *joliet_label;

    memset (&iso_buffer, 0, sizeof (iso9660_primary_volume_descriptor_t));

#ifdef HAVE_LINUX_CDROM_H
    offset = get_iso9660_volume_name_data_track_offset (fd);
#else
    offset = 0;
#endif

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("UTF-16BE", codeset);
    ho_free (codeset);

    joliet_label = NULL;
    for (i = 0, vd_alt_offset = ISO_ROOT_START + ISO_SECTOR_SIZE;
         i < ISO_VD_MAX; i++, vd_alt_offset += ISO_SECTOR_SIZE) {

        lseek (fd, (off_t) vd_alt_offset, SEEK_SET);
        read (fd, &iso_buffer, ISO_SECTOR_SIZE);

        if ((unsigned char) iso_buffer.type[0] == ISO_VD_SET_TERMINATOR)
            break;
        else if ((unsigned char) iso_buffer.type[0] == ISO_VD_SUPPLEMENTARY)
            continue;

        joliet_label = recode (xr, iso_buffer.volume_id);

        if (!joliet_label)
            continue;
        break;
    }

    recode_done (xr);

    lseek (fd, (off_t) ISO_ROOT_START, SEEK_SET);
    read (fd, &iso_buffer, ISO_SECTOR_SIZE);

    if (iso_buffer.volume_id[0] == 0 && !joliet_label) {
        return NULL;
    }

    if (joliet_label) {
        if (strncmp (joliet_label, iso_buffer.volume_id, 16) != 0)
            return joliet_label;
        ho_free (joliet_label);
    }

    return ho_strdup (iso_buffer.volume_id);
}

char *
disc_get_title_iso9660 (const char *device)
{
    char *title = NULL;

    int fd = open (device, O_RDONLY);
    if (fd != -1) {
        char *volume_name = get_iso9660_volume_name (fd);

        if (volume_name) {
            /* Strip whitespace off the end of the string. */
            int pos = 31;
            while (volume_name[pos] == ' ')
                pos--;
            volume_name[pos + 1] = '\0';

            title = ho_strdup (volume_name);
            ho_free (volume_name);
        }

        close (fd);
    }

    if (!title) {
        title = ho_strdup (_("ISO 9660 Volume"));
    }

    return title;
}
