#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ows_applet.py
#
# Copyright (C) 2007,2010 David Villa Alises
#
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


# Libwnck Reference Manual
#     http://library.gnome.org/devel/libwnck/
#
# Panel Applet Writer's Reference Manual
#     http://library.gnome.org/devel/panel-applet/stable/
#
# GConf Reference Manual
#     http://developer.gnome.org/doc/API/2.0/gconf/index.html

import os, sys
import types
import string
import gc
import time
import commands

import gobject
import gtk
import gnomeapplet
import wnck
import gconf
import dbus



import logging as log
log.basicConfig(level=log.DEBUG,
                format='%(asctime)s %(levelname)s %(message)s',
                filename=os.path.join(os.environ['HOME'], '.ows.log'),
                filemode='w')


import traceback
def exception():
    try:
        _type, value, tb = sys.exc_info()
        info = traceback.extract_tb(tb)
        filename, lineno, function, text = info[-1] # last line only
        log.error("%s:%d: %s: %s (in %s)" %\
                  (filename, lineno, _type.__name__, str(value), function))
    finally:
        _type = value = tb = None # clean up


from ows import ows_globals

def strcolor(gdkcolor):
    return "#%02X%02X%02X" % (gdkcolor.red   /256,
                              gdkcolor.green /256,
                              gdkcolor.blue  /256)

def widget_set_visible(wg, val):
    {False: gtk.Widget.hide, True: gtk.Widget.show_all}[val](wg)


def gtk_iteration():
    while gtk.events_pending(): gtk.main_iteration()

debug = False

IMG_DIR = 'images/' if os.path.isdir('images/') else '/usr/share/pixmaps/'

ICONS = [
    (IMG_DIR + 'tile-vertically.svg', 'Tile Vertically'),
    (IMG_DIR + 'tile-horizontally.svg', 'Tile Horizontally'),
    (IMG_DIR + 'tile-mosaic.svg', 'Tile Mosaic')]

factory = gtk.IconFactory()
for filename, stock_name in ICONS:
   pixbuf = gtk.gdk.pixbuf_new_from_file(filename)
   iconset = gtk.IconSet(pixbuf)
   factory.add(stock_name, iconset)
factory.add_default()





class OwsOSD:

    def __init__(self):
        self.values = {}
        self.xml = string.Template('''
        <message id="ows"
           animations="$animation"
           avoid_panels="$avoidpanels"
           drop_shadow="$shadow"
           enable_sound="$sound"
           hide_timeout="$timeout"
           osd_fake_translucent_bg="$translucent"
           osd_font="$fontname"
           osd_halignment="$halign"
           osd_vposition="$valign"
           debug_frame="False"><span foreground="$fontcolor">$msg</span></message>''')

        self.dbus_id = "pt.inescporto.telecom.GnomeOSD"
        self.osd = dbus.SessionBus().get_object(self.dbus_id, "/Server")

    def display(self, msg):
        self.values['msg'] = msg
        text = self.xml.substitute(self.values)
        print text
        self.osd.showMessageFull(text, dbus_interface=self.dbus_id)

    def __setitem__(self, name, value):
        if name == 'halign':
            value = ['left', 'center', 'right'][value]
        if name == 'valign':
            value = ['top', 'center', 'bottom'][value]
        if name == 'timeout':
            value = int(value*1000)

        self.values[name] = value


class PrefItem:
    def __init__(self, name, default, wg_get_cb, wg_set_cb):
        self.name = name
        self.value = default
        self.wg_get_cb = wg_get_cb
        self.wg_set_cb = wg_set_cb

    def render(self, osd):
        osd[self.name] = self.value

    def from_gui(self):
        self.value = self.wg_get_cb()

    def to_gui(self):
        self.wg_set_cb(self.value)

    def __str__(self):
        return "<Pref %s = %s>" % (self.name, self.value)

class ValuePrefItem(PrefItem):
    def __init__(self, name, default, widget):
        PrefItem.__init__(self, name, default,
                          widget.get_value, widget.set_value)

class IntPrefItem(PrefItem):
    def __init__(self, name, default, widget):
        PrefItem.__init__(self, name, default,
                          widget.get_value_as_int, widget.set_value)

class ActivePrefItem(PrefItem):
    def __init__(self, name, default, widget):
        PrefItem.__init__(self, name, default,
                          widget.get_active, widget.set_active)

TOP, MID, BOT = range(3)
LEFT, CENTER, RIGHT = range(3)


class OwsPrefs:

    MAX_WS = 36

    def __init__(self, applet, osd):
        self.applet = applet
        self.setup_gui()

        def single_color_set(color):
            self.wg_radio_single_color.set_active(color)
            self.on_radio_single_color_toggled()

        def show_all_ws(value):
            self.wg_radio_all_ws.set_active(value)
            self.on_radio_all_ws_toggled()

        self.items = [
            ValuePrefItem('timeout', 1, self.wg_spin_timeout),
            ActivePrefItem('halign', RIGHT, self.wg_combo_halign),
            ActivePrefItem('valign', TOP,   self.wg_combo_valign),
            ActivePrefItem('animation', False, self.wg_check_animation),
            ActivePrefItem('sound', False, self.wg_check_sound),
            ActivePrefItem('translucent', False, self.wg_check_translucent),
            ActivePrefItem('avoidpanels', False, self.wg_check_avoidpanels),
            ActivePrefItem('shadow', False, self.wg_check_shadow),
            IntPrefItem('shadow_offset', 2, self.wg_spin_shadow_offset),
            IntPrefItem('hoffset', 0, self.wg_spin_hoffset),
            IntPrefItem('voffset', 0, self.wg_spin_voffset),

            PrefItem('fontname', 'Arial Bold 48',
                     self.wg_fontbutton.get_font_name,
                     self.wg_fontbutton.set_font_name),

            PrefItem('fontcolor', 'green',
                     lambda:strcolor(self.wg_fontcolor.get_color()),
                     lambda x:self.wg_fontcolor.set_color(gtk.gdk.color_parse(x))),

            PrefItem('shadow_color', '#888',
                     lambda:strcolor(self.wg_shadowcolor.get_color()),
                     lambda x:self.wg_shadowcolor.set_color(gtk.gdk.color_parse(x))),
            ]

        self.single_color = PrefItem('single_color', True,
                                     self.wg_radio_single_color.get_active,
                                     single_color_set)

        # show_ws_name: 0:hidden, 1:right, 2:left
        self.show_name = ActivePrefItem('show_name', 0,
                                        self.wg_combo_show_ws_name)

        self.show_all_ws = PrefItem('show_all_ws', True,
                                    self.wg_radio_all_ws.get_active,
                                    show_all_ws)

        self.num_rows = IntPrefItem('num_rows', 1, self.wg_spin_num_rows)

        self.num_ws = IntPrefItem('num_ws',
                                  len(the_screen),
#                                  self.applet.get_ws_count(),
                                  self.wg_spin_num_workspaces)

        self.items.append(self.single_color)
        self.items.append(self.show_name)
        self.items.append(self.show_all_ws)
        self.items.append(self.num_rows)
        self.items.append(self.num_ws)

        self.names = []
        for i in range(self.num_ws.value):
            try:
                name = the_screen.get_workspace_name(i)
            except AttributeError:
                name = 'workspace %s' % (i+1)
            self.names.append(name)

        self.colors = [gtk.gdk.color_parse("white")] * self.MAX_WS

        self.window_view = None

        self.gui.connect_signals(self)
        self.gconf_init()


    def setup_gui(self):
        gui_fname = 'ows/ows.gui'
        if not os.path.exists(gui_fname):
            log.info('loading gui from package')
            gui_fname = '/usr/share/ows/ows.gui'

        self.gui = gtk.Builder()
        self.gui.add_from_file(gui_fname)

        self.window = self.gui.get_object('dialog')
        self.window.connect('delete-event',
                            self.on_button_cancel_clicked)

        self.wg_spin_num_workspaces.set_range(1, self.MAX_WS)



    def __getattr__(self, name):
        "get widget if attribute starts with 'wg_'"
        if not name.startswith('wg_'):
            raise AttributeError("'%s' isn't a object attribute" % name)

        widget = self.gui.get_object(name[3:])
        if widget is None:
            raise AttributeError("Widget '%s' not found" % name)

        self.__dict__[name] = widget
        return widget


    def rebuild_workspace_list(self):
        # free old widgets
        for h in self.wg_vbox_ws:
            for c in h:
                h.remove(c)
                del c
            self.wg_vbox_ws.remove(h)
            del h
        gc.collect()

        #print self.colors
        for i in range(self.num_ws.value):
            hbox = gtk.HBox()
            label = gtk.Label(" %2s: " % (i+1))
            entry = gtk.Entry()
            entry.set_text(self.names[i])
            color = gtk.ColorButton(self.colors[i])
            color.set_relief(gtk.RELIEF_NONE)
            color.set_sensitive(not self.single_color.value)
            hbox.pack_start(label, expand=False, fill=False)
            hbox.pack_start(entry)
            hbox.pack_start(color, expand=False, fill=False)
            self.wg_vbox_ws.pack_start(hbox, expand=False, fill=False)

        self.wg_vbox_ws.show_all()


    def show(self, *args):
        if self.gui is None:
            self.setup_gui()

        self.num_ws.value = len(the_screen)

        self.rebuild_workspace_list()

        for i in self.items:
            i.to_gui()

        self.window.show_all()


    def gui_to_model(self):
        # read names and colors
        i = 0
        for h in self.wg_vbox_ws:
            for c in h:
                if isinstance(c, gtk.ColorButton):
                    self.colors[i] = c.get_color()
                if isinstance(c, gtk.Entry):
                    self.names[i] = c.get_text()
            i += 1

        for i in self.items:
            i.from_gui()


    def on_spin_num_workspaces_value_changed(self, wg):
        self.num_ws.value = wg.get_value_as_int()
        self.rebuild_workspace_list()

    def on_radio_all_ws_toggled(self, *args):
        self.wg_spin_num_rows.set_sensitive(self.wg_radio_all_ws.get_active())

    def on_radio_single_color_toggled(self, *args):
        self.wg_fontcolor.set_sensitive(self.wg_radio_single_color.get_active())
        self.single_color.value = self.wg_radio_single_color.get_active()
        self.rebuild_workspace_list()

    def on_check_shadow_toggled(self, wg):
        self.wg_hbox_shadow.set_sensitive(self.wg_check_shadow.get_active())

    def on_button_apply_clicked(self, button):
        self.gui_to_model()
        self.applet.reconfigure()

    def on_button_cancel_clicked(self, button, *args):
        self.gconf_to_model()
        self.window.hide()
        return True

    def on_button_ok_clicked(self, button):
        self.gui_to_model()
        self.model_to_gconf()
        self.window.hide()
        self.applet.reconfigure()

    def on_notebook_switch_page(self, book, page, n):
        if n == 3 and self.window_view is None:
            self.window_view = WindowView(self.gui)


    def model_to_gconf(self):
        'Save preferences'
        casts = {types.BooleanType: gconf.Client.set_bool,
                 types.IntType:     gconf.Client.set_int,
                 types.FloatType:   gconf.Client.set_float,
                 types.StringType:  gconf.Client.set_string}

        for i in self.items:
            log.debug(i)
            casts[type(i.value)](self.client,
                                 self.gconf_key + i.name, i.value)

        self.client.set_list(self.gconf_key + 'colors', gconf.VALUE_STRING,
                             [strcolor(x) for x in self.colors])


    ##- GConf stuff
    def gconf_init(self):
        self.client = gconf.client_get_default()

        self.applet.add_preferences('/schemas/app/ows')
        self.gconf_key = self.applet.get_preferences_key()
        log.debug("preferences_key: %s" % self.gconf_key)
        #if self.gconf_key is None:
        #     self.gconf_key = "/apps/panel/applets/applet_23/prefs"
        self.gconf_key = '/apps/panel/applets/ows_screen0/prefs/'


        self.client.add_dir(self.gconf_key[:1], gconf.CLIENT_PRELOAD_NONE)

        self.gconf_to_model()


    def gconf_to_model(self):
        'Load preferences'
        casts = {gconf.VALUE_BOOL:   gconf.Value.get_bool,
                 gconf.VALUE_INT:    gconf.Value.get_int,
                 gconf.VALUE_FLOAT:  gconf.Value.get_float,
                 gconf.VALUE_STRING: gconf.Value.get_string}

        for i in self.items:
            gval = self.client.get(self.gconf_key + i.name)
            if gval is None: continue
            i.value = casts[gval.type](gval)
            log.debug(i)

        aux = [gtk.gdk.color_parse(x) for x in
               self.client.get_list(self.gconf_key + 'colors',
                                    gconf.VALUE_STRING)]
        if aux: self.colors = aux


class Pager(wnck.Pager):
    CONTENT, NAMES = range(2)

    def __init__(self, applet, screen):
        wnck.Pager.__init__(self, screen.inner)
        self.applet = applet
        self.screen = screen
        self.__n_rows = 1
        self.__show_all = True
        #self.set_shadow_type(gtk.SHADOW_IN)
        self.mode = None

    def set_n_rows(self, n):
        self.__n_rows = n
        return wnck.Pager.set_n_rows(self, n)

    def set_show_all(self, val):
        self.__show_all = val
        wnck.Pager.set_show_all(self, val)

    def adjust_size(self, pager_h=None):
        if pager_h is None:
            pager_h = self.applet.get_size()

        ratio = the_screen.aspect_ratio

        if not self.__show_all:
            self.set_size_request(int(pager_h * ratio), pager_h)
            return

        pager_w = int(pager_h * ratio * len(the_screen) / (self.__n_rows ** 2))
        log.debug("pager size: %s x %s" % (pager_w, pager_h))

        self.set_size_request(pager_w, pager_h)


class OwsApplet(gnomeapplet.Applet):

    menu = '''
<popup name="button3">

   <menuitem name="Item 4" verb="tile_horiz" label="tile horizontally"
       pixtype="stock" pixname="Tile Horizontally"/>
   <menuitem name="Item 5" verb="tile_vert" label="tile vertically"
       pixtype="stock" pixname="Tile Vertically"/>
   <menuitem name="Item 6" verb="tile_mosaic" label="tile mosaic"
       pixtype="stock" pixname="Tile Mosaic"/>
  <separator/>
   <menuitem name="Item 3" verb="mark_window" label="mark active window"
       pixtype="stock" pixname="gtk-leave-fullscreen"/>
   <menuitem name="Item 3" verb="clean_marks" label="clean all marks"
       pixtype="stock" pixname="gtk-clear"/>
  <separator/>
   <menuitem name="Item 2" verb="Preferences" label="Preferences"
      pixtype="stock" pixname="gtk-preferences"/>
   <menuitem name="Item 1" verb="About" label="About..."
      pixtype="stock" pixname="gtk-about"/>
</popup>
'''
    VERTICAL, HORIZONTAL, MOSAIC = range(3)

    def __init__(self):
        gnomeapplet.Applet.__init__(self)
        self.workspace = None
        self._name_change_handler_id = None
        self.source_timer = 7


    def applet_init(self):
	self.screen = the_screen #wnck.screen_get_default()

        self.osd = OwsOSD()
        self.pager = Pager(self, the_screen)
        self.prefs = OwsPrefs(self, self.osd)

        self.setup_gui()
        self.reconfigure()

        self.window_pager = None

        self.setup_menu(self.menu,
                        [('Preferences', self.prefs.show),
                         ('About', self.about_cb),
                         ('tile_horiz', self.tile_horiz),
                         ('tile_vert', self.tile_vert),
                         ('tile_mosaic', self.tile_mosaic),
                         ('mark_window', self.mark_window),
                         ('clean_marks', self.clean_marks),
                         ])

        # applet signales
        self.connect("change_background", self.background_changed)
        self.connect("change_orient",     self.orient_changed)

        # screen signals
	the_screen.connect("active_workspace_changed",
                            self.on_workspace_changed)

        the_screen.connect("workspace_created",   self.on_layout_changed)
        the_screen.connect("workspace_destroyed", self.on_layout_changed)

        # workspace signals
#        for i in range(self.get_ws_count()):
#            ws = the_screen[i]

        for ws in the_screen:
            ws.connect('name-changed', self.show_workspace_name)


    def setup_gui(self):
        self.button = gtk.Button()
        self.button.set_relief(gtk.RELIEF_NONE)
        self.button.connect('clicked', self.on_button_clicked)
        self.label = gtk.Label()
        self.label.set_use_markup(True)
        self.label.set_padding(0, 0)
        self.button.add(self.label)
#        print 'bw', self.button.get_border_width()

        self.box = None
        self.external_pager = None
        self.orient_changed()


    def setup_extPager(self):
        self.window_pager = gtk.Window()
        self.window_pager.set_title('')
        self.window_pager.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
        self.window_pager.set_position(gtk.WIN_POS_MOUSE)
        self.window_pager.set_keep_above(True)
        self.window_pager.set_property('resizable', False)
        self.window_pager.set_property("skip-pager-hint", True)
        self.window_pager.set_property("skip-taskbar-hint", True)

        self.external_pager = Pager(self, the_screen)
        self.window_pager.add(self.external_pager)
        self.window_pager.connect('delete-event', self.on_window_pager_delete_event)


    def reconfigure(self):
        if self.prefs.show_name.value:
            self.button.set_no_show_all(False)
            self.button.show()
        else:
            self.button.set_no_show_all(True)
            self.button.hide()

        self.orient_changed()

        # pager
        log.debug("set rows %s" % self.pager.set_n_rows(self.prefs.num_rows.value))
        self.pager.set_show_all(self.prefs.show_all_ws.value)
        self.pager.adjust_size()

        for i in self.prefs.items:
            i.render(self.osd)

        # Screen/workspaces
        the_screen.change_workspace_count(self.prefs.num_ws.value)
        for i in range(self.prefs.num_ws.value):
            gtk_iteration()
            ws = the_screen[i]
            if ws is not None:
                ws.change_name(self.prefs.names[i])

        #while gtk.events_pending(): gtk.main_iteration()
        self.show_workspace_name(the_screen.get_active_workspace())
        #self.on_workspace_changed()


#    def get_ws_count(self):
#        while gtk.events_pending(): gtk.main_iteration()
#        return self.screen.get_workspace_count()


    def on_layout_changed(self, *args):
        # FIXME: Hay que controlar la creación y destrucción de
        # workspaces para conectar/desconectar la señal name-changed,
        # tal como está ahora la señal name-changed no funcionará para
        # workspace creados después de añadir el applet.
        log.debug('workspaces: %s' % len(the_screen))
        self.pager.adjust_size()


    def on_button_clicked(self, button):
        'Show external pager'
        if self.window_pager is None:
            self.setup_extPager()

        print self.external_pager.mode

        if self.external_pager.mode == Pager.NAMES:
            self.window_pager.hide()
            self.external_pager.mode = None
            return

        if self.external_pager.mode == Pager.CONTENT:
            self.external_pager.set_display_mode(wnck.PAGER_DISPLAY_NAME)
            self.external_pager.mode = Pager.NAMES

        else:
            self.external_pager.set_display_mode(wnck.PAGER_DISPLAY_CONTENT)
            self.external_pager.set_n_rows(self.prefs.num_rows.value)
            self.external_pager.mode = Pager.CONTENT

        self.external_pager.adjust_size(96)
        self.window_pager.show_all()

        #if button.get_active():
        #    self.external_pager.set_n_rows(self.prefs.sw_num_rows)
        #    self.window_pager.show_all()
        #else:
        #    self.window_pager.hide()


    def on_window_pager_delete_event(self, win, *args):
        self.external_pager.mode = None
        win.hide()
        return True

    def on_workspace_changed(self, screen=None, ws=None):
        while gtk.events_pending(): gtk.main_iteration()
        ws = screen.get_active_workspace()

        if self.external_pager:
            self.window_pager.hide()
            self.external_pager.mode = None

        self.show_workspace_name(ws)
        #if self.source_timer:
        #    gtk.timeout_remove(self.source_timer)
        #
        #self.source_timer = gtk.timeout_add(350, self.show_workspace_name, ws)


    def show_workspace_name(self, ws=None):

        if not self.prefs.single_color.value:
            self.osd['fontcolor'] = strcolor(self.prefs.colors[ws.get_number()])

        self.osd.display(ws.get_name())
	self.label.set_markup('<span size="small">%s</span>' % ws.get_name())
	self.show_all()
        return False


#    def on_workspace_changed(self, screen=None):
#        while gtk.events_pending(): gtk.main_iteration()
#        #self.active_ws = self.screen.get_active_workspace()
#
#        if self.external_pager:
#            self.window_pager.hide()
#            self.external_pager.mode = None
#
#        ws = screen.get_active_workspace()
#
#        log.debug('changed')
#        log.debug(str(ws))
#
#	self.show_workspace_name(ws)
#        return
#
#        if self.source_timer:
#            gobject.source_remove(self.source_timer)
#
#        self.source_timer = gobject.timeout_add(350, self.show_workspace_name, ws)



    def background_changed(self, applet, bgtype, color, pixmap):
        for w in [self, self.pager, self.button]:
            w.modify_bg(gtk.STATE_NORMAL, color)
            if pixmap is not None:
                w.get_style().bg_pixmap[gtk.STATE_NORMAL] = pixmap

    def orient_changed(self, *args):
        angles = {gnomeapplet.ORIENT_LEFT:  270,
                  gnomeapplet.ORIENT_RIGHT:  90}

        if self.box:
            self.box.remove(self.pager)
            self.box.remove(self.button)
            self.remove(self.box)
            del self.box

        orient = self.get_orient()
        if orient in angles.keys():
            self.box = gtk.VBox()
            self.pager.set_orientation(gtk.ORIENTATION_VERTICAL)
        else:
            self.box = gtk.HBox()
            self.pager.set_orientation(gtk.ORIENTATION_HORIZONTAL)

        if self.prefs.show_name.value == 1:
            pack_func = self.box.pack_start
        else:
            pack_func = self.box.pack_end

        pack_func(self.pager)
        pack_func(self.button)
        self.add(self.box)
        self.show_all()

        self.label.set_angle(angles.get(orient, 0))

    def about_cb(self, event, data=None):
        about = gtk.AboutDialog()
        about.set_name(ows_globals.long_name)
        about.set_version(ows_globals.version)
        about.set_authors(ows_globals.author)
        #about.set_artists(ows_globals.artists)
        #about.set_translator_credits(_('translator-credits'))
        about.set_logo(gtk.gdk.pixbuf_new_from_file(ows_globals.image_dir + "/ows.png"))
        about.set_license(ows_globals.license)
        about.set_wrap_license(True)
        about.set_copyright(ows_globals.copyright)

        #gtk.about_dialog_set_url_hook(open_site, ows_globals.website)
        about.set_website(ows_globals.website)
        about.run()
        about.destroy()


    def mark_window(self, event, data=None):
        print "mark"
        the_screen.mark_active_window()

    def clean_marks(self, event, data=None):
        the_screen.clean_marked_windows()

    def tile_vert(self, event, data=None):
        self.tile(self.VERTICAL)

    def tile_horiz(self, event, data=None):
        self.tile(self.HORIZONTAL)

    def tile_mosaic(self, event, data=None):
        self.tile(self.MOSAIC)

    def tile_head(self, head, mode):
        wins = head.windows
        if not wins:
            return

        if len(wins) == 1 and not wins[0].is_above() and \
                not wins[0] in the_screen.get_marked_windows():
            wins[0].maximize()
            return

        if mode == self.VERTICAL:
            head.relocate(len(wins), 1)

        elif mode == self.HORIZONTAL:
            head.relocate(1, len(wins))

        elif mode == self.MOSAIC:
            grids = [(1,1), (2,1), (2,2), (3,2), (4,2),
                     (3,3), (4,3), (4,4)]
            for c,r in grids:
                if c * r >= len(wins): break

            head.relocate(c, r)

        else:
            log.error("tile mode error: %s" % mode)


    def tile(self, mode):
        for head in the_screen.get_heads():
            self.tile_head(head, mode)

class WindowView:
    "Shows a list of windows per head"

    (NAME_COL,
     X_COL,
     Y_COL,
     WIDTH_COL,
     HEIGHT_COL,
     STICKY_COL,
     NUM_COLS) = range(7)

    def __init__(self, gui):
        self.gui = gui
        self.treeview = self.gui.get_object('treeview_windows')
        self.populate_store()

        self.gui.get_object('button_update').connect(
            'clicked', self.on_button_update_clicked)


    def populate_store(self, *args):
        self.store = gtk.TreeStore(str, int, int, int, int, bool)

        for head in the_screen.get_heads():
            iter = self.store.append(None)
            self.store.set(iter,
                           self.NAME_COL, head.name,
                           self.X_COL, head.x,
                           self.Y_COL, head.y,
                           self.WIDTH_COL, head.w,
                           self.HEIGHT_COL, head.h,
                           self.STICKY_COL, False)


            # add children
            for win in head.windows:
                wing = Geometry.from_window(win)
                child_iter = self.store.append(iter);
                self.store.set(child_iter,
                               self.NAME_COL, wing.name,
                               self.X_COL, wing.x,
                               self.Y_COL, wing.y,
                               self.WIDTH_COL, wing.w,
                               self.HEIGHT_COL, wing.h,
                               self.STICKY_COL, win.is_above())

        self.treeview.set_model(self.store)
        self.treeview.expand_all()


    def on_button_update_clicked(self, *args):
        self.populate_store()


class Geometry:
    "size atributes for windows or heads"
    def __init__(self, name, x, y, w, h):
        self.name = name
        self.x = int(x)
        self.y = int(y)
        self.w = int(w)
        self.h = int(h)

    @classmethod
    def from_window(self, win):
        return Geometry(win.get_name(), *win.get_geometry())

    def to_list(self):
        return [self.name, self.x, self.y, self.w, self.h]


class Screen:
    def __init__(self):
        self.inner = wnck.screen_get_default()
        self._marked_windows = []

    @property
    def aspect_ratio(self):
        return float(self.inner.get_width()) / self.inner.get_height()

    @property
    def width(self):
        return self.inner.get_width()

    @property
    def height(self):
        return self.inner.get_height()

    def __len__(self):
        "number of workspaces"
        while gtk.events_pending(): gtk.main_iteration()
        return self.inner.get_workspace_count()

    def __getitem__(self, i):
        "index workspaces"
        return self.inner.get_workspace(i)

    def __iter__(self):
        for ws in self.inner.get_workspaces():
            yield ws

    def connect(self, signal, handler, *args):
        return self.inner.connect(signal, handler, *args)

    def get_active_workspace(self):
        if debug:
            return the_screen[0]
        else:
            return self.inner.get_active_workspace()

    def change_workspace_count(self, n):
        return self.inner.change_workspace_count(n)

    def get_windows(self):
        "All windows"
        return self.inner.get_windows()

    def get_windows_in_workspace(self, ws):
        "All windows in the given workspace"
        return [x for x in self.get_windows()
                  if x.is_visible_on_workspace(ws)]

    def get_visible_windows(self):
        "All windows in active workspace"
        return self.get_windows_in_workspace(self.get_active_workspace())

    def get_normal_windows(self, ws):
        "All normal windows in the given workspace"
        return [x for x in self.get_windows_in_workspace(ws)
                if x.get_window_type() == wnck.WINDOW_NORMAL]

    def get_normal_visible_windows(self):
        "All normal windows in the active workspace"
        return self.get_normal_windows(self.get_active_workspace())

    def mark_active_window(self):
        active = None
        for w in self.get_normal_visible_windows():
            if w.is_active():
                print w
                self._marked_windows.append(w)

    def get_marked_windows(self):
        return self._marked_windows[:]

    def clean_marked_windows(self):
        del self._marked_windows[:]

    def get_workspace_name(self, i):
        return self.inner.get_workspace(i).get_name()

    def get_heads(self):
        retval = []
        xrandr = commands.getoutput('xrandr --current | grep -v "^ " | grep -v "connected (" | grep -v "Screen"')
        for line in xrandr.split('\n'):
            fields = line.split()
            name = fields[0]
            size, x, y = fields[2].split('+')
            w, h = size.split('x')
            retval.append(Head(name, x, y ,w, h))

        return retval


    def __str__(self):
        return "<Screen %sx%s>" % (self.width, self.height)


def print_wins(lis):
    for w in lis:
        print '- %s %s -- %s' % (w.get_application().get_name(), # w.get_name(),
                                 w.get_geometry(),
                                 w.get_client_window_geometry())
    print


class Head(Geometry):
    def __init__(self, name,x,y,w,h):
        Geometry.__init__(self, name,x,y,w,h)

        #adjust head geometry depending on panels
        panels = self.filter_windows([x for x in the_screen.get_visible_windows()
                                      if x.get_window_type() == wnck.WINDOW_DOCK])

        print self
        print "panels:"
        print_wins(panels)
        for p in panels:
            x,y,w,h = p.get_geometry()
            if w > h: # horiz
                self.h -= h
                if y == self.y: # top
                    self.y += h

            else: # vert
                self.w -= w
                if x == self.x: # left
                    self.x += x

        self.windows = self.filter_windows(the_screen.get_normal_visible_windows())
        self.sort(self.windows)

        print self


    def filter_windows(self, wins):
        '''filter windows in this head

        A windows is in a head when its center is inside.
        '''
        retval = []
        for win in wins:
            x,y,w,h = win.get_geometry()
            center_x = x + (w / 2)
            center_y = y + (h / 2)
            if self.x <= center_x < self.x+self.w and self.y <= center_y <= self.y+self.h:
                retval.append(win)

        return retval


    def sort(self, windows):

        def wincmp(win1, win2):
            marked = the_screen.get_marked_windows()

            retval = cmp(win2 in marked, win1 in marked)
            if retval == 0:
                retval = cmp(win2.is_above(), win1.is_above())

            if retval == 0:
                retval = cmp(win1.get_application().get_name(),
                             win2.get_application().get_name())

            return retval;

        windows.sort(cmp=wincmp)


    def relocate(self, cols, rows):

        def change(win, *target):
            print "move «%s» to %s" % (win.get_application().get_name(), target)

            for i in range(20):
                win.set_geometry(wnck.WINDOW_GRAVITY_STATIC, 15, *target)

                time.sleep(0.001)
                gtk_iteration()

                x,y,w,h = win.get_geometry()
                if (x,y,w,h) == target: break

            print "real coords:", x,y,w,h
            return x,y,w,h


        print "\nRelocating %s to %sx%s" % (self, cols, rows)

        for w in self.windows:
            w.unmaximize()

        print 'init:', self.x, self.y
        window_height = self.h / rows

        n_rows = rows
        windex = 0
        iterwins = iter(self.windows)
        goout = False
        ynext = self.y

        for r in range(rows):
            xnext = self.x
            heights = []
            window_width = self.w / cols

            for c in range(cols):
                print "row: %s, col: %s" % (r,c)

                try:
                    win = self.windows[windex]
                    windex += 1
                except IndexError:
                    goout = True
                    break

                print win.get_application().get_name(), xnext, ynext

                if win.is_above() or win in the_screen.get_marked_windows():
                    x,y,w,h = win.get_geometry()
#                    x,y,w,h = change(win, xnext, ynext, w, h)
#                    xnext = x+w
#                    self.x = xnext
#                    self.w -= w

                    wx = self.x + self.w - w
                    x,y,w,h = change(win, wx, ynext, w, h)
                    self.w -= w
                    continue

                if c == cols-1 or windex == len(self.windows):
                    this_width = self.w - xnext + self.x
                else:
                    this_width = window_width

                if r == rows-1 or windex == len(self.windows):
                    this_height = self.h - ynext + self.y
                else:
                    this_height = window_height

                x,y,w,h = change(win, xnext, ynext, this_width, this_height)

                xnext = x+w
                heights.append(y+h)
                print 'heights', heights

            if heights:
                ynext = max(heights)
            else:
                ynext = self.y
                if n_rows > 1:
                    n_rows -= 1

            window_height = self.h / n_rows

            if goout: break


    def __str__(self):
        return "<Head %s %sx%s +%s+%s>" % (self.name, self.w, self.h, self.x, self.y)


the_screen = Screen()
print the_screen


def ows_factory_init(applet, id=None):
    return applet.applet_init()


#To test in a standalone window, type "./ows/ows_applet.py window"
if len(sys.argv) == 2 and sys.argv[1] in ['window', 'debug']:
    if sys.argv[1] == 'debug': debug=True

    console = log.StreamHandler()
    console.setLevel(log.DEBUG)
    console.setFormatter(log.Formatter('%(levelname)-8s: %(message)s'))
    log.getLogger().addHandler(console)

    main_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
    main_window.set_resizable(False)
    main_window.connect("destroy", gtk.main_quit)
    app = OwsApplet()
    app.reparent(main_window)
    ows_factory_init(app)
    main_window.show_all()
    gtk.main()
    sys.exit()


log.debug('-- applet init')
try:
    gobject.type_register(OwsApplet)
    gnomeapplet.bonobo_factory("OAFIID:GNOME_OSDWorkspaceSwitcher_Factory",
                               OwsApplet.__gtype__,
                               "workspace OSD", "0", ows_factory_init)
except Exception, e:
    exception()

