///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file CircleCreationMode.h
 * \brief Contains the definition of the StdObjects::CircleCreationMode class.
 */

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>

#include <stdobjects/StdObjects.h>
#include "CircleObject.h"

namespace StdObjects {

/**
 * \brief This input mode lets the user creat a procedural CircleObject in the viewports.
 *
 * \author Alexander Stukowski
 */
class CircleCreationMode : public SimpleCreationMode
{
public:

	/// \brief Default constructor.
	CircleCreationMode() : SimpleCreationMode(tr("Create Circle"), tr("Circle")) {}

protected:

	/// \brief This creates the actual scene object.
	/// \return The scene object to be inserted into the scene.
	virtual SceneObject::SmartPtr createObject() {
		return new CircleObject();
	}

	/// \brief Will be called when the user presses the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMousePressed(QMouseEvent* event) {
		SimpleCreationMode::onMousePressed(event);

		if(clickCount() == 1) {
			// Take click point as center.
			centerPoint = radiusPoint = event->pos();
			center = ORIGIN;
			if(!viewport()->snapPoint(centerPoint, center))
				onAbort();
			else {
				// Position scene node.
				AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(center - ORIGIN);
				objectNode()->transformationController()->setValue(0, tm);
			}
		}
	}

	/// \brief Will be called when the user releases the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMouseReleased(QMouseEvent* event) {
		SimpleCreationMode::onMouseReleased(event);
		if(clickCount() == 2) {
			if(radiusPoint != centerPoint) onFinish();
			else onAbort();
		}
	}

	/// \brief Will be called when the user moves the mouse while the operation is active.
	/// \param event The mouse event to be handled.
	virtual void onMouseDrag(QMouseEvent* event) {
		SimpleCreationMode::onMouseDrag(event);
		if(clickCount() == 1) {
			radiusPoint = event->pos();
			radius = ORIGIN;
			if(!viewport()->snapPoint(radiusPoint, radius))
				return;

			abortAdjustOperation();
			beginAdjustOperation();

			// Compute circle radius.
			FloatType r = Distance(center, radius);

			// Resize circle.
			static_object_cast<CircleObject>(object())->radiusController()->setValue(0, r);
		}

		// Update viewports immediately.
		VIEWPORT_MANAGER.processViewportUpdates();
	}

private:

	/// The center of the circle in construction plane coordinates.
    Point3 center;

	/// The screen coordinates of the center.
	QPoint centerPoint;

	/// The second point in construction plane coordinates that defines the circle radius.
	Point3 radius;

	/// The screen coordinates of the radius point.
	QPoint radiusPoint;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CircleCreationMode)
};

};
