///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OPENGL_SHADER_H
#define __OPENGL_SHADER_H

#include <core/Core.h>
#include "OpenGLInterface.h"

namespace Core {

class Window3D;	// Defined in Window3D.h

/**
 * \brief Manages a shader program for OpenGL rendering.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT OpenGLShader : public QObject
{
	Q_OBJECT

public:

	/// Constructor - The shader object will be added to the
	/// container's list of shader objects.
	OpenGLShader(Window3D* container, const QString& name);

	/// Loads the source files for the vertex and the fragment shader and compiles them.
	/// Throws an exception when an error occurs.
	void loadShader(const QString& vertexShaderPath, const QString& fragmentShaderPath);

	/// Activates or deactivates the shader.
	void setEnabled(bool on);

	/// Indicates that the shader has been succesfully compiled.
	bool isValid() const { return _isValid; }

	/// Sets the value of a uniform variable in the shader program.
	void sendUniform1i(const char* name, int value);
	/// Sets the value of a uniform variable in the shader program.
	void sendUniform1f(const char* name, float value);
	/// Sets the value of a uniform variable in the shader program.
	void sendUniform2f(const char* name, float value1, float value2);
	/// Sets the value of a uniform variable in the shader program.
	void sendUniform3f(const char* name, const Point3& p);
	/// Sets the value of a uniform variable in the shader program.
	void sendUniform3fv(const char* name, const float* array);

	/// Returns storage location of an vertex attribute used by the vertex shader.
	GLint getAttribLocation(const char* name);

private:

	/// The extension function pointers.
	const OpenGLExtensions& ogl;

	/// The program object handle
	GLhandleARB programObject;
	/// The vertex shader object handle
	GLhandleARB vertexShaderObject;
	/// The fragment shader object handle
	GLhandleARB fragmentShaderObject;

	/// Indicates that the shader has been succesfully compiled.
	bool _isValid;

	/// Prints the compilation log to the console.
	void printInfoLog(GLhandleARB handle);
};

};

#endif // __OPENGL_INTERFACE_H
