///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ViewportInputHandler.h
 * \brief Contains the definition of the Core::ViewportInputHandler.
 */

#ifndef __OVITO_VIEWPORT_INPUT_HANDLER_H
#define __OVITO_VIEWPORT_INPUT_HANDLER_H

#include <core/Core.h>

namespace Core {

class Viewport;			// defined in Viewport.h

/**
 * \brief Abstract base class for viewport input handlers that process mouse events
 *        in the viewport windows.
 *
 * The ViewportInputManager holds a stack of ViewportInputHandler derived objects.
 * The topmost handler on the stack handles the mouse message for the viewport
 * windows.
 *
 * \author Alexander Stukowski
 * \sa ViewportInputManager
 */
class CORE_DLLEXPORT ViewportInputHandler : public PluginClass
{
public:

	/// \brief These are the activation behaviour types for input handlers.
	enum InputHandlerType {
		NORMAL,				///< The handler is temporarely suspended when another handler becomes active.
		TEMPORARY,			///< The handler is completely removed from the stack when another handler becomes active.
		EXCLUSIVE			///< The stack is cleared before the handler becomes active.
	};

	/// \brief Default constructor.
	ViewportInputHandler() : PluginClass(), temporaryNavMode(NULL) {}

	/// \brief Returns the activation behaviour of this input handler.
	/// \return The activation type controls what happens when the handler ist activated and deactivated.
	///         The returned value is interpreted by the ViewportInputManager.
	virtual InputHandlerType handlerActivationType() = 0;

	/// \brief Handles the mouse down events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// The default implementation of this method deactivates the
	/// input handler when the user presses the right mouse button.
	/// It also activates temporary viewport navigation modes like
	/// pan, zoom and orbit when the user presses the appropriate
	/// mouse/key combination.
	///
	/// New implementations of this virtual handler method can decide
	/// whether they pass the mouse event on to the base implementation.
	///
	/// \sa temporaryNavigationMode()
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse up events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// The default implementation of this method deactivates any
	/// temporary viewport navigation mode like pan, zoom and orbit
	/// when it has been activated in the onMouseDown() method.
	virtual void onMouseUp(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse move events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// The default implementation of this method passes the event on to any
	/// temporary viewport navigation mode like pan, zoom and orbit
	/// when it has been activated in the onMouseDown() method.
	virtual void onMouseMove(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse enter events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// This method is called by the system when the mouse cursor
	/// has entered the viewport area.
	virtual void onMouseEnter(Viewport& vp, QEvent* event) {}

	/// \brief Handles the mouse leave events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// This method is called by the system when the mouse cursor
	/// has left the viewport area.
	virtual void onMouseLeave(Viewport& vp, QEvent* event) {}

	/// \brief Handles the mouse wheel events for a Viewport.
	/// \param vp The viewport in which the mouse event occured.
	/// \param event The mouse event.
	///
	/// The default implementation of this method zooms into
	/// or out of the scene depending on the wheel direction.
	virtual void onMouseWheel(Viewport& vp, QWheelEvent* event);

	/// \brief Return the cursor to be used for the viewport windows
	///        while this input handler is active.
	/// \return A cursor.
	///
	/// The default implementation returns the standard arrow cursor.
	///
	/// An input handler can update the current dynamically by calling
	/// updateCursor().
	///
	/// \sa updateCursor()
	virtual QCursor getCursor() { return QCursor(Qt::ArrowCursor); }

	/// \brief Return the temporary navigation mode if the user is currently using the
	///        middle button or the mouse wheel.
	/// \return The viewport navigation mode that temporarly overrides this
	///         input mode or \c NULL if it is not active.
	///
	/// The default implementation of the onMouseDown() event handler method activates
	/// a temporary navigation mode like zoom, pan or orbit on special mouse/key combinations.
	/// This temporary navigation mode then handles all mouse events instead of this
	/// input handler as long as it is active.
	ViewportInputHandler* temporaryNavigationMode() const { return temporaryNavMode; }

	/// \brief Indicates whether this input mode renders into the viewports.
	/// \return \c true if the renderOverlay() method has been overridden for this class; \c false otherwise.
	///
	/// Subclasses should override this method to return \c true if they also override the renderOverlay() method.
	/// The default implementation returns \c false.
	///
	/// \sa renderOverlay()
	virtual bool hasOverlay() { return false; }

	/// \brief Lets the input mode render its overlay content in a viewport.
	/// \param vp The viewport into which the mode should render its specific overlay content.
	/// \param isActive Indicates whether this input is currently active. The renderOverlay()
	///                 method is also called for an inactive input mode if it is suspended due to
	///                 one or more modes on top of it on the mode stack.
	///
	/// This method is called by the system every time the viewports are redrawn and this input
	/// mode is on the input mode stack.
	///
	/// The default implementation of this method does nothing. If a subclasses implements this
	/// method then it should also override the hasOverlay() function.
	///
	/// \sa hasOverlay()
	virtual void renderOverlay(Viewport* vp, bool isActive) {}

protected:

	/// \brief This is called by the system after the input handler has
	///        become the active handler.
	///
	/// Implementations of this virtual method in sub-classes should call the base implementation.
	///
	/// \sa ViewportInputManager::pushInputHandler()
	virtual void onActivated();

	/// \brief This is called by the system after the input handler is
	///        no longer the active handler.
	///
	/// Implementations of this virtual method in sub-classes should call the base implementation.
	///
	/// \sa ViewportInputManager::removeInputHandler()
	virtual void onDeactivated();

	/// \brief Updates the cursor in the viewport.
	///
	/// This method can be called by an input handler to change the current viewport
	/// cursor dynamically. After calling updateCursor() the system will query
	/// the handler for the new cursor through the getCursor() method.
	///
	/// \sa getCursor()
	void updateCursor();

private:

	/// Contains one of the temporary navigation modes if the user is using the
	/// middle button or the mouse wheel.
	ViewportInputHandler* temporaryNavMode;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ViewportInputHandler)

	friend class ViewportInputManager;
};

/**
 * \brief Base class for simple viewport input handlers.
 *
 * Mouse capturing is automatically handled by this class.
 *
 * The following event handling methods should by overridden by
 * sub-classes:
 * onAbort(), onMousePressed(), onMouseReleased(), onMouseDrag() and onMouseFreeMove().
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SimpleInputHandler : public ViewportInputHandler
{
protected:

	/// \brief Default constructor.
	SimpleInputHandler() : ViewportInputHandler(), _viewport(NULL) {}

	////////////////////////// Overridables ////////////////////////

	/// \brief This method is called when the user aborts the operation by clicking the right
	///        mouse button or activating another input mode.
	///
	/// When overriden, the base implementation should be called.
	///
	/// \sa onFinish()
	virtual void onAbort();

	/// \brief Is called when the user finishes the operation.
	///
	/// When overriden, the base implementation should be called.
	///
	/// \sa onAbort()
	virtual void onFinish();

	/// \brief Will be called when the user presses the left mouse button.
	/// \param event The mouse event to be handled.
	///
	/// The viewport that received the mouse event can be accessed via the viewport() method.
	virtual void onMousePressed(QMouseEvent* event) {}

	/// \brief Will be called when the user releases the left mouse button.
	/// \param event The mouse event to be handled.
	///
	/// The viewport that received the mouse event can be accessed via the viewport() method.
	virtual void onMouseReleased(QMouseEvent* event) {}

	/// \brief Will be called when the user moves the mouse while the operation is active.
	/// \param event The mouse event to be handled.
	///
	/// The viewport that received the mouse event can be accessed via the viewport() method.
	virtual void onMouseDrag(QMouseEvent* event) {}

	/// \brief Will be called when the user moves the mouse while the operation is not active.
	/// \param vp The viewport in which the mouse cursor is located.
	/// \param event The mouse event to be handled.
	///
	/// \sa snapPreview()
	virtual void onMouseFreeMove(Viewport& vp, QMouseEvent* event) {}

protected:

	/// Handles the mouse down event for the given viewport.
	void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// Handles the mouse up event for the given viewport.
	void onMouseUp(Viewport& vp, QMouseEvent* event);

	/// Handles the mouse move event for the given viewport.
	void onMouseMove(Viewport& vp, QMouseEvent* event);

	/// Handles the mouse leave event for the given viewport.
	void onMouseLeave(Viewport& vp, QEvent* event);

	/// Will be called when the input is deactivated.
	void onDeactivated();

protected:

	/// \brief Shows a preview snapping marker in the viewport when snapping is enabled.
	/// \param vp The current viewport.
	/// \param event The mouse event.
	///
	/// This method should be called from the onMouseFreeMove() implementation when
	/// this input handler supports snapping.
	void snapPreview(Viewport& vp, QMouseEvent* event);

	/// \brief Returns the current viewport we are working in.
	/// \return The viewport that received the mouse event.
	Viewport* viewport() const { return _viewport; }

private:

	/// The current viewport we are working in.
	Viewport* _viewport;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(SimpleInputHandler)
};

};

#endif // __OVITO_VIEWPORT_INPUT_HANDLER_H
