///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_NAVIGATION_MODES_H
#define __OVITO_NAVIGATION_MODES_H

#include <core/Core.h>
#include "ViewportInputHandler.h"

namespace Core {

/**
 * \brief Base class for viewport navigation modes likes zoom, pan and orbit.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT NavigationMode : public ViewportInputHandler
{
	Q_OBJECT
	
public:

	/// \brief Returns the activation behaviour of this input handler.
	///
	/// Viewport navigation modes are always temporary.
	virtual InputHandlerType handlerActivationType() { return TEMPORARY; }

	/// \brief Handles the mouse down event for the given viewport.
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse up event for the given viewport.
	virtual void onMouseUp(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse move event for the given viewport.
	virtual void onMouseMove(Viewport& vp, QMouseEvent* event);

protected:

	/// Protected constructor.
	NavigationMode() : viewport(NULL) {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos) {}

	/// Computes the new view zoom based on the new mouse position.
	virtual void modifyZoom(Viewport& vp, const QPoint& currentPos) {}

	/// \brief This is called by the system after the input handler is
	///        no longer the active handler.
	virtual void onDeactivated();
	
protected:

	/// Mouse position at first click.
	QPoint startPoint;

	/// The old view transformation matrix.
	AffineTransformation oldViewMatrix;

	/// The old zoom factor.
	FloatType oldZoom;
	
	/// The current viewport we are working in.
	Viewport* viewport;	
};

/******************************************************************************
* The pan viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT PanMode : public NavigationMode 
{
	Q_OBJECT
	
protected:
	
	/// \brief Protected constructor to prevent the creation of second instances.
	PanMode() : NavigationMode() {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_pan.png")); }

public:

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static PanMode* instance() { 
		static intrusive_ptr<PanMode> instance(new PanMode());
		return instance.get(); 
	}
};


/******************************************************************************
* The zoom viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT ZoomMode : public NavigationMode 
{
	Q_OBJECT

protected:

	/// \brief Protected constructor to prevent the creation of second instances.
	ZoomMode() : NavigationMode() {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Computes the new view zoom based on the new mouse position.
	virtual void modifyZoom(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_zoom.png")); }

public:

	/// Zooms the viewport in or out.
	void Zoom(Viewport& vp, FloatType steps);

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static ZoomMode* instance() { 
		static intrusive_ptr<ZoomMode> instance(new ZoomMode());
		return instance.get(); 
	}
};

/******************************************************************************
* The orbit viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT OrbitMode : public NavigationMode 
{
	Q_OBJECT
	
protected:

	/// \brief Protected constructor to prevent the creation of second instances.
	OrbitMode() : NavigationMode(), selectionCenter(true) {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_orbit.png")); }

	/// \brief Handles the mouse down event for the given viewport.
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// Contains the current orbiting center.
	Point3 orbitCenter;
	
	/// Indicates that the camera should orbit around the current selection.
	bool selectionCenter;

public:

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static OrbitMode* instance() { 
		static intrusive_ptr<OrbitMode> instance(new OrbitMode());
		return instance.get(); 
	}
};


};

#endif // __OVITO_NAVIGATION_MODES_H
