///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Window3D.h>
#include <mesh/tri/TriMesh.h>

using namespace Mesh;

namespace Core {

#define CLIP_RIGHT_BIT   0x01
#define CLIP_LEFT_BIT    0x02
#define CLIP_TOP_BIT     0x04
#define CLIP_BOTTOM_BIT  0x08
#define CLIP_NEAR_BIT    0x10
#define CLIP_FAR_BIT     0x20
#define CLIP_USER_BIT    0x40
#define CLIP_ALL_BITS    0x3F

/******************************************************************************
* Draw some text using the current drawing color.
*
* Parameter:
*		x    - X drawing position in pixels.
*		y    - Y drawing position in pixels.
******************************************************************************/
void Window3D::renderText(int x, int y, const QString& text)
{
	if(isRendering() && !text.isEmpty()) {

		QGLWidget::renderText(x, y, text);

		// Clear any error flags.
		while(glGetError() != GL_NO_ERROR);
	}
}

/******************************************************************************
* Computes the bounding box of the given text string when rendered with the current font.
******************************************************************************/
QRect Window3D::textExtent(const QString& text)
{
	return container()->fontMetrics().boundingRect(text);
}

/******************************************************************************
* Returns the ascender height of the current font.
******************************************************************************/
int Window3D::textAscender()
{
	return container()->fontMetrics().ascent();
}

/******************************************************************************
* Renders a polyline.
******************************************************************************/
void Window3D::renderPolyLine(size_t numberOfVertices, bool close, const Box3& boundingBox, const Point3* vertices, const ColorA* vertexColors, const RenderEdgeFlag* edgeFlags)
{
	OVITO_ASSERT_MSG(numberOfVertices >= 2, "Window3D::renderPolyLine()", "Minimum number of polyline vertices is 2.");
	if(isRendering()) {

		// Disable lighting.
		glPushAttrib(GL_LIGHTING_BIT);
		glDisable(GL_LIGHTING);

		glBegin(GL_LINES);
		size_t v1, v2, i;
		if(close) { v2 = numberOfVertices - 1; i = 0; }
		else { v2 = 0; i = 1; }
		for(; i<numberOfVertices; i++) {
			v1 = v2; v2 = i;
			if(edgeFlags && edgeFlags[v1] != RENDER_EDGE_VISIBLE) continue;
			if(vertexColors) glColor4v(vertexColors[v1].constData());
			glVertex3v(vertices[v1].constData());
			if(vertexColors) glColor4v(vertexColors[v2].constData());
			glVertex3v(vertices[v2].constData());
		}
		glEnd();
		glPopAttrib();

		enlargeSceneExtentOS(boundingBox);
	}
	else if(pickingRegion()) {
		hitTestPolyLine(numberOfVertices, close, vertices, edgeFlags);
	}
}

/******************************************************************************
* Renders a set of line segments.
******************************************************************************/
void Window3D::renderLines(size_t numberOfVertices, const Box3& boundingBox, const Point3* vertices, const ColorA* vertexColors, const RenderEdgeFlag* edgeFlags)
{
	OVITO_ASSERT_MSG((numberOfVertices & 1) == 0, "Window3D::renderLines()", "Number of vertices must be even.");
	if(isRendering()) {

		// Disable lighting.
		CHECK_OPENGL(glPushAttrib(GL_LIGHTING_BIT));
		glDisable(GL_LIGHTING);

		glBegin(GL_LINES);
		numberOfVertices >>= 1;
		for(size_t i=0; i<numberOfVertices; i++) {
			if(edgeFlags && edgeFlags[i] != RENDER_EDGE_VISIBLE) continue;
			if(vertexColors) glColor4v((vertexColors++)->constData());
			glVertex3v(vertices->constData());
			vertices++;
			if(vertexColors) glColor4v((vertexColors++)->constData());
			glVertex3v(vertices->constData());
			vertices++;
		}
		glEnd();

		glPopAttrib();

		enlargeSceneExtentOS(boundingBox);
	}
	else if(pickingRegion()) {
		hitTestLines(numberOfVertices, vertices, edgeFlags);
	}
}

/******************************************************************************
* Renders a triangle mesh in wireframe mode.
******************************************************************************/
void Window3D::renderMeshWireframe(const TriMesh& mesh)
{
	if(mesh.vertexCount() == 0 || mesh.faceCount() == 0)
		return;

	// Prepare mesh for wireframe rendering.
	if((mesh.cacheState & TriMesh::RENDER_EDGES_CACHED) == 0) {
		const_cast<TriMesh&>(mesh).buildRenderEdges();
	}
	// Make sure we have the memory layout expected by OpenGL functions.
	OVITO_STATIC_ASSERT(sizeof(TriMesh::RenderEdge) == sizeof(quint32)*2);
	OVITO_STATIC_ASSERT(sizeof(Point3) == sizeof(FloatType)*3);

	if(isRendering()) {

		if(!mesh.renderEdges.isEmpty() && !mesh.vertices().isEmpty()) {
			// Disable lighting.
			CHECK_OPENGL(glPushAttrib(GL_LIGHTING_BIT));
			CHECK_OPENGL(glDisable(GL_LIGHTING));

			CHECK_OPENGL(glEnableClientState(GL_VERTEX_ARRAY));
			CHECK_OPENGL(glVertexPointer(3, GL_FLOATING_POINT, 0, mesh.vertices().constData()));
			if(hasCompiledVertexArraysExtension()) {
				OVITO_ASSERT(glLockArrays != NULL);
				glLockArrays(0, (GLsizei)mesh.vertexCount());
			}
			CHECK_OPENGL(glDrawElements(GL_LINES, (GLsizei)mesh.renderEdges.size()*2, GL_UNSIGNED_INT, mesh.renderEdges.constData()));
			if(hasCompiledVertexArraysExtension()) glUnlockArrays();
			CHECK_OPENGL(glDisableClientState(GL_VERTEX_ARRAY));
			enlargeSceneExtent(mesh.boundingBox());

			CHECK_OPENGL(glPopAttrib());
		}
	}
	else if(pickingRegion()) {
		hitTestMeshWireframe(mesh);
	}
}

/******************************************************************************
* Renders a triangle mesh in shaded mode.
******************************************************************************/
void Window3D::renderMeshShaded(const TriMesh& mesh)
{
 	if(mesh.vertexCount() == 0 || mesh.faceCount() == 0)
		return;

	// Prepare mesh for shaded rendering.
	if((mesh.cacheState & TriMesh::RENDER_VERTICES_CACHED) == 0) {
		const_cast<TriMesh&>(mesh).buildRenderVertices();
	}
	OVITO_ASSERT(mesh.renderVertices);
	OVITO_ASSERT(!mesh.faceGroups.empty());

	// Make sure we comply with the OpenGL GL_T2F_C4F_N3F_V3F memory layout.
	OVITO_STATIC_ASSERT(sizeof(TriMesh::RenderVertex) == sizeof(float)*(2+4+3+3));

	if(isRendering()) {
		CHECK_OPENGL(glPushAttrib(GL_CURRENT_BIT));

		glEnableClientState(GL_VERTEX_ARRAY);
		glEnableClientState(GL_NORMAL_ARRAY);
		glEnableClientState(GL_COLOR_ARRAY);
		CHECK_OPENGL(glInterleavedArrays(GL_T2F_C4F_N3F_V3F, 0, mesh.renderVertices));
		if(hasCompiledVertexArraysExtension()) glLockArrays(0, (GLsizei)mesh.faceCount()*3);
		if(mesh.faceGroups.size() == 1) {
			// There is only one material group.
			// This means we can render all faces at a time.
			realizeMaterial(0);
			CHECK_OPENGL(glDrawArrays(GL_TRIANGLES, 0, (GLsizei)mesh.faceCount()*3));
		}
		else {
			// There are multiple material groups.
			// We have to render them individually.
			for(QVector<TriMesh::MaterialGroup>::const_iterator group = mesh.faceGroups.begin(); group != mesh.faceGroups.end(); ++group) {
				if(group->empty()) continue;
				realizeMaterial(group - mesh.faceGroups.begin());
				CHECK_OPENGL(glDrawElements(GL_TRIANGLES, (GLsizei)group->size(), GL_UNSIGNED_INT, group->constData()));
			}
		}
		if(hasCompiledVertexArraysExtension()) glUnlockArrays();
		glDisableClientState(GL_VERTEX_ARRAY);
		glDisableClientState(GL_NORMAL_ARRAY);
		glDisableClientState(GL_COLOR_ARRAY);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		enlargeSceneExtent(mesh.boundingBox());

		glDisable(GL_COLOR_MATERIAL);
		glPopAttrib();
	}
	else if(pickingRegion()) {
		hitTestMeshShaded(mesh);
	}
}

/******************************************************************************
* Renders a bezier shape.
******************************************************************************/
void Window3D::renderBezierShape(const BezierShape& shape)
{
	for(QVector<BezierCurve>::const_iterator curve = shape.curves().begin(); curve != shape.curves().end(); ++curve) {
		// Create polygon from curve.
		const BezierPolygon& polygon = curve->polygon();

		// Compile points.
		Point3* points = new Point3[polygon.vertexCount()];
		Point3* p = points;
		for(QVector<PolygonVertex>::const_iterator vertex = polygon.vertices().begin(); vertex != polygon.vertices().end(); ++vertex) {
			*p++ = vertex->pos();
		}

		// Render polygon.
		renderPolyLine(polygon.vertexCount(), polygon.isClosed(), curve->boundingBox(), points, NULL, NULL);

		// Clean up
		delete[] points;
	}
}

/******************************************************************************
* Activates the given material.
******************************************************************************/
void Window3D::realizeMaterial(int index)
{
	glDisable(GL_ALPHA_TEST);
	if(_materials.isEmpty() || !isRendering()) {
		_realizedMaterial = -1;
		return;
	}
	index %= _materials.size();
	if(_realizedMaterial == index) return;
	_realizedMaterial = index;

	const Window3DMaterial* mat = _materials[_realizedMaterial];
	if(mat == NULL)
		return;

	CHECK_POINTER(mat);
	float c[4];

	// Enable lighting.
	glEnable(GL_LIGHTING);

	c[0] = (float)mat->ambient.r; c[1] = (float)mat->ambient.g; c[2] = (float)mat->ambient.b; c[3] = (float)mat->opacity;
	glMaterialfv(GL_FRONT, GL_AMBIENT, c);
	c[0] = (float)mat->diffuse.r; c[1] = (float)mat->diffuse.g; c[2] = (float)mat->diffuse.b; c[3] = (float)mat->opacity;
	glMaterialfv(GL_FRONT, GL_DIFFUSE, c);
	c[0] = (float)(mat->specular.r*mat->shinStrength); c[1] = (float)(mat->specular.g*mat->shinStrength);
	c[2] = (float)(mat->specular.b*mat->shinStrength); c[3] = (float)mat->opacity;
	glMaterialfv(GL_FRONT, GL_SPECULAR, c);
	c[0] = (float)(mat->diffuse.r * mat->selfIllum); c[1] = (float)(mat->diffuse.g * mat->selfIllum);
	c[2] = (float)(mat->diffuse.b * mat->selfIllum); c[3] = 1.0f;
	glMaterialfv(GL_FRONT, GL_EMISSION, c);
	glMaterialf(GL_FRONT, GL_SHININESS, (float)pow(2.0, (double)mat->shininess*10.0));

	glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, 0);
}

/******************************************************************************
* Specifies a realtime light.
******************************************************************************/
void Window3D::setLight(int which, const Window3DLight* light)
{
	OVITO_ASSERT_MSG(which >= 0, "Window3D::setLight()", "Light index out of range.");
	if(which >= GL_MAX_LIGHTS) return;

	OVITO_ASSERT_MSG(isRendering(), "Window3D::setLight()", "setLight() may only be called while the window is in rendering mode.");

	_lights[which] = light;
	if(!isRendering()) return;

	// Setup OpenGL light.
	GLenum enumWhich = (GLenum)which;
	if(light == NULL) {
		glDisable(GL_LIGHT0 + enumWhich);
	}
	else {
		if(light->type == Window3DLight::OMNI_LIGHT) {
			AffineTransformation oldWorldTM = worldMatrix();
			setWorldMatrix(IDENTITY);
			float val[4];
			val[0] = light->color.r * light->intensity; val[1] = light->color.g * light->intensity; val[2] = light->color.b * light->intensity; val[3] = 1.0f;
			glLightfv(GL_LIGHT0 + enumWhich, GL_DIFFUSE, val);
			val[0] = light->position.X; val[1] = light->position.Y; val[2] = light->position.Z; val[3] = 1.0f;
			glLightfv(GL_LIGHT0 + enumWhich, GL_POSITION, val);
			val[0] = 1.0f; val[1] = 1.0f; val[2] = 1.0f; val[3] = 1.0f;
			glLightfv(GL_LIGHT0 + enumWhich, GL_SPECULAR, val);
			glEnable(GL_LIGHT0 + enumWhich);
			setWorldMatrix(oldWorldTM);
		}
		else if(light->type == Window3DLight::AMBIENT_LIGHT) {
			float val[4];
			val[0] = light->color.r * light->intensity; val[1] = light->color.g * light->intensity; val[2] = light->color.b * light->intensity; val[3] = 1.0f;
			glLightModelfv(GL_LIGHT_MODEL_AMBIENT, val);
			glDisable(GL_LIGHT0 + enumWhich);
		}
		else if(light->type == Window3DLight::DIRECT_LIGHT) {
			AffineTransformation oldWorldTM = worldMatrix();
			setWorldMatrix(IDENTITY);
			float val[4];
			val[0] = light->color.r * light->intensity; val[1] = light->color.g * light->intensity; val[2] = light->color.b * light->intensity; val[3] = 1.0f;
			glLightfv(GL_LIGHT0 + enumWhich, GL_DIFFUSE, val);
			glLightfv(GL_LIGHT0 + enumWhich, GL_SPECULAR, val);
			val[0] = light->position.X; val[1] = light->position.Y; val[2] = light->position.Z; val[3] = 0.0f;
			glLightfv(GL_LIGHT0 + enumWhich, GL_POSITION, val);
			glEnable(GL_LIGHT0 + enumWhich);
			setWorldMatrix(oldWorldTM);
		}
	}
}

/******************************************************************************
* Renders a marker into the viewport window.
******************************************************************************/
void Window3D::renderMarker(MarkerType mtype, const Point3& pos)
{
	if(isRendering()) {
		Vector4 cp = objectToScreenMatrix() * Vector4(pos.X, pos.Y, pos.Z, 1.0);
		if(abs(cp[3]) <= FLOATTYPE_EPSILON) return;

		FloatType iconSizeX = 5.0 / (FloatType)viewportRectangle().width();
		FloatType iconSizeY = 5.0 / (FloatType)viewportRectangle().height();
		cp[0] /= cp[3];
		cp[1] /= cp[3];
		cp[2] /= cp[3];
		cp[3] = 1.0;

		// Load identity matrix.
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glPushAttrib(GL_ENABLE_BIT);
		glDisable(GL_LIGHTING);

		switch(mtype) {
			case MARKER_CROSS:
		        glBegin(GL_LINES);
				glVertex3(cp[0] - iconSizeX, cp[1], cp[2]);
				glVertex3(cp[0] + iconSizeX, cp[1], cp[2]);
				glVertex3(cp[0], cp[1] - iconSizeY, cp[2]);
				glVertex3(cp[0], cp[1] + iconSizeY, cp[2]);
				glEnd();
				break;
			case MARKER_BOX:
		        glBegin(GL_LINE_LOOP);
				glVertex3(cp[0] - iconSizeX, cp[1] - iconSizeY, cp[2]);
				glVertex3(cp[0] + iconSizeX, cp[1] - iconSizeY, cp[2]);
				glVertex3(cp[0] + iconSizeX, cp[1] + iconSizeY, cp[2]);
				glVertex3(cp[0] - iconSizeX, cp[1] + iconSizeY, cp[2]);
				glEnd();
				break;
			default:
				OVITO_ASSERT(false);
		}

		glPopAttrib();
		// Restore old modelview tm.
		glPopMatrix();
		// Restore old projection tm.
		glMatrixMode(GL_PROJECTION);
		glPopMatrix();

		_sceneExtent.addPoint(worldMatrix() * pos);
	}
	else if(pickingRegion()) {
		Vector4 cp = objectToScreenMatrix() * Vector4(pos.X, pos.Y, pos.Z, 1.0);
		if(abs(cp[3]) <= FLOATTYPE_EPSILON) return;

		int iconSize = 5;

		Point2I sp = viewportToScreen(Point2(cp[0] / cp[3], cp[1] / cp[3]));

		switch(pickingRegion()->type()) {
			case PickRegion::POINT: {
				const PointPickRegion* ppr = static_cast<const PointPickRegion*>(pickingRegion());
				if(DistanceSquared(ppr->center(), sp) <= square(ppr->radius() + iconSize)) {
					FloatType distance;
					if(isPerspectiveProjection())
						distance = cp[3];
					else
						distance = cp[2] / cp[3];
					logHit(distance);
				}
			}
			break;
			case PickRegion::RECTANGLE: {
				const RectanglePickRegion* rectRegion = static_cast<const RectanglePickRegion*>(pickingRegion());
				if(rectRegion->rect().contains(sp.X, sp.Y)) {
					logHit();
				}
			}
			break;
			default: OVITO_ASSERT_MSG(false, "Window3D::renderMarker()", "Hit testing is not implemented yet for this picking region type.");
		}
	}
}

};
