///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __RENDER_SETTINGS_H
#define __RENDER_SETTINGS_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>
#include <core/scene/animation/TimeInterval.h>
#include <core/scene/animation/controller/Controller.h>

#include "FrameBuffer.h"

namespace Core {
	
class PluginRenderer; 		// defined in PluginRenderer.h

/******************************************************************************
* Stores the general settings for rendering output.
******************************************************************************/
class CORE_DLLEXPORT RenderSettings : public RefTarget
{
public:

	/// This enumeration specifies the animation range that should be rendered.
	enum RenderingRangeType {
		CURRENT_FRAME,		///< Render only the current animation.
		ANIMATION_INTERVAL,	///< Render the complete animation interval.
	};
	
public:

	/// Constructor.
	/// Creates an instance of the default renderer class which can be accessed via the renderer() method.
	RenderSettings(bool isLoading = false);	
	
	/// Returns the class of the current renderer or NULL if there is no current renderer.
	PluginClassDescriptor* rendererClass() const;

	/// Selects the type of renderer to use for rendering. The specified class must be derived from PluginRenderer. 
	/// This method will create a new instance of the given renderer class and stores the new renderer in this settings object.
	/// When an error occurs an exception is thrown.  
	/// \undoable
	void setRendererClass(PluginClassDescriptor* rendererClass);
	
	/// Returns the instance of the renderer class. This method will return NULL if no renderer class has been selected
	/// using the setRendererClass() method.
	PluginRenderer* renderer() const { return _renderer; }
	
	/// Returns whether only the current frame or the whole animation will be rendered.
	RenderingRangeType renderingRangeType() const { return d.renderingRangeType; } 
	/// Specifies whether only the current frame or the whole animation should be rendered.
	void setRenderingRangeType(RenderingRangeType mode);
	
	/// Returns the width of the image to be rendered in pixels.
	int imageWidth() const { return d.imageInfo.imageWidth(); }
	/// Sets the width of the image to be rendered in pixels.
	void setImageWidth(int width);
	
	/// Returns the height of the image to be rendered in pixels.
	int imageHeight() const { return d.imageInfo.imageHeight(); }
	/// Sets the height of the image to be rendered in pixels.
	void setImageHeight(int height);

	/// Returns the output filename of the rendered image.
	const QString& imageFilename() const { return d.imageInfo.filename(); }
	/// Sets the output filename of the rendered image.
	void setImageFilename(const QString& filename);

	/// Returns the background color of the rendered image.
	Color backgroundColor() const { return _backgroundColor ? (Color)_backgroundColor->getCurrentValue() : Color(0,0,0); }
	/// Sets the background color of the rendered image.
	void setBackgroundColor(const Color& color) { if(_backgroundColor) _backgroundColor->setCurrentValue(color); }
	/// Returns the controller for the background color of the rendered image.
	VectorController* backgroundColorController() const { return _backgroundColor; }
	/// Sets the controller for the background color of the rendered image.
	void setBackgroundColorController(VectorController* colorController) { _backgroundColor = colorController; }
	
	/// Returns whether the alpha channel will be generated.
	bool generateAlphaChannel() const { return d.generateAlphaChannel; }
	/// Sets whether the alpha channel will be generated.
	void setGenerateAlphaChannel(bool enable);

public:	
	
	Q_PROPERTY(int imageWidth READ imageWidth WRITE setImageWidth)
	Q_PROPERTY(int imageHeight READ imageHeight WRITE setImageHeight)
	Q_PROPERTY(QString imageFilename READ imageFilename WRITE setImageFilename)
	Q_PROPERTY(bool generateAlphaChannel READ generateAlphaChannel WRITE setGenerateAlphaChannel)
	Q_PROPERTY(RenderingRangeType renderingRangeType READ renderingRangeType WRITE setRenderingRangeType)
	Q_ENUMS(RenderingRangeType)

protected:

	/// Saves the class' contents to the given stream. 
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream. 
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// Creates a copy of this object. 
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);	
	
private:

	/// This structure stores all settings in one place to make undo/redo of changes easy.
	struct RenderSettingsData {
	
		/// Specifies whether only the current frame or the whole animation should be rendered.
		RenderingRangeType renderingRangeType;
		
		/// Specifies the properties of the rendered image.
		ImageInfo imageInfo;
		
		/// Controls whether the alpha channel will be generated.
		bool generateAlphaChannel;
	};
	
	/// Stores the actual settings.
	RenderSettingsData d;
	
	/// The instance of the plugin renderer class. 
	ReferenceField<PluginRenderer> _renderer;

	/// Controls the background color of the rendered image.
	ReferenceField<VectorController> _backgroundColor;
	
private:
    
    friend class RenderSettingsEditor;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(RenderSettings)
	DECLARE_REFERENCE_FIELD(_renderer)
	DECLARE_REFERENCE_FIELD(_backgroundColor)	
};

};

#endif // __RENDER_SETTINGS_H
