///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(BooleanPropertyUI, PropertyParameterUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
BooleanPropertyUI::BooleanPropertyUI(PropertiesEditor* parentEditor, const char* propertyName, const QString& checkBoxLabel) : 
	PropertyParameterUI(parentEditor, propertyName)
{
	// Create UI widget.
	_checkBox = new QCheckBox(checkBoxLabel);
	connect(_checkBox, SIGNAL(clicked(bool)), this, SLOT(updatePropertyValue()));	
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
BooleanPropertyUI::BooleanPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) : 
	PropertyParameterUI(parentEditor, propField) 
{
	// Create UI widget.
	_checkBox = new QCheckBox(propField.displayName());
	connect(_checkBox, SIGNAL(clicked(bool)), this, SLOT(updatePropertyValue()));	
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
BooleanPropertyUI::~BooleanPropertyUI()
{
	// Release GUI controls. 
	delete checkBox(); 
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void BooleanPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();	
	
	if(checkBox()) 
		checkBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void BooleanPropertyUI::updateUI()
{
	PropertyParameterUI::updateUI();	
	
	if(checkBox() && editObject()) {
		QVariant val(false);
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid(), "BooleanPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to bool type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid()) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to bool type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}		
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid());
		}
		checkBox()->setChecked(val.toBool());
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void BooleanPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(checkBox()) checkBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field 
* this property UI is bound to.
******************************************************************************/
void BooleanPropertyUI::updatePropertyValue()
{
	if(checkBox() && editObject()) {
		
		UNDO_MANAGER.beginCompoundOperation(tr("Change parameter"));
		 
		if(propertyName()) {
			if(!editObject()->setProperty(propertyName(), checkBox()->isChecked())) {
				OVITO_ASSERT_MSG(false, "BooleanPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			editObject()->setPropertyFieldValue(*propertyField(), checkBox()->isChecked());						
		}
		
		UNDO_MANAGER.endCompoundOperation();
	}
}

};

