///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/ObjectNode.h>
#include "Atoms2POVRay.h"

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(Atoms2POVRayExportInterface, POVRayExportInterface)

/// Exports the given scene object in the POV-Ray format and returns true.
/// Return false if the scene object type is not supported by this interface class.
/// Throws an exception if an error occured.
bool Atoms2POVRayExportInterface::ExportSceneObject(SceneObject* sceneObj, POVRayWriter& writer, ObjectNode* contextNode, const AffineTransformation& objToWorldTM)
{
	AtomsObject* atomsObj = dynamic_object_cast<AtomsObject>(sceneObj);
	if(!atomsObj) return false;

	DataChannel* posChannel = atomsObj->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel || !posChannel->size()) return true;
	DataChannel* transparencyChannel = atomsObj->getStandardDataChannel(DataChannel::TransparencyChannel);

	TimeInterval interval = TimeForever;

	if(!transparencyChannel) {
		// Define a macro for spheres in the POV-Ray file to
		// reduce file size.
		writer << "#macro ATOM(atomPos, atomRadius, atomColor)" << endl;
		writer << "sphere { atomPos, atomRadius" << endl;
		writer << "         texture { pigment { color atomColor } }" << endl;
		writer << "}" << endl;
		writer << "#end" << endl;
	}
	else {
		// Define a macro for spheres with transparency in the POV-Ray file to
		// reduce file size.
		writer << "#macro ATOM(atomPos, atomRadius, atomColor, transparency)" << endl;
		writer << "sphere { atomPos, atomRadius" << endl;
		writer << "         texture { pigment { color rgbt <atomColor.red,atomColor.green,atomColor.blue,transparency> } }" << endl;
		writer << "}" << endl;
		writer << "#end" << endl;
	}

	const Point3* p = posChannel->constDataPoint3();
	QVector<Color> atomColors = atomsObj->getAtomColors(writer.time(), interval);
	QVector<Color>::const_iterator color = atomColors.constBegin();
	QVector<FloatType> atomRadii = atomsObj->getAtomRadii(writer.time(), interval);
	QVector<FloatType>::const_iterator radius = atomRadii.constBegin();
	if(!transparencyChannel) {
		for(size_t i = 0; i < posChannel->size(); i++, ++p, ++color, ++radius) {
			Point3 worldPos = objToWorldTM * (*p);
			writer << "ATOM(" << worldPos << ", " << (*radius) << ", " << (*color) << ")" << endl;
		}
	}
	else {
		const FloatType* transparency = transparencyChannel->constDataFloat();
		for(size_t i = 0; i < posChannel->size(); i++, ++p, ++color, ++radius, ++transparency) {
			Point3 worldPos = objToWorldTM * (*p);
			writer << "ATOM(" << worldPos << ", " << (*radius) << ", " << (*color) << ", " << (*transparency) << ")" << endl;
		}
	}

	// Export simulation cell.
	FloatType lineWidth = atomsObj->simulationCell()->simulationCellLineWidth();
	if(atomsObj->simulationCell()->renderSimulationCell() && lineWidth > 0) {
		Color lineColor = atomsObj->simulationCell()->simulationCellRenderingColor();
		AffineTransformation simBox = objToWorldTM * atomsObj->simulationCell()->cellMatrix();
		Point3 corners[8];
		corners[0] = ORIGIN + simBox.getTranslation();
		corners[1] = corners[0] + simBox.column(0);
		corners[2] = corners[0] + simBox.column(0) + simBox.column(1);
		corners[3] = corners[0] + simBox.column(1);
		corners[4] = corners[0] + simBox.column(2);
		corners[5] = corners[1] + simBox.column(2);
		corners[6] = corners[2] + simBox.column(2);
		corners[7] = corners[3] + simBox.column(2);
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 5," << endl;
		writer << "    " << corners[0] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[1] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[2] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[3] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[0] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 5," << endl;
		writer << "    " << corners[4] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[5] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[6] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[7] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[4] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 2," << endl;
		writer << "    " << corners[0] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[4] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 2," << endl;
		writer << "    " << corners[1] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[5] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 2," << endl;
		writer << "    " << corners[2] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[6] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
		writer << "sphere_sweep { " << endl;
		writer << "    linear_spline 2," << endl;
		writer << "    " << corners[3] << ", " << lineWidth << "," << endl;
		writer << "    " << corners[7] << ", " << lineWidth << endl;
		writer << "    tolerance 1.0e-4" << endl;
		writer << "    texture { pigment { color " << lineColor << " } }" << endl;
		writer << "}" << endl;
	}

	return true;
}

};	// End of namespace AtomViz
