///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/VectorControllerUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/ApplicationManager.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/panels/CommandPanel.h>
#include "SliceModifier.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(SliceModifier, AtomsObjectModifierBase)
DEFINE_REFERENCE_FIELD(SliceModifier, VectorController, "PlaneNormal", _normalCtrl)
DEFINE_REFERENCE_FIELD(SliceModifier, FloatController, "PlaneDistance", _distanceCtrl)
DEFINE_REFERENCE_FIELD(SliceModifier, FloatController, "SliceWidth", _widthCtrl)
DEFINE_PROPERTY_FIELD(SliceModifier, "CreateSelection", _createSelection)
DEFINE_PROPERTY_FIELD(SliceModifier, "Inverse", _inverse)
DEFINE_PROPERTY_FIELD(SliceModifier, "ApplyToSelection", _applyToSelection)
SET_PROPERTY_FIELD_LABEL(SliceModifier, _normalCtrl, "Normal")
SET_PROPERTY_FIELD_LABEL(SliceModifier, _distanceCtrl, "Distance")
SET_PROPERTY_FIELD_LABEL(SliceModifier, _widthCtrl, "Slice width")
SET_PROPERTY_FIELD_LABEL(SliceModifier, _createSelection, "Select atoms (do not delete)")
SET_PROPERTY_FIELD_LABEL(SliceModifier, _inverse, "Invert")
SET_PROPERTY_FIELD_LABEL(SliceModifier, _applyToSelection, "Apply to selected atoms only")
SET_PROPERTY_FIELD_UNITS(SliceModifier, _normalCtrl, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SliceModifier, _distanceCtrl, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SliceModifier, _widthCtrl, WorldParameterUnit)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
SliceModifier::SliceModifier(bool isLoading) : AtomsObjectModifierBase(isLoading),
	_createSelection(false), _inverse(false), _applyToSelection(false)
{
	INIT_PROPERTY_FIELD(SliceModifier, _normalCtrl);
	INIT_PROPERTY_FIELD(SliceModifier, _distanceCtrl);
	INIT_PROPERTY_FIELD(SliceModifier, _widthCtrl);
	INIT_PROPERTY_FIELD(SliceModifier, _createSelection);
	INIT_PROPERTY_FIELD(SliceModifier, _inverse);
	INIT_PROPERTY_FIELD(SliceModifier, _applyToSelection);
	if(!isLoading) {
		_normalCtrl = CONTROLLER_MANAGER.createDefaultController<VectorController>();
		_distanceCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_widthCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();

		setNormal(Vector3(1,0,0));
	}
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval SliceModifier::modifierValidity(TimeTicks time)
{
	// Return an empty validity interval if this modifier is currently being edited
	// to make the system create a cache point just before the modifier in the
	// modifier stack. This will speed up re-evaluation of the modifier stack if the
	// user adjust this modifier's parameters interactively.
	if(isBeingEdited()) return TimeNever;

	// The local validity is otherwise given by the animatable parameters.
	TimeInterval interval = TimeForever;
	_normalCtrl->validityInterval(time, interval);
	_distanceCtrl->validityInterval(time, interval);
	_widthCtrl->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Returns the slicing plane.
******************************************************************************/
Plane3 SliceModifier::slicingPlane(TimeTicks time, TimeInterval& validityInterval)
{
	Plane3 plane;
	_normalCtrl->getValue(time, plane.normal, validityInterval);
	if(plane.normal == NULL_VECTOR) plane.normal = Vector3(0,0,1);
	else plane.normal = Normalize(plane.normal);
	_distanceCtrl->getValue(time, plane.dist, validityInterval);
	if(inverse())
		return -plane;
	else
		return plane;
}

/******************************************************************************
* Modifies the atoms object. The time interval passed
* to the function is reduced to the interval where the modified object is valid/constant.
******************************************************************************/
EvaluationStatus SliceModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	QString statusMessage = tr("Slicing results:\n%n input atoms", 0, input()->atomsCount());

	// Compute filter mask.
	dynamic_bitset<> mask(input()->atomsCount());
	size_t numRejected = filterAtoms(mask, time, validityInterval);
	size_t numKept = input()->atomsCount() - numRejected;

	if(createSelection() == false) {

		statusMessage += tr("\n%n atoms deleted", 0, numRejected);
		statusMessage += tr("\n%n atoms remaining", 0, numKept);
		if(numRejected == 0)
			return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);

		// Replace referenced data channels with real copies and apply filter.
		output()->deleteAtoms(mask);
		OVITO_ASSERT(output()->atomsCount() == input()->atomsCount() - numRejected);
	}
	else {
		statusMessage += tr("\n%n atoms deleted", 0, numRejected);
		statusMessage += tr("\n%n atoms remaining", 0, numKept);

		// Get selection data channel.
		DataChannel* selChannel = outputStandardChannel(DataChannel::SelectionChannel);
		selChannel->setVisible(true);

		int* s = selChannel->dataInt();
		for(size_t i = 0; i < selChannel->size(); i++) {
			*s++ = mask[i];
		}
	}

	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* Performs the actual rejection of atoms.
******************************************************************************/
size_t SliceModifier::filterAtoms(dynamic_bitset<>& mask, TimeTicks time, TimeInterval& validityInterval)
{
	// Get the required data channels.
	DataChannel* posChannel = expectStandardChannel(DataChannel::PositionChannel);
	OVITO_ASSERT(posChannel->size() == mask.size());

	DataChannel* selectionChannel = inputStandardChannel(DataChannel::SelectionChannel);
	if(!applyToSelection()) selectionChannel = NULL;

	FloatType sliceWidth;
	_widthCtrl->getValue(time, sliceWidth, validityInterval);
	sliceWidth /= 2;

	Plane3 plane = slicingPlane(time, validityInterval);

	size_t na = 0;
	const Point3* p = posChannel->constDataPoint3();
	const int* s = NULL;
	if(selectionChannel) s = selectionChannel->constDataInt();

	if(sliceWidth <= 0) {
		for(size_t i = 0; i < posChannel->size(); i++, ++p, ++s) {
			if(plane.pointDistance(*p) > 0) {
				if(selectionChannel && !*s) continue;
				mask.set(i);
				na++;
			}
		}
	}
	else {
		for(size_t i = 0; i < posChannel->size(); i++, ++p, ++s) {
			if(inverse() == (plane.classifyPoint(*p, sliceWidth) == 0)) {
				if(selectionChannel && !*s) continue;
				mask.set(i);
				na++;
			}
		}
	}
	return na;
}

/******************************************************************************
* Makes the modifier render itself into the viewport.
******************************************************************************/
void SliceModifier::renderModifier(TimeTicks time, ObjectNode* contextNode, ModifierApplication* modApp, Viewport* vp)
{
	TimeInterval interval;

	Box3 bb = contextNode->localBoundingBox(time);
	if(bb.isEmpty()) return;

	Plane3 plane = slicingPlane(time, interval);

	FloatType sliceWidth;
	_widthCtrl->getValue(time, sliceWidth, interval);

	if(sliceWidth <= 0) {
		renderPlane(vp, plane, bb, Color(0.8f, 0.3f, 0.3f));
	}
	else {
		plane.dist += sliceWidth / 2;
		renderPlane(vp, plane, bb, Color(0.8f, 0.3f, 0.3f));
		plane.dist -= sliceWidth;
		renderPlane(vp, plane, bb, Color(0.8f, 0.3f, 0.3f));
	}
}

/******************************************************************************
* Renders a plane in the viewport with the given color.
******************************************************************************/
void SliceModifier::renderPlane(Viewport* vp, const Plane3& plane, const Box3& bb, const Color& color) const
{
	// Compute intersection lines of slicing plane and bounding box.
	QVector<Point3> lines;
	Ray3 edges[] = { Ray3(bb[0],bb[1]), Ray3(bb[1],bb[5]), Ray3(bb[5],bb[4]), Ray3(bb[4],bb[0]),
					 Ray3(bb[1],bb[3]), Ray3(bb[0],bb[2]), Ray3(bb[5],bb[7]), Ray3(bb[4],bb[6]),
					 Ray3(bb[2],bb[3]), Ray3(bb[3],bb[7]), Ray3(bb[7],bb[6]), Ray3(bb[6],bb[2]) };

	planeQuadIntersesction(edges[0], edges[1], edges[2], edges[3], plane, lines);
	planeQuadIntersesction(edges[1], edges[4], edges[9], edges[6], plane, lines);
	planeQuadIntersesction(edges[8], edges[9], edges[10], edges[11], plane, lines);
	planeQuadIntersesction(edges[5], edges[3], edges[7], edges[11], plane, lines);
	planeQuadIntersesction(edges[0], edges[4], edges[8], edges[5], plane, lines);
	planeQuadIntersesction(edges[2], edges[6], edges[10], edges[7], plane, lines);

	// If there is not intersection with the simulation box then
	// project the simulation box onto the plane to visualize the plane.
	if(lines.empty()) {
		for(int edge = 0; edge < 12; edge++) {
			lines.push_back(plane.projectPoint(edges[edge].base));
			lines.push_back(plane.projectPoint(edges[edge].base + edges[edge].dir));
		}
	}

	// Render plane-box intersection lines.
	vp->setRenderingColor(color);
	vp->renderLines(lines.size(), bb, &lines.front());
}

/******************************************************************************
* Computes the intersection lines of a plane and a quad.
******************************************************************************/
void SliceModifier::planeQuadIntersesction(const Ray3& r1, const Ray3& r2, const Ray3& r3, const Ray3& r4, const Plane3& plane, QVector<Point3>& lines) const
{
	const Ray3* rays[] = { &r1, &r2, &r3, &r4 };
	Point3 p1, p2;
	bool hasP1 = false;
	FloatType t;
	for(size_t i=0; i<4; i++) {
		t = plane.intersectionT(*rays[i], FLOATTYPE_EPSILON);
		if(t < 0.0 || t > 1.0) continue;
		if(!hasP1) {
			p1 = rays[i]->base + rays[i]->dir * t;
			hasP1 = true;
		}
		else {
			p2 = rays[i]->base + rays[i]->dir * t;
			if(!p2.equals(p1, FLOATTYPE_EPSILON)) {
				lines.push_back(p1);
				lines.push_back(p2);
				return;
			}
		}
	}
}

IMPLEMENT_PLUGIN_CLASS(SliceModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void SliceModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Slicing Plane"), rolloutParams, "atomviz.modifiers.slice");

    // Create the rollout contents.
	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setHorizontalSpacing(0);
	layout->setVerticalSpacing(2);
	layout->setColumnStretch(1, 1);

	// Distance parameter.
	FloatControllerUI* distancePUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _distanceCtrl));
	layout->addWidget(distancePUI->label(), 0, 0);
	layout->addWidget(distancePUI->textBox(), 0, 1);
	layout->addWidget(distancePUI->spinner(), 0, 2);
	distancePUI->setWhatsThis(tr("Specifies the distance of the plane to the origin of the simulation cell coordinate system. "
							   "The distance is specified in the direction of the normal vector."));

	// Normal parameter.
	for(size_t i=0; i<3; i++) {
		VectorControllerUI* normalPUI = new VectorControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _normalCtrl), i);
		layout->addWidget(normalPUI->label(), i+1, 0);
		layout->addWidget(normalPUI->textBox(), i+1, 1);
		layout->addWidget(normalPUI->spinner(), i+1, 2);
		normalPUI->setWhatsThis(tr("Specifies the components of the normal vector that defines the slicing plane orientation."));
	}

	// Slice width parameter.
	FloatControllerUI* widthPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _widthCtrl));
	layout->addWidget(widthPUI->label(), 4, 0);
	layout->addWidget(widthPUI->textBox(), 4, 1);
	layout->addWidget(widthPUI->spinner(), 4, 2);
	widthPUI->setMinValue(0);
	widthPUI->setWhatsThis(tr("When this value is zero then all atoms on one side of the slicing plane are deleted--selected. "
							"For a positive plane width all atoms whose distance to the plane is greater/smaller than "
							"the half width are deleted/selected."));

	// Invert parameter.
	BooleanPropertyUI* invertPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _inverse));
	layout->addWidget(invertPUI->checkBox(), 5, 0, 1, 3);
	invertPUI->setWhatsThis(tr("When this option is enabled the orientation of the slicing plane is reversed."));

	// Create selection parameter.
	BooleanPropertyUI* createSelectionPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _createSelection));
	layout->addWidget(createSelectionPUI->checkBox(), 6, 0, 1, 3);
	createSelectionPUI->setWhatsThis(tr("Controls whether the atoms should be deleted or just selected."));

	// Apply to selection only parameter.
	BooleanPropertyUI* applyToSelectionPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SliceModifier, _applyToSelection));
	layout->addWidget(applyToSelectionPUI->checkBox(), 7, 0, 1, 3);
	applyToSelectionPUI->setWhatsThis(tr("When this option is enabled then the slicing modifier is only applied "
									   "to the atoms that are currently selected. If there is no selection then all atoms will be affected by the modifier."));

	// Add buttons for view alignment functions.
	QPushButton* alignViewToPlaneBtn = new QPushButton(tr("Align view to plane"), rollout);
	connect(alignViewToPlaneBtn, SIGNAL(clicked(bool)), this, SLOT(onAlignViewToPlane()));
	layout->addWidget(alignViewToPlaneBtn, 8, 0, 1, 3);
	QPushButton* alignPlaneToViewBtn = new QPushButton(tr("Align plane to view"), rollout);
	connect(alignPlaneToViewBtn, SIGNAL(clicked(bool)), this, SLOT(onAlignPlaneToView()));
	layout->addWidget(alignPlaneToViewBtn, 9, 0, 1, 3);

	pickAtomPlaneInputMode = new PickAtomPlaneInputMode();
	pickAtomPlaneInputModeAction = new ViewportModeAction("SliceModifier.AlignPlaneToAtoms", pickAtomPlaneInputMode);
	pickAtomPlaneInputModeActionProxy = new ActionProxy(pickAtomPlaneInputModeAction);
	pickAtomPlaneInputModeActionProxy->setParent(this);
	pickAtomPlaneInputModeActionProxy->setText(tr("Align plane to atoms"));
	QWidget* alignPlaneToAtomsBtn = pickAtomPlaneInputModeActionProxy->requestWidget(rollout);
	layout->addWidget(alignPlaneToAtomsBtn, 10, 0, 1, 3);

	// Status label.
	layout->addWidget(statusLabel(), 11, 0, 1, 3);
}

/******************************************************************************
* Aligns the slicing plane to the viewing direction.
******************************************************************************/
void SliceModifierEditor::onAlignPlaneToView()
{
	TimeInterval interval;

	Viewport* vp = VIEWPORT_MANAGER.activeViewport();
	if(!vp) return;

	// Get the object to world transformation for the currently selected object.
	ObjectNode* node = dynamic_object_cast<ObjectNode>(DATASET_MANAGER.currentSet()->selection()->firstNode());
	if(!node) return;
	const AffineTransformation& nodeTM = node->getWorldTransform(ANIM_MANAGER.time(), interval);
	AffineTransformation localToWorldTM = node->objectTransform() * nodeTM;

	// Get the base point of the current slicing plane in local coordinates.
	SliceModifier* mod = static_object_cast<SliceModifier>(editObject());
	if(!mod) return;
	Plane3 oldPlaneLocal = mod->slicingPlane(ANIM_MANAGER.time(), interval);
	Point3 basePoint = ORIGIN + oldPlaneLocal.normal * oldPlaneLocal.dist;

	// Get the orientation of the projection plane of the current viewport.
	Vector3 dirWorld = Normalize(vp->currentView().inverseViewMatrix * Vector3(0, 0, 1));
	Plane3 newPlaneLocal(basePoint, localToWorldTM.inverse() * dirWorld);
	if(abs(newPlaneLocal.normal.X) < FLOATTYPE_EPSILON) newPlaneLocal.normal.X = 0;
	if(abs(newPlaneLocal.normal.Y) < FLOATTYPE_EPSILON) newPlaneLocal.normal.Y = 0;
	if(abs(newPlaneLocal.normal.Z) < FLOATTYPE_EPSILON) newPlaneLocal.normal.Z = 0;

	UNDO_MANAGER.beginCompoundOperation(tr("Align plane to view"));
	mod->setNormal(Normalize(newPlaneLocal.normal));
	mod->setDistance(newPlaneLocal.dist);
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Aligns the current viewing direction to the slicing plane.
******************************************************************************/
void SliceModifierEditor::onAlignViewToPlane()
{
	TimeInterval interval;

	Viewport* vp = VIEWPORT_MANAGER.activeViewport();
	if(!vp) return;

	// Get the object to world transformation for the currently selected object.
	ObjectNode* node = dynamic_object_cast<ObjectNode>(DATASET_MANAGER.currentSet()->selection()->firstNode());
	if(!node) return;
	const AffineTransformation& nodeTM = node->getWorldTransform(ANIM_MANAGER.time(), interval);
	AffineTransformation localToWorldTM = node->objectTransform() * nodeTM;

	// Transform the current slicing plane to the world coordinate system.
	SliceModifier* mod = static_object_cast<SliceModifier>(editObject());
	if(!mod) return;
	Plane3 planeLocal = mod->slicingPlane(ANIM_MANAGER.time(), interval);
	Plane3 planeWorld = localToWorldTM * planeLocal;

	// Calculate the intersection point of the current viewing direction with the current slicing plane.
	Ray3 viewportRay = vp->viewportRay(Point2(0,0));
	FloatType t = planeWorld.intersectionT(viewportRay);
	Point3 intersectionPoint;
	if(t != FLOATTYPE_MAX)
		intersectionPoint = viewportRay.point(t);
	else
		intersectionPoint = ORIGIN + localToWorldTM.getTranslation();

	if(vp->currentView().isPerspective) {
		FloatType distance = Distance(ORIGIN + vp->currentView().inverseViewMatrix.getTranslation(), intersectionPoint);
		vp->settings()->setViewType(Viewport::VIEW_PERSPECTIVE);
		vp->settings()->setViewMatrix(AffineTransformation::lookAt(intersectionPoint + planeWorld.normal * distance,
			intersectionPoint, Vector3(0,0,1)));
	}
	else {
		vp->settings()->setViewType(Viewport::VIEW_ORTHO);
		vp->settings()->setViewMatrix(AffineTransformation::lookAt(ORIGIN, ORIGIN + (-planeWorld.normal), Vector3(0,0,1))
			* AffineTransformation::translation(Vector3(-intersectionPoint.X, -intersectionPoint.Y, -intersectionPoint.Z)));
	}
	vp->updateViewport(true);
}

/******************************************************************************
* This is called by the system after the input handler has become the active handler.
******************************************************************************/
void PickAtomPlaneInputMode::onActivated()
{
	MAIN_FRAME->statusBar()->showMessage(tr("Select three atoms to define the slicing plane."));
}

/******************************************************************************
* This is called by the system after the input handler is no longer the active handler.
******************************************************************************/
void PickAtomPlaneInputMode::onDeactivated()
{
	pickedAtoms.clear();
	MAIN_FRAME->statusBar()->clearMessage();
}

/******************************************************************************
* Handles the mouse down events for a Viewport.
******************************************************************************/
void PickAtomPlaneInputMode::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	ViewportInputHandler::onMouseDown(vp, event);

	if(event->button() == Qt::LeftButton) {

		if(pickedAtoms.size() >= 3) {
			pickedAtoms.clear();
			VIEWPORT_MANAGER.updateViewports();
		}

		PickAtomResult pickResult;
		if(pickAtom(vp, event->pos(), ANIM_MANAGER.time(), pickResult)) {

			// Do not select the same atom twice.
			if(pickedAtoms.size() >= 1 && pickedAtoms[0].worldPos.equals(pickResult.worldPos, FLOATTYPE_EPSILON)) return;
			if(pickedAtoms.size() >= 2 && pickedAtoms[1].worldPos.equals(pickResult.worldPos, FLOATTYPE_EPSILON)) return;

			pickedAtoms.push_back(pickResult);
			VIEWPORT_MANAGER.updateViewports();

			if(pickedAtoms.size() == 3) {

				// Get the slice modifier that is currently being edited.
				SliceModifier* mod = dynamic_object_cast<SliceModifier>(MAIN_FRAME->commandPanel()->editObject());
				if(mod)
					alignPlane(mod);
			}
		}
	}

}

/******************************************************************************
* Aligns the modifier's slicing plane to the three selected atoms.
******************************************************************************/
void PickAtomPlaneInputMode::alignPlane(SliceModifier* mod)
{
	OVITO_ASSERT(pickedAtoms.size() == 3);

	try {
		Plane3 worldPlane(pickedAtoms[0].worldPos, pickedAtoms[1].worldPos, pickedAtoms[2].worldPos, true);
		if(worldPlane.normal.equals(NULL_VECTOR, FLOATTYPE_EPSILON))
			throw Exception(tr("Cannot determine the new slicing plane. The three selected atoms are colinear."));

		// Get the object to world transformation for the currently selected object.
		ObjectNode* node = dynamic_object_cast<ObjectNode>(DATASET_MANAGER.currentSet()->selection()->firstNode());
		if(!node) return;
		TimeInterval interval;
		const AffineTransformation& nodeTM = node->getWorldTransform(ANIM_MANAGER.time(), interval);
		AffineTransformation localToWorldTM = node->objectTransform() * nodeTM;

		// Transform new plane from world to object space.
		Plane3 localPlane = localToWorldTM.inverse() * worldPlane;

		// Flip new plane orientation if necessary to align it with old orientation.
		if(DotProduct(localPlane.normal, mod->normal()) < 0)
			localPlane = -localPlane;

		localPlane.normalizePlane();
		UNDO_MANAGER.beginCompoundOperation(tr("Align plane to atoms"));
		mod->setNormal(localPlane.normal);
		mod->setDistance(localPlane.dist);
		UNDO_MANAGER.endCompoundOperation();
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
* Lets the input mode render its overlay content in a viewport.
******************************************************************************/
void PickAtomPlaneInputMode::renderOverlay(Viewport* vp, bool isActive)
{
	ViewportInputHandler::renderOverlay(vp, isActive);

	Q_FOREACH(const PickAtomResult& pa, pickedAtoms) {
		renderSelectionMarker(vp, pa);
	}
}

};	// End of namespace AtomViz
