///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __COORDINATION_NUMBER_MODIFIER_H
#define __COORDINATION_NUMBER_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/PropertiesEditor.h>

#include <atomviz/AtomViz.h>
#include "../../AtomsObjectAnalyzerBase.h"

#include <atomviz/atoms/datachannels/BondsDataChannel.h>
#include <atomviz/utils/NearestNeighborList.h>
#include <atomviz/utils/OnTheFlyNeighborList.h>

namespace AtomViz {

/******************************************************************************
* This modifier computes the coordination number for each atom.
******************************************************************************/
class ATOMVIZ_DLLEXPORT CoordinationNumberModifier : public AtomsObjectAnalyzerBase
{
public:

	/// Default constructor.
	CoordinationNumberModifier(bool isLoading = false);

	/// Returns the minimum coordination number from the list above.
	int minimumCoordinationNumber() const { return minCoordination; }

	/// Returns the maximum coordination number from the list above.
	int maximumCoordinationNumber() const { return maxCoordination; }

	/// Returns the data channel with the coordination number for each of the atoms.
	DataChannel* coordinationChannel() const { return _coordinationChannel; }

	/// Returns the data channel with the bonds of each of the atoms.
	BondsDataChannel* bondsChannel() const { return _bondsChannel; }

	/// \brief Returns whether the analysis results are saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	bool storeResultsWithScene() const { return coordinationChannel() ? coordinationChannel()->serializeData() : false; }

	/// \brief Returns whether analysis results are saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	/// \undoable
	void setStoreResultsWithScene(bool on) { if(coordinationChannel()) coordinationChannel()->setSerializeData(on); }

public:

	Q_PROPERTY(bool storeResultsWithScene READ storeResultsWithScene WRITE setStoreResultsWithScene)

protected:

	// RefTarget virtual functions:
	virtual void saveToStream(ObjectSaveStream& stream);
	virtual void loadFromStream(ObjectLoadStream& stream);
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// Performs the analysis.
	/// Throws an exception on error.
	/// Returns false when the operation has been canceled by the user.
	bool calculate(AtomsObject* atomsObject, bool suppressDialogs = false);

	/// The minimum coordination number.
	int minCoordination;

	/// The maximum coordination number.
	int maxCoordination;

	/// This stores the computed coordination numbers.
	ReferenceField<DataChannel> _coordinationChannel;

	/// This stores the bonds between atoms.
	ReferenceField<BondsDataChannel> _bondsChannel;

	/// Controls the output of bonds.
	PropertyField<bool> _generateBonds;

	/// Specifies the maximum number of bonds per atom that can be handled.
	PropertyField<int> _maxBonds;

private:

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	class Kernel {
	public:
		// Constructor that takes references to the input and output arrays.
		Kernel(const OnTheFlyNeighborList& _nnlist, DataChannel* _coordinationChannel, BondsDataChannel* _bondsChannel) : nnlist(_nnlist), coordinationChannel(_coordinationChannel), bondsChannel(_bondsChannel) {
			// This call is necessary to deep copy the memory array of the data channel before accessing it from multiple threads.
			coordinationChannel->dataInt();
			if(bondsChannel) bondsChannel->dataInt();
		}

		// The actual kernel function that is called by the Qt concurrent framework for each atom.
		void operator()(int atomIndex);

	private:
		/// Input nearest-neighbor list
		const OnTheFlyNeighborList& nnlist;
		/// Output data channel.
		DataChannel* coordinationChannel;
		/// Output data channel.
		BondsDataChannel* bondsChannel;
	};

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CoordinationNumberModifier)
	DECLARE_REFERENCE_FIELD(_coordinationChannel)
	DECLARE_REFERENCE_FIELD(_bondsChannel)
	DECLARE_PROPERTY_FIELD(_generateBonds)
	DECLARE_PROPERTY_FIELD(_maxBonds)
};

/******************************************************************************
* A properties editor for the CoordinationNumberModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT CoordinationNumberModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the Recalculate button.
	void onRecalculate();

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CoordinationNumberModifierEditor)
};

};	// End of namespace AtomViz

#endif // __COORDINATION_NUMBER_MODIFIER_H
