/**
  @file dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "dbus.h"

static DBusConnection *connection = NULL;

DBusConnection *get_dbus_connection(void)
{
    return connection;
}

static const char *name_error_string(int ret)
{
    switch (ret) {
        case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
            return "in queue";
        case DBUS_REQUEST_NAME_REPLY_EXISTS:
            return "name exists";
        case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
            return "already owner";
        default:
            return "";
    }
}

gboolean setup_dbus_connection(const char *name,
        void (*handler_init)(DBusConnection *connection))
{
    DBusError derr;

    g_assert(connection == NULL);

    dbus_error_init(&derr);
    connection = dbus_bus_get(DBUS_BUS_SYSTEM, &derr);
    if (connection == NULL) {
        error("System D-BUS connection failed: %s", derr.message);
        dbus_error_free(&derr);
        return FALSE;
    }

    dbus_connection_setup_with_g_main(connection, NULL);

    if (name) {
        int ret;

        dbus_error_init(&derr);
        ret = dbus_bus_request_name(connection, name, 0, &derr);
        if (ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
            error("Could not aquire D-BUS name '%s': %s (ret: %d)",
                   name, name_error_string(ret), ret);
            if (dbus_error_is_set(&derr)) {
                debug("%s", derr.message);
                dbus_error_free(&derr);
            }
            return FALSE;
        }
    }

    if (handler_init)
        handler_init(connection);

    return TRUE;
}

void close_dbus_connection(void)
{
    g_assert(connection != NULL);
    dbus_connection_unref(connection);
    connection = NULL;
}

