///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#include <sstream>
#include <cppunit/extensions/HelperMacros.h>
#include <openvdb/Exceptions.h>
#include <sys/types.h> // for stat()
#include <sys/stat.h>
#include <stdlib.h> // for system()
#include <tbb/task.h>
#include <openvdb/Types.h>
#include <openvdb/tree/LeafNode.h>
#include <openvdb/tree/Tree.h>
#include <openvdb/tree/ValueAccessor.h>
#include <openvdb/openvdb.h>

#define ASSERT_DOUBLES_EXACTLY_EQUAL(expected, actual) \
    CPPUNIT_ASSERT_DOUBLES_EQUAL((expected), (actual), /*tolerance=*/0.0);


typedef float ValueType;
typedef openvdb::tree::Tree4<ValueType, 5, 4, 3>::Type TreeType;


class TestValueAccessor: public CppUnit::TestFixture
{
public:
    virtual void setUp() { openvdb::initialize(); }
    virtual void tearDown() { openvdb::uninitialize(); }

    CPPUNIT_TEST_SUITE(TestValueAccessor);
    CPPUNIT_TEST(testValueAccessor);
    CPPUNIT_TEST(testValueAccessorRW);
    CPPUNIT_TEST(testConstValueAccessor);
    CPPUNIT_TEST(testConstValueAccessorRW);
    CPPUNIT_TEST(testMultithreadedAccessor);
    CPPUNIT_TEST(testAccessorRegistration);
    CPPUNIT_TEST(testGetNode);
    CPPUNIT_TEST_SUITE_END();

    void testValueAccessor();
    void testValueAccessorRW();
    void testConstValueAccessor();
    void testConstValueAccessorRW();
    void testMultithreadedAccessor();
    void testAccessorRegistration();
    void testGetNode();

private:
    template<typename AccessorT> void accessorTest();
    template<typename AccessorT> void constAccessorTest();
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestValueAccessor);


////////////////////////////////////////


void
TestValueAccessor::testValueAccessor()
{
    accessorTest<openvdb::tree::ValueAccessor<TreeType> >();
}

void
TestValueAccessor::testValueAccessorRW()
{
    accessorTest<openvdb::tree::ValueAccessorRW<TreeType> >();
}

void
TestValueAccessor::testConstValueAccessor()
{
    constAccessorTest<openvdb::tree::ValueAccessor<const TreeType> >();
}

void
TestValueAccessor::testConstValueAccessorRW()
{
    constAccessorTest<openvdb::tree::ValueAccessorRW<const TreeType> >();
}


////////////////////////////////////////


template<typename AccessorT>
void
TestValueAccessor::accessorTest()
{
    const ValueType background = 5.0f, value = -9.345f;
    const openvdb::Coord c0(5, 10, 20), c1(50000, 20000, 30000);

    {
        TreeType tree(background);
        CPPUNIT_ASSERT(!tree.isValueOn(c0));
        CPPUNIT_ASSERT(!tree.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c1));
        tree.setValue(c0, value);
        CPPUNIT_ASSERT(tree.isValueOn(c0));
        CPPUNIT_ASSERT(!tree.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c1));
    }
    {
        TreeType tree(background);
        AccessorT acc(tree);
        ValueType v;

        CPPUNIT_ASSERT(!tree.isValueOn(c0));
        CPPUNIT_ASSERT(!tree.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c0));
        CPPUNIT_ASSERT(!acc.isCached(c1));
        CPPUNIT_ASSERT(!acc.probeValue(c0,v));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
        CPPUNIT_ASSERT(!acc.probeValue(c1,v));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
        CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c0));
        CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c1));
        CPPUNIT_ASSERT(!acc.isVoxel(c0));
        CPPUNIT_ASSERT(!acc.isVoxel(c1));

        tree.setValue(c0, value);

        CPPUNIT_ASSERT(tree.isValueOn(c0));
        CPPUNIT_ASSERT(!tree.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c1));
        CPPUNIT_ASSERT(acc.probeValue(c0,v));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, v);
        CPPUNIT_ASSERT(!acc.probeValue(c1,v));
        ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));// leaf-level voxel value
        CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c1));// background value
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(openvdb::Coord(7, 10, 20)));
        CPPUNIT_ASSERT_EQUAL( 2, acc.getValueDepth(openvdb::Coord(8, 10, 20)));
        CPPUNIT_ASSERT( acc.isVoxel(c0));// leaf-level voxel value
        CPPUNIT_ASSERT(!acc.isVoxel(c1));
        CPPUNIT_ASSERT( acc.isVoxel(openvdb::Coord(7, 10, 20)));
        CPPUNIT_ASSERT(!acc.isVoxel(openvdb::Coord(8, 10, 20)));

        ASSERT_DOUBLES_EXACTLY_EQUAL(background, acc.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c1)); // uncached background value
        CPPUNIT_ASSERT(!acc.isValueOn(c1));// inactive background value
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c0));
        CPPUNIT_ASSERT(acc.isCached(c0)); // active, leaf-level voxel value
        CPPUNIT_ASSERT(acc.isValueOn(c0));

        acc.setValue(c1, value);

        CPPUNIT_ASSERT(acc.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c1));
        CPPUNIT_ASSERT(acc.isCached(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c0));
        CPPUNIT_ASSERT(acc.isCached(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c1));
        CPPUNIT_ASSERT(acc.isVoxel(c0));
        CPPUNIT_ASSERT(acc.isVoxel(c1));

        tree.setValueOff(c1);

        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c0));
        CPPUNIT_ASSERT( acc.isCached(c1));
        CPPUNIT_ASSERT( acc.isValueOn(c0));
        CPPUNIT_ASSERT(!acc.isValueOn(c1));

        acc.setValueOn(c1);

        CPPUNIT_ASSERT(!acc.isCached(c0));
        CPPUNIT_ASSERT( acc.isCached(c1));
        CPPUNIT_ASSERT( acc.isValueOn(c0));
        CPPUNIT_ASSERT( acc.isValueOn(c1));

        acc.setValueOnSum(c1, value);

        CPPUNIT_ASSERT(acc.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(2*value, tree.getValue(c1));
        CPPUNIT_ASSERT(acc.isCached(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(2*value, acc.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c0));
        CPPUNIT_ASSERT(acc.isCached(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c1));
        CPPUNIT_ASSERT(acc.isVoxel(c0));
        CPPUNIT_ASSERT(acc.isVoxel(c1));

        acc.setValueOnly(c1, 3*value);

        CPPUNIT_ASSERT(acc.isValueOn(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, tree.getValue(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(3*value, tree.getValue(c1));
        CPPUNIT_ASSERT(acc.isCached(c1));
        ASSERT_DOUBLES_EXACTLY_EQUAL(3*value, acc.getValue(c1));
        CPPUNIT_ASSERT(!acc.isCached(c0));
        ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c0));
        CPPUNIT_ASSERT(acc.isCached(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));
        CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c1));
        CPPUNIT_ASSERT(acc.isVoxel(c0));
        CPPUNIT_ASSERT(acc.isVoxel(c1));

        acc.clear();
        CPPUNIT_ASSERT(!acc.isCached(c0));
        CPPUNIT_ASSERT(!acc.isCached(c1));
    }
}


template<typename AccessorT>
void
TestValueAccessor::constAccessorTest()
{
    const ValueType background = 5.0f, value = -9.345f;
    const openvdb::Coord c0(5, 10, 20), c1(50000, 20000, 30000);
    ValueType v;

    TreeType tree(background);
    AccessorT acc(tree);

    CPPUNIT_ASSERT(!tree.isValueOn(c0));
    CPPUNIT_ASSERT(!tree.isValueOn(c1));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c0));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, tree.getValue(c1));
    CPPUNIT_ASSERT(!acc.isCached(c0));
    CPPUNIT_ASSERT(!acc.isCached(c1));
    CPPUNIT_ASSERT(!acc.probeValue(c0,v));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
    CPPUNIT_ASSERT(!acc.probeValue(c1,v));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
    CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c0));
    CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c1));
     CPPUNIT_ASSERT(!acc.isVoxel(c0));
     CPPUNIT_ASSERT(!acc.isVoxel(c1));

    tree.setValue(c0, value);

    CPPUNIT_ASSERT(tree.isValueOn(c0));
    CPPUNIT_ASSERT(!tree.isValueOn(c1));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, acc.getValue(c1));
    CPPUNIT_ASSERT(!acc.isCached(c1));
    CPPUNIT_ASSERT(!acc.isCached(c0));
    CPPUNIT_ASSERT(acc.isValueOn(c0));
    CPPUNIT_ASSERT(!acc.isValueOn(c1));
    CPPUNIT_ASSERT(acc.probeValue(c0,v));
    ASSERT_DOUBLES_EXACTLY_EQUAL(value, v);
    CPPUNIT_ASSERT(!acc.probeValue(c1,v));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, v);
    CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));
    CPPUNIT_ASSERT_EQUAL(-1, acc.getValueDepth(c1));
    CPPUNIT_ASSERT( acc.isVoxel(c0));
    CPPUNIT_ASSERT(!acc.isVoxel(c1));

    ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c0));
    CPPUNIT_ASSERT(acc.isCached(c0));
    ASSERT_DOUBLES_EXACTLY_EQUAL(background, acc.getValue(c1));
    CPPUNIT_ASSERT(acc.isCached(c0));
    CPPUNIT_ASSERT(!acc.isCached(c1));
    CPPUNIT_ASSERT(acc.isValueOn(c0));
    CPPUNIT_ASSERT(!acc.isValueOn(c1));

    tree.setValue(c1, value);

    ASSERT_DOUBLES_EXACTLY_EQUAL(value, acc.getValue(c1));
    CPPUNIT_ASSERT(!acc.isCached(c0));
    CPPUNIT_ASSERT(acc.isCached(c1));
    CPPUNIT_ASSERT(acc.isValueOn(c0));
    CPPUNIT_ASSERT(acc.isValueOn(c1));
    CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c0));
    CPPUNIT_ASSERT_EQUAL( 3, acc.getValueDepth(c1));
    CPPUNIT_ASSERT(acc.isVoxel(c0));
    CPPUNIT_ASSERT(acc.isVoxel(c1));

    // The next two lines should not compile, because the acc references a const tree:
    //acc.setValue(c1, value);
    //acc.setValueOff(c1);

    acc.clear();
    CPPUNIT_ASSERT(!acc.isCached(c0));
    CPPUNIT_ASSERT(!acc.isCached(c1));
}


void
TestValueAccessor::testMultithreadedAccessor()
{
#define MAX_COORD 5000

    typedef openvdb::tree::ValueAccessorRW<TreeType> AccessorT;
    // Substituting the following typedef typically results in assertion failures:
    //typedef openvdb::tree::ValueAccessor<TreeType> AccessorT;

    // Task to perform multiple reads through a shared accessor
    struct ReadTask: public tbb::task {
        AccessorT& acc;
        ReadTask(AccessorT& c): acc(c) {}
        tbb::task* execute()
        {
            for (int i = -MAX_COORD; i < MAX_COORD; ++i) {
                ASSERT_DOUBLES_EXACTLY_EQUAL(double(i), acc.getValue(openvdb::Coord(i)));
            }
            return NULL;
        }
    };
    // Task to perform multiple writes through a shared accessor
    struct WriteTask: public tbb::task {
        AccessorT& acc;
        WriteTask(AccessorT& c): acc(c) {}
        tbb::task* execute()
        {
            for (int i = -MAX_COORD; i < MAX_COORD; ++i) {
                float f = acc.getValue(openvdb::Coord(i));
                ASSERT_DOUBLES_EXACTLY_EQUAL(float(i), f);
                acc.setValue(openvdb::Coord(i), i);
                ASSERT_DOUBLES_EXACTLY_EQUAL(float(i), acc.getValue(openvdb::Coord(i)));
            }
            return NULL;
        }
    };
    // Parent task to spawn multiple parallel read and write tasks
    struct RootTask: public tbb::task {
        AccessorT& acc;
        RootTask(AccessorT& c): acc(c) {}
        tbb::task* execute()
        {
            ReadTask* r[3]; WriteTask* w[3];
            for (int i = 0; i < 3; ++i) {
                r[i] = new(allocate_child()) ReadTask(acc);
                w[i] = new(allocate_child()) WriteTask(acc);
            }
            set_ref_count(6 /*children*/ + 1 /*wait*/);
            for (int i = 0; i < 3; ++i) {
                spawn(*r[i]); spawn(*w[i]);
            }
            wait_for_all();
            return NULL;
        }
    };

    TreeType tree(/*background=*/0.5);
    AccessorT acc(tree);
    // Populate the tree.
    for (int i = -MAX_COORD; i < MAX_COORD; ++i) {
        acc.setValue(openvdb::Coord(i), i);
    }

    // Run multiple read and write tasks in parallel.
    RootTask& root = *new(tbb::task::allocate_root()) RootTask(acc);
    tbb::task::spawn_root_and_wait(root);

#undef MAX_COORD
}


void
TestValueAccessor::testAccessorRegistration()
{
    using openvdb::Index;

    const float background = 5.0f, value = -9.345f;
    const openvdb::Coord c0(5, 10, 20);

    openvdb::FloatTree::Ptr tree(new openvdb::FloatTree(background));
    openvdb::tree::ValueAccessor<openvdb::FloatTree> acc(*tree);

    // Set a single leaf voxel via the accessor and verify that
    // the cache is populated.
    acc.setValue(c0, value);
    CPPUNIT_ASSERT_EQUAL(Index(1), tree->leafCount());
    CPPUNIT_ASSERT_EQUAL(tree->getRootNode().getLevel(), tree->nonLeafCount());
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::LeafNodeType>() != NULL);

    // Reset the voxel to the background value and verify that no nodes
    // have been deleted and that the cache is still populated.
    tree->setValueOff(c0, background);
    CPPUNIT_ASSERT_EQUAL(Index(1), tree->leafCount());
    CPPUNIT_ASSERT_EQUAL(tree->getRootNode().getLevel(), tree->nonLeafCount());
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::LeafNodeType>() != NULL);

    // Prune the tree and verify that only the root node remains and that
    // the cache has been cleared.
    tree->prune();
    CPPUNIT_ASSERT_EQUAL(Index(0), tree->leafCount());
    CPPUNIT_ASSERT_EQUAL(Index(1), tree->nonLeafCount()); // root node only
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::LeafNodeType>() == NULL);

    // Set the leaf voxel again and verify that the cache is repopulated.
    acc.setValue(c0, value);
    CPPUNIT_ASSERT_EQUAL(Index(1), tree->leafCount());
    CPPUNIT_ASSERT_EQUAL(tree->getRootNode().getLevel(), tree->nonLeafCount());
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::LeafNodeType>() != NULL);

    // Delete the tree and verify that the cache has been cleared.
    tree.reset();
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::RootNodeType>() == NULL);
    CPPUNIT_ASSERT(acc.getNode<openvdb::FloatTree::LeafNodeType>() == NULL);
}


void
TestValueAccessor::testGetNode()
{
    typedef TreeType::LeafNodeType LeafT;

    const ValueType background = 5.0f, value = -9.345f;
    const openvdb::Coord c0(5, 10, 20);

    TreeType tree(background);
    tree.setValue(c0, value);
    {
        openvdb::tree::ValueAccessor<TreeType> acc(tree);
        // Prime the cache.
        acc.getValue(c0);
        // Verify that the cache contains a leaf node.
        LeafT* node = acc.getNode<LeafT>();
        CPPUNIT_ASSERT(node != NULL);

        // Erase the leaf node from the cache and verify that it is gone.
        acc.eraseNode<LeafT>();
        node = acc.getNode<LeafT>();
        CPPUNIT_ASSERT(node == NULL);
    }
    {
        // As above, but with a const tree.
        openvdb::tree::ValueAccessor<const TreeType> acc(tree);
        acc.getValue(c0);
        const LeafT* node = acc.getNode<const LeafT>();
        CPPUNIT_ASSERT(node != NULL);

        acc.eraseNode<LeafT>();
        node = acc.getNode<const LeafT>();
        CPPUNIT_ASSERT(node == NULL);
    }
}

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
