///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#include <cppunit/extensions/HelperMacros.h>
#include <openvdb/Exceptions.h>
#include <openvdb/openvdb.h>
#include <openvdb/math/Ray.h>
#include <openvdb/Types.h>
#include <openvdb/math/Transform.h>

#ifndef REL_EQ_TOL
#define REL_EQ_TOL(v0, v1, eps) (fabs(v0 - v1) < eps)
#endif

#ifndef SMALLVALUE
#define SMALLVALUE (1.e-6)
#endif

typedef float     Real;

class TestRay : public CppUnit::TestCase
{
public:
    CPPUNIT_TEST_SUITE(TestRay);
    //CPPUNIT_TEST(testInfinity);
    CPPUNIT_TEST(testRay);
    CPPUNIT_TEST(testRay2);
    CPPUNIT_TEST(testMerge);
    CPPUNIT_TEST_SUITE_END();

    //void testInfinity();
    void testRay();
    void testRay2();
    void testMerge();
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestRay);

/*
  the Ray class makes use of infinity=1/0 so we test for it
*/
/*
void
TestRay::testInfinity()
{
    const double one=1, zero = 0, infinity = one / zero;
    CPPUNIT_ASSERT_DOUBLES_EQUAL( infinity , infinity,0);//not a NAN
    CPPUNIT_ASSERT_DOUBLES_EQUAL( infinity , infinity+1,0);//not a NAN
    CPPUNIT_ASSERT_DOUBLES_EQUAL( infinity , infinity*10,0);//not a NAN
    CPPUNIT_ASSERT( zero <   infinity);
    CPPUNIT_ASSERT( zero >  -infinity);
    CPPUNIT_ASSERT_DOUBLES_EQUAL( zero ,  one/infinity,0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL( zero , -one/infinity,0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL( infinity  ,  one/zero,0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(-infinity  , -one/zero,0);

    //std::cerr << "inf:        "   << infinity << "\n";
    //std::cerr << "1 / inf:    "   << one / infinity << "\n";
    //std::cerr << "1 / (-inf): "   << one / (-infinity) << "\n";
    //std::cerr << " inf * 0:   "   << infinity * 0 << "\n";
    //std::cerr << "-inf * 0:   "   << (-infinity) * 0 << "\n";
    //std::cerr << "(inf):      "   << (bool)(infinity) << "\n";
    //std::cerr << "inf == inf: "   << (infinity == infinity) << "\n";
    //std::cerr << "inf > 0:    "   << (infinity > 0) << "\n";
    //std::cerr << "-inf > 0:   "   << ((-infinity) > 0) << "\n";

}
*/
void TestRay::testRay()
{
    using namespace openvdb;

    math::Transform::Ptr xform = math::Transform::createLinearTransform();
    
    xform->rotate(M_PI, math::Y_AXIS );
    xform->translate(math::Vec3d(1, 2, 3));
    xform->scale(math::Vec3d(0.1, 2.2, 0.3));

    typedef openvdb::math::Ray<double>  RayType;
    openvdb::math::Vec3<double>  orig(9,1,1), dir(1,0,0);

    RayType ray0(orig,dir);
    RayType ray1( ray0 );

    ray1.worldToIndex( xform );
    ray1.indexToWorld( xform );

    const openvdb::math::Vec3<double> orig1 = ray1.getOrigin();
    CPPUNIT_ASSERT( REL_EQ_TOL( orig[0], orig1[0], SMALLVALUE ) );
    CPPUNIT_ASSERT( REL_EQ_TOL( orig[1], orig1[1], SMALLVALUE ) );
    CPPUNIT_ASSERT( REL_EQ_TOL( orig[2], orig1[2], SMALLVALUE ) );

    const openvdb::math::Vec3<double> dir1 = ray1.getDirection();
    
    CPPUNIT_ASSERT( REL_EQ_TOL( dir[0], dir1[0], SMALLVALUE ) );
    CPPUNIT_ASSERT( REL_EQ_TOL( dir[1], dir1[1], SMALLVALUE ) );
    CPPUNIT_ASSERT( REL_EQ_TOL( dir[2], dir1[2], SMALLVALUE ) );
}

void TestRay::testRay2()
{
    typedef openvdb::math::Ray<float>  RayType;
    RayType currRay;
    //RayType::Vec3R start(1.5,1.5,1.5);// works
    //RayType::Vec3R dir(1.5,1.5,1.5);// works
    //openvdb::Vec3f start(1.5,1.5,1.5);// works
    //openvdb::Vec3f dir(1.5,1.5,1.5);// works
    openvdb::math::Vec3s start(1.5,1.5,1.5);// works
    openvdb::math::Vec3s dir(1.5,1.5,1.5);// works
    //openvdb::math::Vec3f start(1.5,1.5,1.5);//does NOT work
    //openvdb::math::Vec3f dir(1.5,1.5,1.5);//does NOT work
    currRay.setOrigin(start);
    currRay.setDirection(dir);
    CPPUNIT_ASSERT(currRay.getOrigin()==start);
    CPPUNIT_ASSERT(currRay.getDirection()==dir);
}

void TestRay::testMerge()
{
    openvdb::Vec3R origin(0,5,0), normal(0,1,0);
    openvdb::Real tmin(0), tmax(50), value=5.0;
    openvdb::math::Ray<openvdb::Real>
        ray(origin,normal,std::numeric_limits<double>::epsilon(),tmin,tmax);
    typedef openvdb::math::RayIntervalSet<openvdb::Real,float> IntervalSetType;
    typedef openvdb::math::RayInterval<openvdb::Real,float> IntervalType;
    IntervalSetType intervals;
    CPPUNIT_ASSERT(intervals.size()==0);
    intervals.insert(IntervalType(0.300000,0.599758,value));
    intervals.insert(IntervalType(0.599758,1.757490,IntervalType::Hit()));
    intervals.insert(IntervalType(1.757490,2.912980,IntervalType::Hit()));
    intervals.insert(IntervalType(2.912980,4.066220,IntervalType::Hit()));
    intervals.insert(IntervalType(4.066220,5.027510,IntervalType::Hit()));
    CPPUNIT_ASSERT(intervals.size()==5);
    //intervals.print();
    intervals.merge(ray);
    //intervals.print();
    CPPUNIT_ASSERT(intervals.size()==4);
    IntervalSetType::const_iterator iter=intervals.begin();
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t0,tmin,/*tolorance=*/0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t1,0.3,/*tolorance=*/0);
    CPPUNIT_ASSERT( iter->isEmpty());
    CPPUNIT_ASSERT(!iter->isConstant());
    CPPUNIT_ASSERT(!iter->isVariable());
    ++iter;
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t0,0.3,/*tolorance=*/0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t1,0.599758,/*tolorance=*/0);
    CPPUNIT_ASSERT(!iter->isEmpty());
    CPPUNIT_ASSERT( iter->isConstant());
    CPPUNIT_ASSERT(!iter->isVariable());
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->value,value,/*tolorance=*/0);
    ++iter;
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t0,0.599758,/*tolorance=*/0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t1,5.027510,/*tolorance=*/0);
    CPPUNIT_ASSERT(!iter->isEmpty());
    CPPUNIT_ASSERT(!iter->isConstant());
    CPPUNIT_ASSERT( iter->isVariable());
    ++iter;
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t0,5.027510,/*tolorance=*/0);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(iter->t1,tmax,/*tolorance=*/0);
    CPPUNIT_ASSERT( iter->isEmpty());
    CPPUNIT_ASSERT(!iter->isConstant());
    CPPUNIT_ASSERT(!iter->isVariable());
}

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
