///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#include <cppunit/extensions/HelperMacros.h>
#include <openvdb/Exceptions.h>
#include <openvdb/math/Plane.h>
#include <openvdb/Types.h>

#ifndef REL_EQ_TOL
#define REL_EQ_TOL(v0, v1, eps) (fabs(v0 - v1) < eps)
#endif

#ifndef SMALLVALUE
#define SMALLVALUE (1.e-6)
#endif

typedef float     Real;

class TestPlane : public CppUnit::TestCase
{
public:
    CPPUNIT_TEST_SUITE(TestPlane);
    CPPUNIT_TEST(testConstructors);
    CPPUNIT_TEST(testBasic);
    CPPUNIT_TEST(testLerp);
    CPPUNIT_TEST(testSlerp);
    CPPUNIT_TEST_SUITE_END();

    void testConstructors();
    void testBasic();
    void testLerp();
    void testSlerp();
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestPlane);

void
TestPlane::testConstructors()
{
    typedef openvdb::Vec3R                      Vec3R;
    typedef openvdb::math::Ray<openvdb::Real>   RayType;
    typedef openvdb::math::Plane<openvdb::Real> PlaneType;

    {//default is in-valid
        PlaneType P;
        CPPUNIT_ASSERT(!P.isValid());
    }

    {//From point and normal - no norm
        PlaneType P(Vec3R(0.5,0.6,0.7), Vec3R(1,2,3));
        CPPUNIT_ASSERT(P.isValid());
        CPPUNIT_ASSERT(!REL_EQ_TOL(P.getNorm().length(),1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[0],1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[1],2,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[2],3,SMALLVALUE));
    }

    {//From point and normal - norm
        PlaneType P(Vec3R(0.5,0.6,0.7), Vec3R(1,2,3),PlaneType::Normalize());
        CPPUNIT_ASSERT(P.isValid());
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm().length(),1,SMALLVALUE));
    }

    {//From point and normal - no norm
        PlaneType P(Vec3R(0.5,0.6,0.7), Vec3R(1,0,0));
        CPPUNIT_ASSERT(P.isValid());
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm().length(),1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[0],1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[1],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[2],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[0],0.5,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[1],0.6,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[2],0.7,SMALLVALUE));
    }

    {//Interpolation between two points
        PlaneType P(Vec3R(1,1,0),Vec3R(2,1,0),0.25);
        CPPUNIT_ASSERT(P.isValid());
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm().length(),1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[0],1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[1],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[2],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[0],1.25,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[1],1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[2],0,SMALLVALUE));

    }

     {//3 points in plane forming a right-hand rotation
        const Vec3R p0(0,0,1), p1(1,0,1), p2(0,1,1);
        PlaneType P(p0,p1,p2);
        CPPUNIT_ASSERT(P.isValid());
        //std::cerr << P << std::endl;
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[0],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[1],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getNorm()[2],1,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[0],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[1],0,SMALLVALUE));
        CPPUNIT_ASSERT(REL_EQ_TOL(P.getPoint()[2],1,SMALLVALUE));
    }
}

void
TestPlane::testBasic()
{
    typedef openvdb::Vec3R                      Vec3R;
    typedef openvdb::math::Ray<openvdb::Real>   RayType;
    typedef openvdb::math::Plane<openvdb::Real> PlaneType;

    const Vec3R pos(1,2,3), norm(1,0,0);
    PlaneType P1(pos,norm), P2(P1);

    CPPUNIT_ASSERT(P1.isValid());
    CPPUNIT_ASSERT(P2.isValid());
    CPPUNIT_ASSERT(P1==P2);

    CPPUNIT_ASSERT( P1.isInside(Vec3R(0,0,0)));
    CPPUNIT_ASSERT( P1.isOn(Vec3R(1,0,0)));
    CPPUNIT_ASSERT( P1.isOutside(Vec3R(2,0,0)));

    P2.flip();
    CPPUNIT_ASSERT(P1.isValid());
    CPPUNIT_ASSERT(P2.isValid());
    CPPUNIT_ASSERT(P1!=P2);

    CPPUNIT_ASSERT( P2.isOutside(Vec3R(0,0,0)));
    CPPUNIT_ASSERT( P2.isOn(Vec3R(1,0,0)));
    CPPUNIT_ASSERT( P2.isInside(Vec3R(2,0,0)));

    const Vec3R orig(0,0,0), dir(1,0,0);
    RayType R(orig,dir);
    openvdb::Real t = 0.0;
    CPPUNIT_ASSERT(P1.intersects(R));
    CPPUNIT_ASSERT(P2.intersects(R));
    CPPUNIT_ASSERT(P1.intersects(R,t));
    CPPUNIT_ASSERT(REL_EQ_TOL(t,1,SMALLVALUE));
    CPPUNIT_ASSERT(P2.intersects(R,t));
    CPPUNIT_ASSERT(REL_EQ_TOL(t,1,SMALLVALUE));



    /*

    {// Test from frustum
        PlaneType P2(8,Vec3R(-1,0,0));//left
        CPPUNIT_ASSERT(P2.isInside(Vec3R(9,1,1)));
        CPPUNIT_ASSERT(!P2.isOutside(Vec3R(9,1,1)));
        PlaneType P3(16,Vec3R(1,0,0));//right
        CPPUNIT_ASSERT(P3.isInside(Vec3R(9,1,1)));
        CPPUNIT_ASSERT(!P3.isOutside(Vec3R(9,1,1)));
        PlaneType P4(16,Vec3R(0,1,0));//up
        CPPUNIT_ASSERT(P4.isInside(Vec3R(9,1,1)));
        CPPUNIT_ASSERT(!P4.isOutside(Vec3R(9,1,1)));
        PlaneType P5(8,Vec3R(0,-1,0));//down
        CPPUNIT_ASSERT(!P5.isInside(Vec3R(9,1,1)));
        CPPUNIT_ASSERT(P5.isOutside(Vec3R(9,1,1)));
        std::cerr << "Passed Plane tests" << std::endl;
    }
    */
    PlaneType P3(Vec3R(1,1,1),norm);
    //std::cerr << P3 << std::endl;
    CPPUNIT_ASSERT( P3.isValid());
    CPPUNIT_ASSERT( P3.isInside(Vec3R(0.5,0.5,0.5)));
    CPPUNIT_ASSERT(!P3.isOutside(Vec3R(0.5,0.5,0.5)));
    CPPUNIT_ASSERT(!P3.isInside(Vec3R(1.5,0.5,0.5)));
    CPPUNIT_ASSERT( P3.isOutside(Vec3R(1.5,0.5,0.5)));

    PlaneType P4(Vec3R(1,1,1),-norm);
    //std::cerr << P4 << std::endl;
    CPPUNIT_ASSERT( P4.isValid());
    CPPUNIT_ASSERT(!P4.isInside(Vec3R(0.5,0.5,0.5)));
    CPPUNIT_ASSERT( P4.isOutside(Vec3R(0.5,0.5,0.5)));
    CPPUNIT_ASSERT( P4.isInside(Vec3R(1.5,0.5,0.5)));
    CPPUNIT_ASSERT(!P4.isOutside(Vec3R(1.5,0.5,0.5)));

    PlaneType P5(Vec3R(-1,-1,-1),norm);
    //std::cerr << P5 << std::endl;
    CPPUNIT_ASSERT( P5.isValid());
    CPPUNIT_ASSERT(!P5.isInside(Vec3R(0.5,0.5,0.5)));
    CPPUNIT_ASSERT(!P5.isInside(Vec3R(1.5,0.5,0.5)));
    CPPUNIT_ASSERT(!P5.isInside(Vec3R(-0.5,-0.5,-0.5)));
    CPPUNIT_ASSERT( P5.isInside(Vec3R(-1.5,-0.5,-0.5)));
    CPPUNIT_ASSERT(!P5.intersects(R));
    CPPUNIT_ASSERT(!P5.intersects(R,t));
    CPPUNIT_ASSERT(REL_EQ_TOL(t,-1,SMALLVALUE));
}

void
TestPlane::testLerp()
{
    typedef openvdb::Vec3R                      Vec3R;
    typedef openvdb::math::Ray<openvdb::Real>   RayType;
    typedef openvdb::math::Plane<openvdb::Real> PlaneType;

    const Vec3R norm1(-1,1,0), norm2(1,1,0);

    // Note that P is repsectively (-0.5,0.5,0) and (0.5,0.5,0)
    //PlaneType P1(Vec3R(-1,0,0),norm1,PlaneType::Normalize());
    //PlaneType P2(Vec3R( 1,0,0),norm2,PlaneType::Normalize());
    PlaneType P1(Vec3R(-0.5,0.5,0),norm1,PlaneType::Normalize());
    PlaneType P2(Vec3R( 0.5,0.5,0),norm2,PlaneType::Normalize());

    //std::cerr << P1 << std::endl;
    CPPUNIT_ASSERT(P1.isValid());
    //std::cerr << P2 << std::endl;
    CPPUNIT_ASSERT(P2.isValid());
    PlaneType P3 = P1.lerp(P2,0.5f);
    //std::cerr << "testLerp: " << P3 << std::endl;
    // midway has N=(0,1,0) and P=(0,0.5,0)
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[0],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[1],1,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[2],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[0],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[1],0.5,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[2],0,SMALLVALUE));
}

void
TestPlane::testSlerp()
{
    typedef openvdb::Vec3R                      Vec3R;
    typedef openvdb::math::Ray<openvdb::Real>   RayType;
    typedef openvdb::math::Plane<openvdb::Real> PlaneType;

    const Vec3R norm1(-1,1,0), norm2(1,1,0);

    // Note that P is repsectively (-0.5,0.5,0) and (0.5,0.5,0)
    //PlaneType P1(Vec3R(-1,0,0),norm1,PlaneType::Normalize());
    //PlaneType P2(Vec3R( 1,0,0),norm2,PlaneType::Normalize());
    PlaneType P1(Vec3R(-0.5,0.5,0),norm1,PlaneType::Normalize());
    PlaneType P2(Vec3R( 0.5,0.5,0),norm2,PlaneType::Normalize());

    //std::cerr << P1 << std::endl;
    CPPUNIT_ASSERT(P1.isValid());
    //std::cerr << P2 << std::endl;
    CPPUNIT_ASSERT(P2.isValid());
    PlaneType P3 = P1.slerp(P2,0.5f);
    //std::cerr << "testSlerp: " << P3 << std::endl;
    // midway has N=(0,1,0) and P=(0,0.5,0)
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[0],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[1],1,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getNorm()[2],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[0],0,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[1],0.5,SMALLVALUE));
    CPPUNIT_ASSERT(REL_EQ_TOL(P3.getPoint()[2],0,SMALLVALUE));
}

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
