///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
///
/// @author Ken Museth
///
/// @file LevelSetSphere.h
///
/// @brief Doh, generates a narrow band level set of sphere
///
#ifndef OPENVDB_TOOLS_SPHERETOLEVELSET_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_SPHERETOLEVELSET_HAS_BEEN_INCLUDED

#include <openvdb/openvdb.h>
#include <openvdb/util/NullInterrupter.h>

namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {


/// @brief @return a FloatGrid containing a narrow-band level set of a sphere
///
/// @param radius  radius of the sphere in world units
/// @param center  center of the sphere in world units
/// @param dx      Size of voxels in world units
/// @param w       Half-width of narrow-band in voxel units
/// @param interrupt pointer to optional interrupter. Use template
/// argument util::NullInterrupter if no interruption is desired.    
///
/// @note the leapfrog algorithm employed in this method is best
/// suited for a single large sphere. For multiple small spheres consider
/// using the faster algorithm in ParticlesToLevelSet.h
template<typename InterruptType>
FloatGrid::Ptr makeSphere(float radius,//sphere radius in world units
                          const openvdb::Vec3f &center,//sphere center in world units
                          float dx,// voxel size in world units
                          float w,// half-width of narrow-band in voxel units
                          InterruptType* interrupt = NULL)
{
    if (!(radius>0.0f)) OPENVDB_THROW(ValueError, "radius must be positive");
    if (!(dx>0.0f)) OPENVDB_THROW(ValueError, "voxel size must be positive");
    if (!(w>1)) OPENVDB_THROW(ValueError, "half-width must be larger than one");

    // Define an empty grid using the voxel size and narrow-band half-width.
    FloatGrid::Ptr grid = FloatGrid::create(dx * w);
    grid->setTransform(math::Transform::createLinearTransform(dx));
    grid->setName("LevelSetSphere");
    grid->setGridClass(GRID_LEVEL_SET);

    // Define radius of sphere and narrow-band in voxel units
    const float r0 = radius/dx, rmax = r0 + w;

    // Define center of sphere in voxel units
    const openvdb::Vec3f c(center[0]/dx, center[1]/dx, center[2]/dx);
    
    // Radius below the Nyquist frequency
    if (r0 < 1.5f)  return grid;

    // Define index coordinates and their respective bounds
    openvdb::Coord ijk;
    int &i = ijk[0], &j = ijk[1], &k = ijk[2], m=1;
    const int imin=math::Floor(c[0]-rmax), imax=math::Ceil(c[0]+rmax);
    const int jmin=math::Floor(c[1]-rmax), jmax=math::Ceil(c[1]+rmax);
    const int kmin=math::Floor(c[2]-rmax), kmax=math::Ceil(c[2]+rmax);

    // Allocate an ValueAccessor for accelerated random access
    FloatGrid::Accessor accessor = grid->getAccessor();
    
    if (interrupt) interrupt->start("Generating level set of sphere");
    // Compute signed distances to sphere using leapfrogging in k
    for ( i = imin; i <= imax; ++i ) {
        if (interrupt && interrupt->wasInterrupted()) return grid;
        const float x2 = math::Pow2(i - c[0]);
        for ( j = jmin; j <= jmax; ++j ) {
            const float x2y2 = math::Pow2(j - c[1]) + x2;
            for (k=kmin; k<=kmax; k += m) {
                m = 1;
                /// Distance in voxel units to sphere
                const float v = math::Sqrt(x2y2 + math::Pow2(k-c[2]))-r0,
                            d = math::Abs(v);
                if ( d < w )// inside narrow band
                    accessor.setValue(ijk, dx*v);// distance in world units
                else// outside narrow band
                    m += math::Floor(d-w);// leapfrog
            }//end leapfrog over k
        }//end loop over j
    }//end loop over i

    // Define consistant signed distances outside the narrow-band
    grid->signedFloodFill();
    if (interrupt) interrupt->end();
    return grid;
}    

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_SPHERETOLEVELSET_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
