///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_TOOLS_GRIDSAMPLING_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_GRIDSAMPLING_HAS_BEEN_INCLUDED

#include <iostream>
#include <openvdb/Types.h>
#include <openvdb/math/Transform.h>
#include <openvdb/tree/ValueAccessor.h>
#include <openvdb/tools/SpaceSampling.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Base sampling class that provides the interface for sampling values
/// in a grid continuously.
/// Since grids can only be used for discrete voxel sampling, a GridSampling
/// must be used to sample arbitrary continuous points in space.
template<typename TreeType>
class GridSampling
{
public:
    typedef boost::shared_ptr<GridSampling<TreeType> >    Ptr;
    typedef typename TreeType::ValueType                  ValueType;
    typedef typename tree::ValueAccessor<const TreeType>  ConstAccessor;

    /// Constructor
    explicit GridSampling(const TreeType& tree): mTree(tree) {}

    /// Destructor
    virtual ~GridSampling() {}

    /// Sample a point in voxel space in the grid.
    /// @param pt the point in voxel-coordinates in the grid to sample.
    ValueType sampleVoxel(const Vec3R& pt) const
        { return sampleVoxel(pt[0], pt[1], pt[2]); }

    /// Sample a point in voxel space in the grid.
    /// @param x x-coordinate of point in voxel-coordinates of grid
    /// @param y y-coordinate of point in voxel-coordinates of grid
    /// @param z z-coordinate of point in voxel-coordinates of grid
    virtual ValueType
    sampleVoxel(Real x, Real y, Real z) const = 0;

protected:
    // The grid we are sampling.
    const TreeType& mTree;

private:
    // Disallow copying of instances of this class.
    GridSampling(const GridSampling& other);
    GridSampling& operator=(const GridSampling& other);
};


/// @brief Base sampling class that provides the interface for sampling values
/// in a grid continuously.
/// Since grids can only be used for discrete voxel sampling, a GridSampler
/// must be used to sample arbitrary continuous points in (world or index) space.
template<typename TreeOrAccessorType, typename SamplerType>
class GridSampler
{
public:
    typedef boost::shared_ptr<GridSampler<TreeOrAccessorType, SamplerType> >  Ptr;
    typedef typename TreeOrAccessorType::ValueType                            ValueType;


    /// @param tree  a tree to be sampled, or a ValueAccessor for the tree
    /// @param transform is used when sampling world space locations.
    ///     (by default an identity transform is used)
    GridSampler(const TreeOrAccessorType& tree,
        const math::Transform& transform = math::Transform()):
        mTree(tree), mTransform(transform) {}

    ~GridSampler() {};

    /// Sample a point in index space in the grid.
    /// @param x x-coordinate of point in index-coordinates of grid
    /// @param y y-coordinate of point in index-coordinates of grid
    /// @param z z-coordinate of point in index-coordinates of grid
    template <typename RealType>
    ValueType sampleVoxel( const RealType& x, const RealType& y, const RealType& z) const {
        return isSample(Vec3d(x,y,z));
    }

    /// Sample in index space @param ispoint is the location in index space
    ValueType isSample( const Vec3d& ispoint) const {
        ValueType result = zeroVal<ValueType>();
        SamplerType::sample(mTree, ispoint, result);
        return result;
    }

    /// Sample in world space @param wspoint is the location in world space
    ValueType wsSample( const Vec3d& wspoint) const {
        ValueType result = zeroVal<ValueType>();
        SamplerType::sample(mTree, mTransform.worldToIndex(wspoint), result);
        return result;
    }

private:
    // Disallow copying:  The member data could be a Accessor which can not
    // be shared by multiple threads
    GridSampler(const GridSampler& other);
    GridSampler& operator=(const GridSampler& other);

    const TreeOrAccessorType& mTree;
    const math::Transform&    mTransform;
};


} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_GRIDSAMPLING_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
