///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file Cpt.h

#ifndef OPENVDB_TOOLS_CPT_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_CPT_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Operators.h>
#include <tbb/parallel_reduce.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Computes the Closest-Point-Transform (CPT) from a distance field
///
/// @note The current implementation assumes all the input distance
/// values are represented by leaf voxels and not tiles. This is true
/// for all narrow-band level sets which this class was originally
/// developed for. In the future we will expand this class to also
/// handle tile values.
template<typename InGridType, typename MapType, math::DScheme Scheme>
class CptImpl
{
public:
    typedef typename InGridType::TreeType       InTreeType;
    typedef typename InGridType::ValueType      Real;
    typedef math::Vec3<Real>                    Vec3Type;
    typedef typename InTreeType::LeafNodeType   LeafType;
    typedef typename tree::IteratorRange<typename InTreeType::LeafCIter> RangeType;
    typedef typename InTreeType::template ValueConverter<Vec3Type>::Type OutTreeType;
    typedef Grid<OutTreeType>                   OutGridType;

    CptImpl(const InGridType& grid, const MapType& map):
        mInputGrid(&grid),
        mOutputGrid(OutGridType::create(Vec3Type(0, 0, 0))),
        mMap(&map)
    {
    }

    /// Copy constructor called by tbb::parallel_reduce threads
    CptImpl(const CptImpl& other, tbb::split):
        mInputGrid(other.mInputGrid),
        mOutputGrid(OutGridType::create(Vec3Type(0, 0, 0))), // new local grid
        mMap(other.mMap),
        mUseWorldTransform(other.mUseWorldTransform)
    {
    }

    ~CptImpl() {}

    typename OutGridType::Ptr process(bool threaded = true, bool useWorldTransform = true)
    {
        mUseWorldTransform = useWorldTransform;

        RangeType range(mInputGrid->tree().cbeginLeaf());
        if (threaded) {
            tbb::parallel_reduce(range, *this);
        } else {
            (*this)(range);
        }
        return mOutputGrid;
    }

    void operator()(RangeType& range)
    {
        const bool useWorldTransform = mUseWorldTransform;
        typename InGridType::ConstAccessor inAccessor = mInputGrid->getConstAccessor();
        typename OutGridType::Accessor outAccessor = mOutputGrid->getAccessor();


        if (useWorldTransform) { // the results are recorded in world space
            for ( ; range; ++range) {
                for (typename LeafType::ValueOnCIter v = range.iterator()->beginValueOn(); v; ++v) {
                    const Coord ijk = v.getCoord();
                    Vec3Type cpt = math::CPT_RANGE<MapType, Scheme>::result(*mMap, inAccessor, ijk);
                    outAccessor.setValue(ijk, cpt);

                }
            }
        } else { // the results are recorded in index space
            for ( ; range; ++range) {
                for (typename LeafType::ValueOnCIter v = range.iterator()->beginValueOn(); v; ++v) {
                    const Coord ijk = v.getCoord();
                    Vec3Type cpt = math::CPT<MapType, Scheme>::result(*mMap, inAccessor, ijk);
                    outAccessor.setValue(ijk, cpt);
                }
            }
        }
    }

    void join(const CptImpl& other) { mOutputGrid->merge(*other.mOutputGrid); }

protected:
    const InGridType*           mInputGrid;
    typename OutGridType::Ptr   mOutputGrid;
    const MapType*              mMap;
    bool                        mUseWorldTransform;
}; // end of CptImp class


template<typename InGridT>
class CptFunctor
{
public:
    typedef InGridT                                                      InGridType;
    typedef typename InGridT::ValueType                                  Real;
    typedef typename InGridT::TreeType                                   InTreeType;
    typedef math::Vec3<Real>                                             Vec3Type;
    typedef typename InTreeType::template ValueConverter<Vec3Type>::Type OutTreeType;
    typedef Grid<OutTreeType>                                            OutGridType;

    CptFunctor(const InGridType& grid, bool threaded, bool worldspace):
        mInputGrid(&grid), mThreaded(threaded), mWorldSpace(worldspace)
    {
    }

    template<typename MapType>
    void operator()(const MapType& map)
    {
        CptImpl<InGridType, MapType, math::CD_2ND> cptImpl(*mInputGrid, map);
        mOutputGrid = cptImpl.process(mThreaded, mWorldSpace); // cache the result
    }

    // for output of the result
    typename OutGridType::Ptr result() { return mOutputGrid; }

private:
    const InGridType*           mInputGrid;
    typename OutGridType::Ptr   mOutputGrid;
    bool                        mThreaded;
    bool                        mWorldSpace;
};


/// @brief Computes the Closest-Point-Transform (CPT) from a distance field
///
/// @note The current implementation assumes all the input distance
/// values are represented by leaf voxels and not tiles. This is true
/// for all narrow-band level sets which this class was originally
/// developed for. In the future we will expand this class to also
/// handle tile values.
template<typename InGridType>
class Cpt
{
public:
    typedef typename InGridType::TreeType   InTreeType;
    typedef typename InGridType::ValueType  Real;
    typedef math::Vec3<Real>                Vec3Type;
    typedef typename InTreeType::template ValueConverter<Vec3Type>::Type OutTreeType;
    typedef Grid<OutTreeType>               OutGridType;

    Cpt(const InGridType& grid): mInputGrid(&grid) {}
    ~Cpt() {}

    typename OutGridType::Ptr process(bool threaded = true, bool useWorldTransform = true)
    {
        CptFunctor<InGridType> cptFunctor(*mInputGrid, threaded, useWorldTransform);
        processTypedMap(mInputGrid->transform(), cptFunctor);
        return cptFunctor.result();
    }

protected:
    const InGridType* mInputGrid;
}; // end of Cpt class

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_CPT_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
