//                                               -*- C++ -*-
/**
 *  @file  ClaytonCopulaFactory.cxx
 *  @brief Factory for ClaytonCopula distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu, 10 mai 2007) $
 *  Id:      $Id: ClaytonCopulaFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <cmath>
#include "ClaytonCopulaFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
ClaytonCopulaFactory::ClaytonCopulaFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
ClaytonCopulaFactory * ClaytonCopulaFactory::clone() const
{
  return new ClaytonCopulaFactory(*this);
}

/* Here is the interface that all derived class must implement */

ClaytonCopula * ClaytonCopulaFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a ClaytonCopula distribution from an empty sample";
  if (sample.getDimension() != 2) throw InvalidArgumentException(HERE) << "Error: cannot build a ClaytonCopula distribution from a sample of dimension not equal to 2";
  NumericalScalar tau(sample.computeKendallTau().operator()(0, 1));
  if (tau == 1) throw InvalidArgumentException(HERE) << "Error: cannot build a ClaytonCopula distribution from a sample with Kendall tau equals to 1";
  return ClaytonCopula(2.0 * tau / (1.0 - tau)).clone();
}

ClaytonCopula * ClaytonCopulaFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    ClaytonCopula copula;
    copula.setParametersCollection(parameters);
    return copula.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a ClaytonCopula from the given parameters";
    }
}

ClaytonCopula * ClaytonCopulaFactory::build() const
{
  return ClaytonCopula().clone();
}


END_NAMESPACE_OPENTURNS
