//                                               -*- C++ -*-
/**
 *  @file  SimulationSensitivityAnalysis.hxx
 *  @brief Implementation of SimulationResult
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-23 13:46:12 +0200 (Fri, 23 May 2008) $
 *  Id:      $Id: SimulationSensitivityAnalysis.hxx 821 2008-05-23 11:46:12Z schueller $
 */
#ifndef OPENTURNS_SIMULATIONSENSITIVITYANALYSIS_HXX
#define OPENTURNS_SIMULATIONSENSITIVITYANALYSIS_HXX

#include <cmath>
#include "PersistentObject.hxx"
#include "NumericalSample.hxx"
#include "NumericalPointWithDescription.hxx"
#include "ComparisonOperator.hxx"
#include "NumericalMathFunction.hxx"
#include "Graph.hxx"
#include "Event.hxx"

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class SimulationSensitivityAnalysis
 * Implementation of Simulation sensitivity analysis, it means computation of standardized importance factors
 */
class SimulationSensitivityAnalysis
  : public PersistentObject
{

  CLASSNAME;
public:

  typedef NumericalMathFunction IsoProbabilisticTransformation;

  /** Default constructor */
  SimulationSensitivityAnalysis();

  /** Standard constructor */
  SimulationSensitivityAnalysis(const NumericalSample & inputSample,
                                const NumericalSample & outputSample,
                                const IsoProbabilisticTransformation & transformation,
                                const ComparisonOperator & comparisonOperator,
                                const NumericalScalar threshold);

  /** Standard constructor */
  SimulationSensitivityAnalysis(const Event & event);

  /** Virtual constructor */
  virtual SimulationSensitivityAnalysis * clone() const;

  /** Mean point in event domain computation */
  NumericalPoint computeMeanPointInEventDomain() const;
  NumericalPoint computeMeanPointInEventDomain(const NumericalScalar threshold) const;

  /** Importance factors computation */
  NumericalPointWithDescription computeImportanceFactors() const;
  NumericalPointWithDescription computeImportanceFactors(const NumericalScalar threshold) const;

public:

  /** Importance factors drawing */
  Graph drawImportanceFactors() const;
  Graph drawImportanceFactorsRange(const Bool probabilityScale = true) const;

  /** Input sample accessors */
  NumericalSample getInputSample() const;

  /** Output sample accessors */
  NumericalSample getOutputSample() const;

  /** Threshold accessors */
  NumericalScalar getThreshold() const;
  void setThreshold(const NumericalScalar threshold);

  /** Iso-probabilistic transformation accessor */
  IsoProbabilisticTransformation getTransformation() const;

  /** Comparison operator accessors */
  ComparisonOperator getComparisonOperator() const;
  void setComparisonOperator(const ComparisonOperator & comparisonOperator);

  /** String converter */
  virtual String __repr__() const;

  /** Method save() stores the object through the StorageManager */
  virtual void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  virtual void load(Advocate & adv);

protected:

  /* Input sample of the model */
  NumericalSample inputSample_;

  /* Associated output sample */
  NumericalSample outputSample_;

  /* Iso probabilistic transformation associated with the input sample distribution */
  IsoProbabilisticTransformation transformation_;

  /* Comparison operator defining the event we are interested in */
  ComparisonOperator comparisonOperator_;

  /* Main threshold of interest */
  NumericalScalar threshold_;

}; // class SimulationSensitivityAnalysis

END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_SIMULATIONSENSITIVITYANALYSIS_HXX */
