//                                               -*- C++ -*-
/**
 *  @file  HaltonSequence.cxx
 *  @brief Implementation of the Halton' sequence
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#include <cstdlib>

#include "HaltonSequence.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(HaltonSequence);


/* Constructor with parameters */
HaltonSequence::HaltonSequence(const UnsignedLong dimension) :
  LowDiscrepancySequenceImplementation(dimension)
{
  initialize(dimension);
}


/* Virtual constructor */
HaltonSequence * HaltonSequence::clone() const
{
  return new HaltonSequence(*this);
}


/* Initialize the sequence */
void HaltonSequence::initialize(const UnsignedLong dimension)
{
  if (dimension == 0) throw InvalidArgumentException(HERE) << "Dimension must be > 0.";
  dimension_ = dimension;
  base_ = ComputeFirstPrimeNumbers(dimension);
  seed_ = ResourceMap::GetAsUnsignedLong( "HaltonSequence-InitialSeed" );
}

/* Generate a pseudo-random vector of independant numbers uniformly distributed over [0, 1[ */
NumericalPoint HaltonSequence::generate()
{
  NumericalPoint realization(dimension_);
  // Loop over the components
  for (UnsignedLong i = 0; i < dimension_; ++i)
    {
      NumericalScalar xI(0.0);
      const uint64_t radix(base_[i]);
      const NumericalScalar inverseRadix(1.0 / radix);
      NumericalScalar inverseRadixN(inverseRadix);
      uint64_t currentSeed(seed_);
      while (currentSeed > 0)
        {
          xI += (currentSeed % radix) * inverseRadixN;
          currentSeed /= radix;
          inverseRadixN *= inverseRadix;
        }
      realization[i] = xI;
    }
  ++seed_;
  return realization;
}


/* String converter */
String HaltonSequence::__repr__() const
{
  OSS oss;
  oss << "class=" << HaltonSequence::GetClassName()
      << " derived from " << LowDiscrepancySequenceImplementation::__repr__()
      << " base=" << base_
      << " seed=" << seed_;
  return oss;
}

/* String converter */
String HaltonSequence::__str__(const String & offset) const
{
  OSS oss(false);
  oss << "class=" << HaltonSequence::GetClassName()
      << " derived from " << LowDiscrepancySequenceImplementation::__str__(offset)
      << " base=" << base_
      << " seed=" << seed_;
  return oss;
}

END_NAMESPACE_OPENTURNS
