//                                               -*- C++ -*-
/**
 *  @file  BoundConstrainedAlgorithm.hxx
 *  @brief BoundConstrainedAlgorithm implements an algorithm for finding the
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: BoundConstrainedAlgorithm.hxx 2392 2012-02-17 18:35:43Z schueller $
 */
#ifndef OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX
#define OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX

#include "TypedInterfaceObject.hxx"
#include "Threadable.hxx"
#include "BoundConstrainedAlgorithmImplementation.hxx"

BEGIN_NAMESPACE_OPENTURNS



/**
 * @class BoundConstrainedAlgorithm
 * BoundConstrainedAlgorithm implements an interface to an algorithm for finding the
 * point of an interval that minimize the given objective function
 */

class BoundConstrainedAlgorithm
  : public TypedInterfaceObject<BoundConstrainedAlgorithmImplementation>,
    public Threadable
{

  CLASSNAME;
public:

  typedef Pointer<BoundConstrainedAlgorithmImplementation>          Implementation;
  typedef BoundConstrainedAlgorithmImplementation::OptimizationProblem      OptimizationProblem;
#ifndef SWIG
  typedef BoundConstrainedAlgorithmImplementation::Result                   Result;
#else
  typedef BoundConstrainedAlgorithmImplementationResult                     Result;
#endif

  /** Default constructor */
  explicit BoundConstrainedAlgorithm();

  /** Constructor from an implementation */
  BoundConstrainedAlgorithm(const BoundConstrainedAlgorithmImplementation & implementationconst);

  /** Constructor from a Pointer to an implementation */
  BoundConstrainedAlgorithm(const Implementation & p_implementation);

  /** Constructor with parameters: no constraint, starting from the origin */
  BoundConstrainedAlgorithm(const NumericalMathFunction & objectiveFunction,
                            const Bool verbose = false);

  /** Constructor with parameters: bound constraints, starting from the given point */
  BoundConstrainedAlgorithm(const NumericalMathFunction & objectiveFunction,
                            const Interval & boundConstraints,
                            const NumericalPoint & startingPoint,
                            const OptimizationProblem optimization  = BoundConstrainedAlgorithmImplementationResult::MINIMIZATION,
                            const Bool verbose = false);

  /** Performs the actual computation. Must be overloaded by the actual optimisation algorithm */
  virtual void run();

  /** Starting point accessor */
  NumericalPoint getStartingPoint() const;
  void setStartingPoint(const NumericalPoint & startingPoint);

  /** Objective function accessor */
  NumericalMathFunction getObjectiveFunction() const;
  void setObjectiveFunction(const NumericalMathFunction & objectiveFunction);

  /** Bound constraints accessor */
  Interval getBoundConstraints() const;
  void setBoundConstraints(const Interval & boundConstraints);

  /** Optimization problem accessor */
  OptimizationProblem getOptimizationProblem() const;
  void setOptimizationProblem(const OptimizationProblem optimization);

  /** Result accessor */
  Result getResult() const;
  void setResult(const Result & result);

  /** Maximum iterations number accessor */
  UnsignedLong getMaximumEvaluationsNumber() const;
  void setMaximumEvaluationsNumber(const UnsignedLong maximumEvaluationsNumber);

  /** Maximum absolute error accessor */
  NumericalScalar getMaximumAbsoluteError() const;
  void setMaximumAbsoluteError(const NumericalScalar maximumAbsoluteError);

  /** Maximum relative error accessor */
  NumericalScalar getMaximumRelativeError() const;
  void setMaximumRelativeError(const NumericalScalar maximumRelativeError);

  /** Maximum objective error accessor */
  NumericalScalar getMaximumObjectiveError() const;
  void setMaximumObjectiveError(const NumericalScalar maximumObjectiveError);

  /** Maximum constraint error accessor */
  NumericalScalar getMaximumConstraintError() const;
  void setMaximumConstraintError(const NumericalScalar maximumConstraintError);

  /** String converter */
  virtual String __repr__() const;

  /** Verbose accessor */
  Bool getVerbose() const;
  void setVerbose(const Bool verbose);

} ; /* class BoundConstrainedAlgorithm */

END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX */
