//                                               -*- C++ -*-
/**
 *  @file  Path.hxx
 *  @brief This class provides all the mechanism of path manipulation
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2012-02-24 14:22:52 +0100 (Fri, 24 Feb 2012) $
 *  Id:      $Id: Path.hxx 2401 2012-02-24 13:22:52Z souchaud $
 */
#ifndef OPENTURNS_PATH_HXX
#define OPENTURNS_PATH_HXX

#include <iostream>              // for std::ostream
#include <vector>                // for std::vector
#include "OTprivate.hxx"

BEGIN_NAMESPACE_OPENTURNS



/**
 * @class Path
 * @brief Provides mechanism for path manipulation
 *
 * The class defines the search algorithms used to find a file or a directory
 * through the filesystem. It also defines the standard hardcoded paths and the
 * way temporary files are built.
 */

class Path
{
public:

  typedef std::vector<FileName> DirectoryList;

  /**
   * Return the Open TURNS prefix
   *
   * The search rule is: check the following paths, in that order:
   *    + if the env var OPENTURNS_HOME exists, return ${OPENTURNS_HOME} if it exists and is a
   *      directory,
   *    + otherwise return the installation path ${prefix}, where 'prefix' is the installation path
   *      of the platform as provided at configuration time.
   */
  static FileName GetInstallationDirectory();

  /**
   * Return the path of the module standard directory
   *
   * The search rule is: check the following paths, in that order:
   *    + if the env var OPENTURNS_HOME exists, return ${OPENTURNS_HOME}/lib/openturns/module if it
   *      exists and is a directory,
   *    + otherwise return the installation path ${prefix}/lib/openturns/module, where 'prefix' is
   *      the installation path of the platform as provided at configuration time.
   */
  static FileName GetModuleDirectory();

  /**
   * Analyse the process environment and return the directory where
   * installed wrappers and DTD are.
   *
   * The search rule is: check the following paths, in that order:
   *    + if the env var OPENTURNS_HOME exists, return ${OPENTURNS_HOME}/share/openturns/wrappers
   *      if it exists and is a directory,
   *    + otherwise return the installation path ${prefix}/share/openturns/wrappers, where 'prefix'
   *      is the installation path of the platform as provided at configuration time.
   */
  static FileName GetStandardWrapperDirectory();

  /**
   * Analyse the process environment
   * and return a list of directories to search in for wrappers.
   *
   * The search rule is :look for the file in the following directories, in that order :
   *    + if the env var OPENTURNS_WRAPPER_PATH exists, in directories listed in
   *      ${OPENTURNS_WRAPPER_PATH} (see openturnsWrapperPathVariableName_)
   *    + in directory ${HOME}/openturns/wrappers (see homeSubdirectory_)
   *    + in the standard wrapper directory, as defined by the method
   *      GetStandardWrapperDirectory().
   */
  static DirectoryList GetWrapperDirectoryList();

  /**
   * Analyse the process environment
   * and return a list of directories to search in for configuration files.
   *
   * The search rule is :look for the file in the following directories, in that order :
   *    + if the env var OPENTURNS_CONFIG_PATH exists, in directories listed in
   *      ${OPENTURNS_CONFIG_PATH} (see openturnsConfigPathVariableName_)
   *    + in directory ${HOME}/openturns/etc (see homeSubdirectory_)
   *    + in standard config directory (either ${OPENTURNS_HOME}/etc/openturns if OPENTURNS_HOME
   *      is defined and if this path is a directory, or the path ${prefix}/etc/openturns
   *      where 'prefix' is the installation path of the platform as provided at configuration time.
   */
  static DirectoryList GetConfigDirectoryList();

  /**
   * Find a file named \em name located in one of the directories
   * listed in \em dirList. The function returns the full path (directory/name)
   * of the first match.
   */
  static FileName FindFileByNameInDirectoryList(const FileName & name,
                                                const DirectoryList & dirList);

  /** Get the temporary directory set in the openturns conf file.
   *
   * On Windows, if temporary-directory is an env var, return the content of the env var,
   *   if not, just return the content of temporary-directory.
   * This function is mostly useful on windows in order to get the TEMP env var.
   */
  static FileName GetTemporaryDirectory();

  /** Build a temporary file name given a pattern
   * @param pattern A filename containing at least the 'XXXXXX' string
   * @result A new unique filename based on the \em pattern
   */
  static FileName BuildTemporaryFileName(const FileName & pattern);

  /** Create a temporary directory.
   */
  static FileName CreateTemporaryDirectory(const FileName & directoryPrefix);

  /** Delete a temporary directory
   */
  static void DeleteTemporaryDirectory(const FileName & directoryName);


protected:

  /**
   * @var static const char * Path::OpenturnsWrapperPathVariableName_
   *
   * The environment variable name
   */
  static const char * OpenturnsWrapperPathVariableName_;

  /**
   * @var static const char * Path::OpenturnsConfigPathVariableName_
   *
   * The environment variable name
   */
  static const char * OpenturnsConfigPathVariableName_;

  /**
   * @var static const char * Path::HomeWrapperSubdirectory_
   *
   * The HOME subdirectory path
   */
  static const char * HomeWrapperSubdirectory_;

  /**
   * @var static const char * Path::HomeConfigSubdirectory_
   *
   * The HOME subdirectory path
   */
  static const char * HomeConfigSubdirectory_;

  /**
   * @var static const char * Path::PrefixWrapperSubdirectory_
   *
   * The 'prefix' subdirectory path
   */
  static const char * PrefixWrapperSubdirectory_;

  /**
   * @var static const char * Path::PrefixConfigSubdirectory_
   *
   * The 'prefix' subdirectory path
   */
  static const char * PrefixConfigSubdirectory_;

  /**
   * @var static const char * Path:PrefixModuleSubdirectory_:
   *
   * The 'prefix' subdirectory path
   */
  static const char * PrefixModuleSubdirectory_;

public:

#ifdef WIN32
  /** Convert slash to antislash. */
  static void AntislashFileName(FileName & filename);

  /** Add windows backslash to filename for compatibility with R. */
  static void DoubleslashFileName(FileName & filename);
#endif

protected:

private:

  /** Default constructor */
  Path();

}; /* class Path */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_PATH_HXX */
