//                                               -*- C++ -*-
/**
 * @file  PythonWrappingFunctions.hxx
 * @brief This file provides functions to ease Python wrapping
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2009-09-14 14:39:35 +0200 (Mon, 14 Sep 2009) $
 */

#ifndef OPENTURNS_PYTHONWRAPPINGFUNCTIONS_HXX
#define OPENTURNS_PYTHONWRAPPINGFUNCTIONS_HXX

#include "Python.h"
#include "Collection.hxx"
#include "Histogram.hxx"

namespace OpenTURNS { 

  /** These templates are just declared, not defined. Only specializations are. */
  template <class CPP_Type>                    struct traitsPythonType;
  template <class PYTHON_Type>                 static inline int      isAPython(PyObject * pyObj);
  template <class PYTHON_Type>                 static inline String   namePython();
  template <class PYTHON_Type, class CPP_Type> static inline CPP_Type convert(PyObject * pyObj);
  template <class PYTHON_Type>                 static inline void     check(PyObject * pyObj);
  template <class PYTHON_Type, class CPP_Type> static inline CPP_Type checkAndConvert(PyObject * pyObj);
  template <class T>                           static inline T *      buildObjectFromPySequence(PyObject * pyObj);



  /** Specializations */


  /* PyObject */
  struct _PyObject_ {};

  template <>
  inline
  int
  isAPython<_PyObject_>(PyObject * pyObj)
  {
    return 1;
  }

  template <>
  inline
  String
  namePython<_PyObject_>()
  {
    return "object";
  }




  /* PyBool */
  struct _PyBool_ {};

  template <>
  inline
  int
  isAPython<_PyBool_>(PyObject * pyObj)
  {
    return PyBool_Check( pyObj );
  }

  template <>
  inline
  String
  namePython<_PyBool_>()
  {
    return "bool";
  }

  template <>
  struct traitsPythonType<Bool>
  {
    typedef _PyBool_ Type;
  };

  template <>
  inline
  Bool
  convert<_PyBool_,Bool>(PyObject * pyObj)
  {
    return pyObj == Py_True;
  }





  /* PyInt */
  struct _PyInt_ {};

  template <>
  inline
  int
  isAPython<_PyInt_>(PyObject * pyObj)
  {
    return PyInt_Check( pyObj );
  }

  template <>
  inline
  String
  namePython<_PyInt_>()
  {
    return "integer";
  }

  template <>
  struct traitsPythonType<UnsignedLong>
  {
    typedef _PyInt_ Type;
  };

  template <>
  inline
  UnsignedLong
  convert<_PyInt_,UnsignedLong>(PyObject * pyObj)
  {
    return PyInt_AsUnsignedLongMask( pyObj );
  }





  /* PyFloat */
  struct _PyFloat_ {};

  template <>
  inline
  int
  isAPython<_PyFloat_>(PyObject * pyObj)
  {
    return PyNumber_Check( pyObj );
  }

  template <>
  inline
  String
  namePython<_PyFloat_>()
  {
    return "double";
  }

  template <>
  struct traitsPythonType<NumericalScalar>
  {
    typedef _PyFloat_ Type;
  };

  template <>
  inline
  NumericalScalar
  convert<_PyFloat_,NumericalScalar>(PyObject * pyObj)
  {
    return PyFloat_AsDouble( pyObj );
  }




  /* PyString */
  struct _PyString_ {};

  template <>
  inline
  int
  isAPython<_PyString_>(PyObject * pyObj)
  {
    return PyString_Check( pyObj );
  }

  template <>
  inline
  String
  namePython<_PyString_>()
  {
    return "string";
  }

  template <>
  struct traitsPythonType<String>
  {
    typedef _PyString_ Type;
  };

  template <>
  inline
  String
  convert<_PyString_,String>(PyObject * pyObj)
  {
    return PyString_AsString( pyObj );
  }





  /* PySequence */
  struct _PySequence_ {};

  template <>
  inline
  int
  isAPython<_PySequence_>(PyObject * pyObj)
  {
    return PySequence_Check( pyObj );
  }

  template <>
  inline
  String
  namePython<_PySequence_>()
  {
    return "sequence object";
  }

  template <>
  struct traitsPythonType<NumericalComplex>
  {
    typedef _PySequence_ Type;
  };

  template <>
  inline
  NumericalComplex
  convert<_PySequence_,NumericalComplex>(PyObject * pyObj)
  {
    check<_PySequence_>( pyObj );
    if (PySequence_Fast_GET_SIZE( pyObj ) != 2) 
      throw Base::Common::InvalidArgumentException(HERE) << "Sequence passed as argument is not a pair (NumericalScalar, NumericalScalar)";
    PyObject * item_0 = PySequence_Fast_GET_ITEM( pyObj, 0 );
    check<_PyFloat_>( item_0 );
    PyObject * item_1 = PySequence_Fast_GET_ITEM( pyObj, 1 );
    check<_PyFloat_>( item_1 );
    return NumericalComplex( convert<_PyFloat_,NumericalScalar>( item_0 ),
			     convert<_PyFloat_,NumericalScalar>( item_1 ) );
  }





  template <class PYTHON_Type>
  static inline
  void
  check(PyObject * pyObj)
  {
    if (! isAPython<PYTHON_Type>( pyObj )) {
      throw Base::Common::InvalidArgumentException(HERE) << "Object passed as argument is not a " << namePython<PYTHON_Type>();
    }
  }


  template <class PYTHON_Type, class CPP_Type>
  static inline
  CPP_Type
  checkAndConvert(PyObject * pyObj)
  {
    check<PYTHON_Type>( pyObj );
    return convert<PYTHON_Type,CPP_Type>( pyObj );
  }





  template <class T>
  static inline
  Base::Type::Collection<T> *
  buildCollectionFromPySequence(PyObject * pyObj, int sz = 0)
  {
    check<_PySequence_>( pyObj );

    const UnsignedLong size = PySequence_Fast_GET_SIZE( pyObj );
    if ((sz != 0) && (sz != size)) {
      throw Base::Common::InvalidArgumentException(HERE) << "Sequence object has incorrect size " << size << ". Must be " << sz << ".";
    }
    Base::Type::Collection<T> * p_coll = new Base::Type::Collection<T>( size );

    for(UnsignedLong i=0; i<size; ++i) {
      PyObject * elt = PySequence_Fast_GET_ITEM( pyObj, i );
      check<typename traitsPythonType<T>::Type>( elt );
      (*p_coll)[i] = convert<typename traitsPythonType<T>::Type,T>( elt );
    }

    return p_coll;
  }

  


} /* namespace OpenTURNS */

#endif /* OPENTURNS_PYTHONWRAPPINGFUNCTIONS_HXX */
