//                                               -*- C++ -*-
/**
 *  @file  LinearModelTest.cxx
 *  @brief StatTest implements statistical tests
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: LinearModelTest.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include <fstream>
#include "LinearModelTest.hxx"
#include "LinearModelFactory.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "OTconfig.hxx"
#include "Log.hxx"
#include "Os.hxx"

namespace OpenTURNS
{
  namespace Uncertainty
  {
    namespace StatTest
    {

      typedef Base::Common::Log                        Log;
      typedef Base::Common::Path                       Path;
      typedef Base::Common::Os                         Os;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Stat::LinearModelFactory           LinearModelFactory;

      /* Default constructor */
      LinearModelTest::LinearModelTest()
      {
	// Nothing to do
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMAdjustedRSquared(const NumericalSample & firstSample,
								      const NumericalSample & secondSample,
								      const LinearModel & linearModel,
								      const NumericalScalar level)
      {
	return RunTwoSamplesALinearModelRTest(firstSample, secondSample, linearModel, level, "LmAdjustedRSquare");
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMAdjustedRSquared(const NumericalSample & firstSample,
								      const NumericalSample & secondSample,
								      const NumericalScalar level)
      {
        return LMAdjustedRSquared(firstSample, secondSample, LinearModelFactory().buildLM(firstSample, secondSample, level), level);
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMFisher(const NumericalSample & firstSample,
							    const NumericalSample & secondSample,
							    const LinearModel & linearModel,
							    const NumericalScalar level)
      {
	return RunTwoSamplesALinearModelRTest(firstSample, secondSample, linearModel, level, "LmFisher");
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMFisher(const NumericalSample & firstSample,
							    const NumericalSample & secondSample,
							    const NumericalScalar level)
      {
        return LMFisher(firstSample, secondSample, LinearModelFactory().buildLM(firstSample, secondSample, level), level);
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMResidualMean(const NumericalSample & firstSample,
								  const NumericalSample & secondSample,
								  const LinearModel & linearModel,
								  const NumericalScalar level)
      {
	return RunTwoSamplesALinearModelRTest(firstSample, secondSample, linearModel, level, "LmResidualMean");
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMResidualMean(const NumericalSample & firstSample,
								  const NumericalSample & secondSample,
								  const NumericalScalar level)
      {
        return LMResidualMean(firstSample, secondSample, LinearModelFactory().buildLM(firstSample, secondSample, level), level);
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMRSquared(const NumericalSample & firstSample,
							      const NumericalSample & secondSample,
							      const LinearModel & linearModel,
							      const NumericalScalar level)
      {
	return RunTwoSamplesALinearModelRTest(firstSample, secondSample, linearModel, level, "LmRsquared");
      }

      /*  */
      LinearModelTest::TestResult LinearModelTest::LMRSquared(const NumericalSample & firstSample,
							      const NumericalSample & secondSample,
							      const NumericalScalar level)
      {
        return LMRSquared(firstSample, secondSample, LinearModelFactory().buildLM(firstSample, secondSample, level), level);
      }

      /* Generic invocation of a R script for testing a linear model against two samples */
      LinearModelTest::TestResult LinearModelTest::RunTwoSamplesALinearModelRTest(const NumericalSample & firstSample,
										  const NumericalSample & secondSample,
										  const LinearModel & linearModel,
										  const NumericalScalar level,
										  const String & testName)
	/* throw(InternalException) */
      {
	String firstDataFileName(firstSample.storeToTemporaryFile());
	String secondDataFileName(secondSample.storeToTemporaryFile());
	NumericalSample regression(1, linearModel.getRegression());
	String regressionFileName(regression.storeToTemporaryFile());
	String resultFileName(Path::BuildTemporaryFileName("RResult.txt.XXXXXX"));
	String commandFileName(Path::BuildTemporaryFileName("RCmd.R.XXXXXX"));
	std::ofstream cmdFile(commandFileName.c_str(), std::ios::out);
	// Fill-in the command file
	cmdFile << "library(rotRPackage)" << std::endl;
	cmdFile << "options(digits=17)" << std::endl;
	cmdFile << "options(warn=-1)" << std::endl;
	cmdFile << "firstSample <- data.matrix(read.table(\"" << firstDataFileName << "\"))" << std::endl;
	cmdFile << "secondSample <- data.matrix(read.table(\"" << secondDataFileName << "\"))" << std::endl;
	cmdFile << "regression <- t(data.matrix(read.table(\"" << regressionFileName << "\")))" << std::endl;
	cmdFile << "res <- test" << testName;
	cmdFile << "(firstSample, regression, secondSample, " << level << ")" << std::endl;
	cmdFile << "f <- file(\"" << resultFileName << "\",\"wt\")" << std::endl;
	cmdFile << "cat(res$test, res$testResult, res$threshold, res$pValue, sep=\"\\n\", file=f)" << std::endl;
	cmdFile << "close(f)" << std::endl;
	cmdFile.close();
	OSS systemCommand;
#ifdef R_EXECUTABLE_PATH
	systemCommand << ResourceMap::GetInstance().get("R-executable-command") << " --no-save --silent < \"" << commandFileName << "\"" << Os::GetDeleteCommandOutput();
#else
	throw NotYetImplementedException(HERE) << "LinearModelTest::RunTwoSamplesALinearModelRTest() need R";
#endif
	int returnCode(system(String(systemCommand).c_str()));
	if (returnCode != 0) throw InternalException(HERE) << "Error: unable to execute the system command " << String(systemCommand) << " returned code is " << returnCode;
	// Parse result file
	std::ifstream resultFile(resultFileName.c_str(), std::ios::in);
	String testType;
	resultFile >> testType;
	Bool testResult;
	resultFile >> testResult;
	NumericalScalar pThreshold;
	resultFile >> pThreshold;
	NumericalScalar pValue;
	resultFile >> pValue;

	// Clean-up everything
	if (remove(firstDataFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << firstDataFileName);
	if (remove(secondDataFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << secondDataFileName);
	if (remove(regressionFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << regressionFileName);
	if (remove(resultFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << resultFileName);
	if (remove(commandFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot remove file " << commandFileName);

	return TestResult(testType, testResult, pValue, pThreshold);
      }


    } // namespace StatTest
  } // namespace Uncertainty
} // namespace OpenTURNS
