//                                               -*- C++ -*-
/**
 *  @file  LinearTaylor.cxx
 *  @brief First order polynomial response surface by Taylor expansion
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: LinearTaylor.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "LinearTaylor.hxx"
#include "LinearNumericalMathEvaluationImplementation.hxx"
#include "ConstantNumericalMathGradientImplementation.hxx"
#include "ConstantNumericalMathHessianImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      typedef Base::Func::LinearNumericalMathEvaluationImplementation   LinearNumericalMathEvaluationImplementation;
      typedef Base::Func::ConstantNumericalMathGradientImplementation ConstantNumericalMathGradientImplementation;
      typedef Base::Func::ConstantNumericalMathHessianImplementation  ConstantNumericalMathHessianImplementation;
      typedef Base::Type::SymmetricTensor                             SymmetricTensor;

      CLASSNAMEINIT(LinearTaylor);

      LinearTaylor::LinearTaylor(const String & name):
	PersistentObject(name)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      LinearTaylor::LinearTaylor(const NumericalPoint & center,
				 const NumericalMathFunction & inputFunction,
				 const String & name):
	PersistentObject(name),
	center_(center),
	inputFunction_(inputFunction)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      LinearTaylor * LinearTaylor::clone() const
      {
	return new LinearTaylor(*this);
      }

      /* String converter */
      String LinearTaylor::__repr__() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName()
	    << " center=" << center_
	    << " function=" << inputFunction_
	    << " responseSurface=" << responseSurface_
	    << " constant=" << constant_
	    << " linear=" << linear_;
	return oss;
      }

      /* Response surface computation */
      void LinearTaylor::run()
      {
	/* Compute the three first terms of the Taylor expansion */
	constant_ = inputFunction_(center_);
	linear_ = inputFunction_.gradient(center_);
	/* Build the several implementations and set it into the response surface */
	responseSurface_.setEvaluationImplementation(new LinearNumericalMathEvaluationImplementation(center_, constant_, linear_));
	responseSurface_.setGradientImplementation(new ConstantNumericalMathGradientImplementation(linear_));
	responseSurface_.setHessianImplementation(new ConstantNumericalMathHessianImplementation(SymmetricTensor(center_.getDimension(), constant_.getDimension())));
	responseSurface_.setDescription(inputFunction_.getDescription());
      }

      /* Center accessor */
      LinearTaylor::NumericalPoint LinearTaylor::getCenter() const
      {
	return center_;
      }

      /* Constant accessor */
      LinearTaylor::NumericalPoint LinearTaylor::getConstant() const
      {
	return constant_;
      }

      /* Linear accessor */
      LinearTaylor::Matrix LinearTaylor::getLinear() const
      {
	return linear_;
      }

      /* Function accessor */
      LinearTaylor::NumericalMathFunction LinearTaylor::getInputFunction() const
      {
	return inputFunction_;
      }

      /* Response surface accessor */
      LinearTaylor::NumericalMathFunction LinearTaylor::getResponseSurface() const
      {
	return responseSurface_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
