//                                               -*- C++ -*-
/**
 *  @file  RationalKernel.cxx
 *  @brief Implementation of the rational kernel
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: RationalKernel.hxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "RationalKernel.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(RationalKernel);


      /* Constructor with parameters */
      RationalKernel::RationalKernel(const NumericalScalar constant)
	: SVMKernelImplementation(),
	  constant_(constant)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      RationalKernel * RationalKernel::clone() const
      {
	return new RationalKernel(*this);
      }


      /* String converter */
      String RationalKernel::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " constant=" << getConstant();
      }


      /* Constant term accessor */
      NumericalScalar RationalKernel::getConstant() const
      {
	return constant_;
      }

      void RationalKernel::setConstant(NumericalScalar constant)
      {
	constant_ = constant;
      }


      /* Accessor to the parameter used for cross-validation */
      NumericalScalar RationalKernel::getParameter() const
      {
	return constant_;
      }

      void RationalKernel::setParameter(NumericalScalar value)
      {
	constant_ = value;
      }


      /* Parameters value and description accessor */
      RationalKernel::NumericalPointWithDescription RationalKernel::getParameters() const
      {
	NumericalPointWithDescription parameters(0);
	Description description(0);
	parameters.add(constant_);
	description.add("constant");
	parameters.setDescription(description);
	return parameters;
      }

      void RationalKernel::setParameters(const NumericalPointWithDescription & parameters)
      {
	if(parameters.getDimension() > 0)
	  constant_ = parameters[0];
      }

      /* Operator () */
      NumericalScalar RationalKernel::operator() (const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	NumericalPoint difference(x1 - x2);
	NumericalScalar norm2 = difference.norm2();
	NumericalScalar value = 1.0 - norm2 / (norm2 + constant_);
	return value;
      }


      /* Partial gradient */
      RationalKernel::NumericalPoint RationalKernel::partialGradient(const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	UnsignedLong dimension = x1.getDimension();
	NumericalPoint difference(x1 - x2);
	NumericalScalar norm2 = difference.norm2();
	NumericalPoint partialGradient(dimension);
	for(UnsignedLong i = 0; i < dimension; ++ i)
	  {
	    partialGradient[i] = - 2.0 * constant_ * (x1[i] - x2[i]) / ((norm2 + constant_) * (norm2 + constant_));
	  }
	return partialGradient;
      }


      /* Partial hessian */
      RationalKernel::SymmetricMatrix RationalKernel::partialHessian(const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	UnsignedLong dimension = x1.getDimension();
	NumericalPoint difference(x1 - x2);
	NumericalScalar norm2 = difference.norm2();
	NumericalScalar denominator = (norm2 + constant_) * (norm2 + constant_) * (norm2 + constant_);
	SymmetricMatrix partialHessian(dimension);
	for(UnsignedLong i = 0; i < dimension; ++ i)
	  {
	    for(UnsignedLong j = 0; j <= i; ++ j)
	      {
		if(i == j)
		  {
		    partialHessian(i, i) = 2.0 * constant_ * (- (norm2 + constant_) + 4.0 * (x1[i] - x2[i]) * (x1[i] - x2[i])) / denominator;
		  }
		else
		  {
		    partialHessian(i, j) = 8.0 * constant_ * (x1[i] - x2[i]) * (x1[j] - x2[j]) / denominator;
		  }
	      }
	  }
	return partialHessian;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
