//                                               -*- C++ -*-
/**
 *  @file  MarginalTransformationEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: MarginalTransformationEvaluation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "MarginalTransformationEvaluation.hxx"
#include "Uniform.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(MarginalTransformationEvaluation);

      typedef Uncertainty::Distribution::Uniform       Uniform;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      /* Parameter constructor */
      MarginalTransformationEvaluation::MarginalTransformationEvaluation(const DistributionCollection & inputDistributionCollection,
                                                                         const DistributionCollection & outputDistributionCollection):
        NumericalMathEvaluationImplementation(),
        inputDistributionCollection_(inputDistributionCollection),
        outputDistributionCollection_(outputDistributionCollection),
        direction_(FROMTO)
      {
        Description description;
        const UnsignedLong size(inputDistributionCollection.getSize());
        // Check that the collections of input and output distributions have the same size
        if (outputDistributionCollection_.getSize() != size) throw InvalidArgumentException(HERE) << "Error: a MarginalTransformationEvaluation cannot be built using collections of input and output distributions of different size";
        // First, check that the distributions are all 1D
        for (UnsignedLong i = 0; i < size; ++i)
          {
            if (inputDistributionCollection_[i].getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: a MarginalTransformationEvaluation cannot be built using distributions with dimension > 1.";
            if (outputDistributionCollection_[i].getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: a MarginalTransformationEvaluation cannot be built using distributions with dimension > 1.";
          }
        // Second, build the description of the transformation
        for (UnsignedLong i = 0; i < size; ++i)
          {
            OSS oss;
            oss << "x" << i;
            description.add(oss);
          }
        for (UnsignedLong i = 0; i < size; ++i)
          {
            OSS oss;
            oss << "y" << i;
            description.add(oss);
          }
        setDescription(description);
      }

      /* Parameter constructor */
      MarginalTransformationEvaluation::MarginalTransformationEvaluation(const DistributionCollection & distributionCollection,
                                                                         const UnsignedLong direction):
        NumericalMathEvaluationImplementation(),
        inputDistributionCollection_(0),
        outputDistributionCollection_(0),
        direction_(direction)
      {
        switch (direction)
          {
          case FROM:
            // From the given marginals to the uniform ones
            inputDistributionCollection_ = distributionCollection;
            outputDistributionCollection_ = DistributionCollection(distributionCollection.getSize(), Uniform(0.0, 1.0));
            break;
          case TO:
            // From uniform marginals to the given ones
            inputDistributionCollection_ = DistributionCollection(distributionCollection.getSize(), Uniform(0.0, 1.0));
            outputDistributionCollection_ = distributionCollection;
            break;
          default:
            throw InvalidArgumentException(HERE) << "Error: wrong value given for direction";
          }
        *this = MarginalTransformationEvaluation(inputDistributionCollection_, outputDistributionCollection_);
        // We must overwrite the value of direction_ by the given one, as the call of the general constructor has set the value to FROMTO
        direction_ = direction;
        // Get all the parameters
        // The notion of parameters is used only for transformation from or to a standard space, so we have to extract the parameters of either the input distributions or the output distribution depending on the direction
        const UnsignedLong size(distributionCollection.getSize());
        NumericalPointWithDescription parameters(0);
        Description parametersDescription(0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            // The marginal distribution is 1D, so the collection of parameters is of size 1
            NumericalPointWithDescription marginalParameters(distributionCollection[i].getParametersCollection()[0]);
            const UnsignedLong marginalParametersSize(marginalParameters.getSize());
            for (UnsignedLong j = 0; j < marginalParametersSize; ++j)
              {
                parameters.add(marginalParameters[j]);
                parametersDescription.add(marginalParameters.getDescription()[j]);
              }
            parameters.setDescription(parametersDescription);
          } // get all the parameters
        setParameters(parameters);
      }

      /* Virtual constructor */
      MarginalTransformationEvaluation * MarginalTransformationEvaluation::clone() const
      {
        return new MarginalTransformationEvaluation(*this);
      }

      /* Evaluation */
      MarginalTransformationEvaluation::NumericalPoint MarginalTransformationEvaluation::operator () (const NumericalPoint & in) const
      /* throw (InvalidArgumentException, InternalException) */
      {
        const UnsignedLong dimension(getOutputDimension());
        NumericalPoint result(dimension);
        // The marginal transformation apply G^{-1} o F to each component of the input, where F is the ith input CDF and G the ith output CDf
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
            NumericalScalar inputCDF(inputDistributionCollection_[i].computeCDF(in[i]));
            // For accuracy reason, check if we are in the upper tail of the distribution
            const Bool upperTail(inputCDF > 0.99);
            if (upperTail) inputCDF = inputDistributionCollection_[i].computeCDF(in[i], upperTail);
            // The upper tail CDF is defined by CDF(x, upper) = P(X>x)
            // The upper tail quantile is defined by Quantile(CDF(x, upper), upper) = x
	    const NumericalScalar q(outputDistributionCollection_[i].computeQuantile(inputCDF, upperTail)[0]);
            result[i] = q;
          }
        return result;
      }

      /* Gradient according to the marginal parameters.
       *
       * F is the CDF of the ith marginal input distribution
       * Q is the quantile function of the ith output distribution
       *
       * F : RxRs -> [0,1]
       *    (x, pf) -> F(x, pf)
       *
       * Q : [0,1]xRt -> R
       *    (y, pq) -> G(y, pq)
       *
       * Let p = [pg, pf] be the parameter vector of H, with:
       *
       * H : RmxRt+s -> Rn
       *     (x, p)  -> Q(F(x, pq), pf)
       *
       * We have:
       *
       * (dH/dp)(x, p) = [(dQ/dy)(F(x, pf), pq) . (dF/dpf)(x, pf), 0] + [0, (dQ/dpq)(F(x, pf), pq)]
       *
       * The computation of (dQ/dy) leads to:
       *
       * (dQ/dy)(y, pq) = 1 / (pdf(Q(y, pq)))
       *
       * where pdf is the PDF of the distribution associated with the quantile function Q.
       *
       * The computation of (dQ/dpq) leads to:
       *
       * (dQ/dpq)(y, pq) = -(dcdf/dpq)(Q(y, pq), pq) / pdf(Q(y, pq), pq)
       *
       * where dcdf/dpq is the gradient according to its parameters of the CDF of the distribution
       * associated with the quantile function Q.
       *
       * the needed gradient is [(dH/dp)(x,p)]^t
       *
       */
      MarginalTransformationEvaluation::Matrix MarginalTransformationEvaluation::parametersGradient(const NumericalPoint & in) const
      {
        const NumericalPoint parameters(getParameters());
        const UnsignedLong parametersDimension(parameters.getDimension());
        const UnsignedLong inputDimension(getInputDimension());
        Matrix result(parametersDimension, inputDimension);
        UnsignedLong rowIndex(0);
        switch (direction_)
          {
          case FROM:
            /*
	     * Here, we suppose that pq is empty, so dQ/dpq = 0
             * For each row, store (dF/dpf)(x, pf) / pdf(Q(F(x, pf)))
             */
            for (UnsignedLong j = 0; j < inputDimension; ++j)
              {
		const NumericalPoint x(1, in[j]);
		const Distribution inputMarginal(inputDistributionCollection_[j]);
		const Distribution outputMarginal(outputDistributionCollection_[j]);
		const NumericalScalar denominator(outputMarginal.computePDF(outputMarginal.computeQuantile(inputMarginal.computeCDF(x))));
		if (denominator > 0.0)
		  {
		    const NumericalPoint normalizedCDFGradient(inputMarginal.computeCDFGradient(x) * (1.0 / denominator));
		    const UnsignedLong marginalParametersDimension(normalizedCDFGradient.getDimension());
		    for (UnsignedLong i = 0; i < marginalParametersDimension; ++i)
		      {
			result(rowIndex, j) = normalizedCDFGradient[i];
			++rowIndex;
		      }
		  }
              } // FROM
            return result;
          case TO:
            /*
             * Here, we suppose that pf is empty, so dF/dpf = 0
	     * For each row, store -(dcdf/dpq)(Q(F(x), pq), pq) / pdf(Q(F(x), pq), pq)
             */
            for (UnsignedLong j = 0; j < inputDimension; ++j)
              {
		const NumericalPoint x(1, in[j]);
		const Distribution inputMarginal(inputDistributionCollection_[j]);
		const Distribution outputMarginal(outputDistributionCollection_[j]);
		const NumericalPoint q(outputMarginal.computeQuantile(inputMarginal.computeCDF(x)));
		const NumericalScalar denominator(outputMarginal.computePDF(q));
		if (denominator > 0.0)
		  {
		    const NumericalPoint normalizedCDFGradient(outputMarginal.computeCDFGradient(q) * (-1.0 / denominator));
		    const UnsignedLong marginalParametersDimension(normalizedCDFGradient.getDimension());
		    for (UnsignedLong i = 0; i < marginalParametersDimension; ++i)
		      {
			result(rowIndex, j) = normalizedCDFGradient[i];
			++rowIndex;
		      }
		  }
	      } // TO
            return result;
          default:
            // Should never go there
            throw NotYetImplementedException(HERE);
          }
      }

      /* Accessor for input point dimension */
      UnsignedLong MarginalTransformationEvaluation::getInputDimension() const
      /* throw(InternalException) */
      {
        return inputDistributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong MarginalTransformationEvaluation::getOutputDimension() const
      /* throw(InternalException) */
      {
        return inputDistributionCollection_.getSize();
      }

      /* Direction accessor */
      void MarginalTransformationEvaluation::setDirection(const TranformationDirection direction)
      {
        direction_ = direction;
      }

      UnsignedLong MarginalTransformationEvaluation::getDirection() const
      {
        return direction_;
      }

      /* Input distribution collection accessor */
      void MarginalTransformationEvaluation::setInputDistributionCollection(const DistributionCollection & inputDistributionCollection)
      {
        inputDistributionCollection_ = inputDistributionCollection;
      }

      MarginalTransformationEvaluation::DistributionCollection MarginalTransformationEvaluation::getInputDistributionCollection() const
      {
        return inputDistributionCollection_;
      }

      /* Output distribution collection accessor */
      void MarginalTransformationEvaluation::setOutputDistributionCollection(const DistributionCollection & outputDistributionCollection)
      {
        outputDistributionCollection_ = outputDistributionCollection;
      }

      MarginalTransformationEvaluation::DistributionCollection MarginalTransformationEvaluation::getOutputDistributionCollection() const
      {
        return outputDistributionCollection_;
      }

      /* String converter */
      String MarginalTransformationEvaluation::__repr__() const
      {
        OSS oss;
        oss << "class=" << MarginalTransformationEvaluation::GetClassName()
            << " description=" << getDescription()
            << " input marginals=" << inputDistributionCollection_
            << " output marginals=" << outputDistributionCollection_;
        return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

