//                                               -*- C++ -*-
/**
 *  @file  NonCenteredFiniteDifferenceGradient.cxx
 *  @brief Class for the creation of a numerical math gradient implementation
 *         form a numerical math evaluation implementation by using noncentered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NonCenteredFiniteDifferenceGradient.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cstdlib>

#include "NonCenteredFiniteDifferenceGradient.hxx"
#include "PersistentObjectFactory.hxx"
#include "ResourceMap.hxx"


namespace OpenTURNS {

  namespace Base {

    namespace Diff {

      CLASSNAMEINIT(NonCenteredFiniteDifferenceGradient);

      typedef Func::NumericalMathEvaluationImplementation NumericalMathEvaluationImplementation;
      typedef Common::ResourceMap                         ResourceMap;

      static Common::Factory<NonCenteredFiniteDifferenceGradient> RegisteredFactory("NonCenteredFiniteDifferenceGradient");

      const NumericalScalar NonCenteredFiniteDifferenceGradient::DefaultEpsilon = 1.0e-7;//strtod(ResourceMap::GetInstance().get("NonCenteredFiniteDifferenceGradient-DefaultEpsilon").c_str(), NULL);

      /* Default constructor */
      NonCenteredFiniteDifferenceGradient::NonCenteredFiniteDifferenceGradient() :
        NumericalMathGradientImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      NonCenteredFiniteDifferenceGradient::NonCenteredFiniteDifferenceGradient(const NumericalPoint & epsilon,
									       const EvaluationImplementation & p_evaluation)
	/* throw(InvalidDimensionException)*/
	: NumericalMathGradientImplementation(),
	  epsilon_(epsilon),
	  p_evaluation_(p_evaluation)
      {
	/* Check if the dimension of the constant term is compatible with the linear and quadratic terms */
        if (epsilon.getDimension() != p_evaluation->getInputDimension()) throw InvalidDimensionException(HERE) << "Epsilon dimension is incompatible with the given evaluation";
	/* Check if any epsilon component is exactly zero */
	for(UnsignedLong i = 0; i < epsilon.getDimension(); i++)
	  {
	    if (epsilon[i] == 0.0) throw InvalidArgumentException(HERE) << "At least one of the components of epsilon is equal to 0.0, namely component " << i;
	  }
      }

      /* Parameter constructor */
      NonCenteredFiniteDifferenceGradient::NonCenteredFiniteDifferenceGradient(const NumericalScalar epsilon,
									       const EvaluationImplementation & p_evaluation)
      /* throw(InvalidDimensionException) */
	: NumericalMathGradientImplementation(),
	  epsilon_(NumericalPoint(p_evaluation->getInputDimension(), epsilon)),
	  p_evaluation_(p_evaluation)
      {
	/* Check if epsilon is exactly zero */
	if (epsilon == 0.0) throw InvalidArgumentException(HERE) << "The given scalar epsilon is equal to 0.0";
      }

      /* Virtual constructor */
      NonCenteredFiniteDifferenceGradient * NonCenteredFiniteDifferenceGradient::clone() const
      {
	return new NonCenteredFiniteDifferenceGradient(*this);
      }

      /* Comparison operator */
      Bool NonCenteredFiniteDifferenceGradient::operator ==(const NonCenteredFiniteDifferenceGradient & other) const
      {
	return (epsilon_ == other.getEpsilon());
      }
  
      /* String converter */
      String NonCenteredFiniteDifferenceGradient::__repr__() const {
	OSS oss;
	oss << "class=" << NonCenteredFiniteDifferenceGradient::GetClassName()
	    << " name=" << getName()
	    << " epsilon=" << epsilon_
            << " evaluation=" << p_evaluation_->__repr__();
	return oss;
      }
  
      /* Accessor for epsilon */
      NonCenteredFiniteDifferenceGradient::NumericalPoint NonCenteredFiniteDifferenceGradient::getEpsilon() const
      {
	return epsilon_;
      }

      /* Accessor for the evaluation */
      NonCenteredFiniteDifferenceGradient::EvaluationImplementation NonCenteredFiniteDifferenceGradient::getEvaluation() const
      {
	return p_evaluation_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Gradient () */
      NonCenteredFiniteDifferenceGradient::Matrix NonCenteredFiniteDifferenceGradient::gradient(const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	UnsignedLong inputDimension(in.getDimension());
	if (inputDimension != epsilon_.getDimension()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	/* At which points do we have to compute the evaluation for the decentered finite difference. We need 1+dim pionts. */
	Stat::NumericalSample gridPoints(inputDimension + 1, in);
	for(UnsignedLong i = 0; i < inputDimension; i++)
	  {
	    gridPoints[i][i] += epsilon_[i];
	  }
	/* Evaluate the evaluation */
	Stat::NumericalSample gridValues(p_evaluation_->operator()(gridPoints));
	/* Get the value at the center of the grid */
	NumericalPoint center(gridValues[inputDimension]);
	/* Compute the gradient */
	Matrix result(p_evaluation_->getInputDimension(), p_evaluation_->getOutputDimension());
	for (UnsignedLong i = 0; i < result.getNbRows(); i++)
	  {
	    for (UnsignedLong j = 0; j < result.getNbColumns(); j++)
	      {
		/* result(i, j) = (f_j(x + e_i) - f_j(x)) / e_i ~ df_j / dx_i */
		result(i, j) = (gridValues[i][j] - center[j]) / epsilon_[i];
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong NonCenteredFiniteDifferenceGradient::getInputDimension() const
	/* throw(InternalException) */
      {
	return p_evaluation_->getInputDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong NonCenteredFiniteDifferenceGradient::getOutputDimension() const
	/* throw(InternalException) */
      {
	return p_evaluation_->getOutputDimension();
      }
      
      /* Method save() stores the object through the StorageManager */
      void NonCenteredFiniteDifferenceGradient::save(StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.saveAttribute( "epsilon_", epsilon_ );
	adv.saveAttribute( "evaluation_", *p_evaluation_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NonCenteredFiniteDifferenceGradient::load(StorageManager::Advocate & adv)
      {
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluation;
	NumericalMathGradientImplementation::load(adv);
	adv.loadAttribute( "epsilon_", epsilon_ );
	adv.loadAttribute( "evaluation_", evaluation );
	p_evaluation_ = evaluation.getImplementation();
      }
      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
