#! /usr/bin/env python

from openturns import *
from math import *

TESTPREAMBLE()
RandomGenerator().SetSeed(0)

try :
    # Problem parameters
    dimension = 3
    a = 7.0
    b = 0.1
    # Reference analytical values
    meanTh = a/2
    covTh = (b**2 * pi**8) / 18.0 + (b * pi**4) / 5.0 + (a**2) / 8.0 + 1.0 / 2.0
    sob_1 = [(b * pi**4 / 5.0 + b**2 * pi**8 / 50.0 + 1.0/2.0) / covTh, (a**2 / 8.0) / covTh, 0.0]
    sob_2 = [0.0, (b**2 * pi**8 / 18.0 - b**2 * pi**8 / 50.0) / covTh, 0.0]
    sob_3 = [0.0]
    sob_T1 = [sob_1[0] + sob_2[0] + sob_2[1] + sob_3[0], sob_1[1] + sob_2[0] + sob_2[2] + sob_3[0], sob_1[2] + sob_2[1] + sob_2[2] + sob_3[0]]
    sob_T2 = [sob_2[0] + sob_2[1] + sob_3[0], sob_2[0] + sob_2[2] + sob_3[0], sob_2[1] + sob_2[2] + sob_3[0]]
    sob_T3 = [sob_3[0]]
    # Create the Ishigami function
    inputVariables = Description(dimension)
    inputVariables[0] = "xi1"
    inputVariables[1] = "xi2"
    inputVariables[2] = "xi3"
    outputVariables = Description(1)
    outputVariables[0] = "y"
    formula = Description(1)
    formula[0] = "sin(xi1) + (" + str(a) + ") * (sin(xi2)) ^ 2 + (" + str(b) + ") * xi3^4 * sin(xi1)"
    model = NumericalMathFunction(inputVariables, outputVariables, formula)
    
    # Create the input distribution
    marginals = DistributionCollection(dimension)
    marginals[0] = Distribution(Uniform(-pi, pi))
    marginals[1] = Distribution(Uniform(-pi, pi))
    marginals[2] = Distribution(Uniform(-pi, pi))
    distribution = Distribution(ComposedDistribution(marginals))

    # Create the orthogonal basis
    polynomialCollection = PolynomialFamilyCollection(dimension)
    polynomialCollection[0] = OrthogonalUniVariatePolynomialFamily(LegendreFactory())
    polynomialCollection[1] = OrthogonalUniVariatePolynomialFamily(LegendreFactory())
    polynomialCollection[2] = OrthogonalUniVariatePolynomialFamily(LegendreFactory())
    enumerateFunction = EnumerateFunction(dimension)
    productBasis = OrthogonalBasis(OrthogonalProductPolynomialFactory(polynomialCollection, enumerateFunction))

    # Create the adaptive strategy
    # We can choose amongst several strategies
    # First, the most efficient (but more complex!) strategy
    listAdaptiveStrategy = list()
    indexMax = 200
    basisDimension = 20
    threshold = 1.0e-6
    listAdaptiveStrategy.append(AdaptiveStrategy(CleaningStrategy(productBasis, indexMax, basisDimension, threshold, False)))
    # Second, the most used (and most basic!) strategy
    degree = 6
    listAdaptiveStrategy.append(AdaptiveStrategy(FixedStrategy(productBasis, enumerateFunction.getStrateCumulatedCardinal(degree))))
    # Third, a slight enhancement with respect to the basic strategy
    degree = 3
    listAdaptiveStrategy.append(AdaptiveStrategy(SequentialStrategy(productBasis, enumerateFunction.getStrateCumulatedCardinal(degree), False)))

    for adaptiveStrategyIndex in range(len(listAdaptiveStrategy)):
        adaptiveStrategy = listAdaptiveStrategy[adaptiveStrategyIndex]
        # Create the projection strategy
        samplingSize = 250
        listProjectionStrategy = list()
        # We have only the LeastSquaresStrategy up to now (0.13.0) but we can choose several sampling schemes
        # Monte Carlo sampling
        listProjectionStrategy.append(ProjectionStrategy(LeastSquaresStrategy(MonteCarloExperiment(samplingSize))))
        # LHS sampling
        listProjectionStrategy.append(ProjectionStrategy(LeastSquaresStrategy(LHSExperiment(samplingSize))))
        # Low Discrepancy sequence
        listProjectionStrategy.append(ProjectionStrategy(LeastSquaresStrategy(LowDiscrepancyExperiment(LowDiscrepancySequence(SobolSequence()),samplingSize))))
        for projectionStrategyIndex in range(len(listProjectionStrategy)):
            projectionStrategy = listProjectionStrategy[projectionStrategyIndex]
            # Create the polynomial chaos algorithm
            maximumResidual = 1.0e-10
            algo = FunctionalChaosAlgorithm(model, distribution, adaptiveStrategy, projectionStrategy)
            algo.setMaximumResidual(maximumResidual)
            RandomGenerator.SetSeed(0)
            algo.run()

            # Examine the results
            result = algo.getResult()
            print "###################################"
            print adaptiveStrategy
            print projectionStrategy
            #print "coefficients=", result.getCoefficients()
            residual = result.getResidual()
            print "residual=%.4f" % residual

            # Post-process the results
            vector = FunctionalChaosRandomVector(result)
            mean = vector.getMean()[0]
            print "mean=%.4e" % mean, "absolute error=%.1e" % fabs(mean - meanTh)
            variance = vector.getCovariance()[0, 0]
            print "variance=%.4e" % variance, "absolute error=%.1e" % fabs(variance - covTh)
            for i in range(dimension):
                value = vector.getSobolIndex(i)
                print "Sobol index", i, "= %.4e" % value, "absolute error=%.1e" % fabs(value - sob_1[i])
            indices = Indices(2)
            k = 0
            for i in range(dimension):
                indices[0] = i
                for j in range(i+1, dimension):
                    indices[1] = j
                    value = vector.getSobolIndex(indices)
                    print "Sobol index", indices, "=%.4e" % value, "absolute error=%.1e" % fabs(value - sob_2[k])
                    k = k+1
            indices = Indices(3)
            indices[0] = 0
            indices[1] = 1
            indices[2] = 2
            value = vector.getSobolIndex(indices)
            print "Sobol index", indices, "=%.4e" % value, "absolute error=%.1e" % fabs(value - sob_3[0])
            for i in range(dimension):
                value = vector.getSobolTotalIndex(i)
                print "Sobol total index", i, "=%.4e" % value, "absolute error=%.1e" % fabs(value - sob_T1[i])
            indices = Indices(2)
            k = 0
            for i in range(dimension):
                indices[0] = i
                for j in range(i+1, dimension):
                    indices[1] = j
                    value = vector.getSobolIndex(indices)
                    print "Sobol total index", indices, "=%.4e" % value, "absolute error=%.1e" % fabs(value - sob_2[k])
                    k = k+1
            indices = Indices(3)
            indices[0] = 0
            indices[1] = 1
            indices[2] = 2
            value = vector.getSobolTotalIndex(indices)
            print "Sobol total index ", indices, "=%.4e" % value, "absolute error=%.1e" % fabs(value - sob_3[0])
    
except : 
   import sys
   print "t_FunctionalChaos_ishigami.py", sys.exc_type, sys.exc_value
