//                                               -*- C++ -*-
/**
 *  @file  t_NumericalSample_computation.cxx
 *  @brief The test file of class NumericalSample for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: t_NumericalSample_computation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "NumericalSample.hxx"
#include "Normal.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Func;
using namespace OT::Uncertainty::Distribution;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    NumericalSample sample(4, 3);
    sample[0][0] = 1;
    sample[0][1] = 0;
    sample[0][2] = 9;
    sample[1][0] = 2;
    sample[1][1] = 3;
    sample[1][2] = 5;
    sample[2][0] = 5;
    sample[2][1] = 1;
    sample[2][2] = 8;
    sample[3][0] = 6;
    sample[3][1] = 7;
    sample[3][2] = 2;

    fullprint << "sample=" << sample << std::endl;
    fullprint << "min=" << sample.getMin() << std::endl;
    fullprint << "max=" << sample.getMax() << std::endl;
    fullprint << "mean=" << sample.computeMean() << std::endl;
    fullprint << "covariance=" << sample.computeCovariance() << std::endl;
    fullprint << "standard deviation=" << sample.computeStandardDeviation() << std::endl;
    fullprint << "standard deviation per component=" << sample.computeStandardDeviationPerComponent() << std::endl;
    fullprint << "Pearson correlation=" << sample.computePearsonCorrelation() << std::endl;
    fullprint << "Spearman correlation=" << sample.computeSpearmanCorrelation() << std::endl;
    fullprint << "Kendall tau=" << sample.computeKendallTau() << std::endl;
    fullprint << "range per component=" << sample.computeRangePerComponent() << std::endl;
    fullprint << "median per component=" << sample.computeMedianPerComponent() << std::endl;
    fullprint << "Variance=" << sample.computeVariancePerComponent() << std::endl;
    fullprint << "Skewness=" << sample.computeSkewnessPerComponent() << std::endl;
    fullprint << "Kurtosis=" << sample.computeKurtosisPerComponent() << std::endl;
    fullprint << "Marginal 1=" << sample.getMarginal(1) << std::endl;
    NumericalSample::Indices indices(2);
    indices[0] = 2;
    indices[1] = 0;
    fullprint << "Marginal [2, 0]=" << sample.getMarginal(indices) << std::endl;

    NumericalScalar prob(0.25);
    fullprint << "Quantile per component(" << prob << ")=" << sample.computeQuantilePerComponent(prob) << std::endl;
    //    fullprint << "Quantile(" << prob << ")=" << sample.computeQuantile(prob) << std::endl;
    NumericalPoint pointCDF(sample.getDimension(), 0.25);
    fullprint << "Empirical CDF(" << pointCDF << "=" << sample.computeEmpiricalCDF(pointCDF) << std::endl;
    UnsignedLong dim(3);
    CorrelationMatrix R(dim);
    for (UnsignedLong i = 1; i < dim; i++)
      {
	R(i, i - 1) = 0.25;
      }
    CorrelationMatrix Rtmp(dim);
    for (UnsignedLong i = 0; i < dim; i++)
      {
	for (UnsignedLong j = 0; j < i; j++)
	  {
	    Rtmp(i, j) = 6.0 * asin(R(i, j) / 2.0) / M_PI;
	  }
      }
    fullprint << "Pearson correlation (exact)=" << R << std::endl;
    fullprint << "Spearman correlation (exact)=" << Rtmp << std::endl;
    UnsignedLong size(10000);
    NumericalPoint p1(dim, 0.0);
    NumericalPoint p2(dim, 1.0);
    Normal normal(Normal(p1, p2, R));
    fullprint << "Normal=" << normal << std::endl;
    fullprint << "covariance=" << normal.getCovariance() << std::endl;
    NumericalSample normalSample(normal.getNumericalSample(size));
    fullprint << "Empirical covariance=" << normalSample.computeCovariance() << std::endl;
    CorrelationMatrix RPearson(normalSample.computePearsonCorrelation());
    fullprint << "Pearson correlation=" << RPearson << std::endl;
    CorrelationMatrix RSpearman(normalSample.computeSpearmanCorrelation());
    fullprint << "Spearman correlation=" << RSpearman << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
