//                                               -*- C++ -*-
/**
 *  @file  Rayleigh.cxx
 *  @brief The Rayleigh distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-09-24 09:07:16 +0200 (mer, 24 sep 2008) $
 *  Id:      $Id: Rayleigh.cxx 942 2008-09-24 07:07:16Z lebrun $
 */
#include <cmath>
#include "Rayleigh.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Rayleigh);

      /* Default constructor */
      Rayleigh::Rayleigh()
	: NonEllipticalDistribution("Rayleigh"),
	  sigma_(1.0),
	  gamma_(0.0)
      {
	setDimension(1);
	computeRange();
      }

      /* Parameters constructor */
      Rayleigh::Rayleigh(const NumericalScalar sigma,
			 const NumericalScalar gamma)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("Rayleigh"),
	  sigma_(0.0), gamma_(gamma)
      {
	setSigma(sigma);
	setDimension(1);
	computeRange();
      }

      /* Comparison operator */
      Bool Rayleigh::operator ==(const Rayleigh & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write Rayleigh::operator ==(...)
	  sameObject = (sigma_ == other.sigma_) && (gamma_ == other.gamma_);
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Rayleigh::__repr__() const
      {
	OSS oss;
	oss << "class=" << Rayleigh::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " sigma=" << sigma_
	    << " gamma=" << gamma_;
	return oss;
      }
  
      /* Virtual constructor */
      Rayleigh * Rayleigh::clone() const
      {
	return new Rayleigh(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Rayleigh::computeRange()
      {
	const NumericalPoint lowerBound(1, gamma_);
	const NumericalPoint upperBound(1, gamma_ + sqrt(SpecFunc::LogMaxNumericalScalar));
	const Interval::BoolCollection finiteLowerBound(1, true);
	const Interval::BoolCollection finiteUpperBound(1, false);
	setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }


      /* Get one realization of the distribution */
      Rayleigh::NumericalPoint Rayleigh::getRealization() const
      {
	return NumericalPoint(1, gamma_ + sigma_ * sqrt(-2.0 * log(RandomGenerator::Generate())));
      }
     

      /* Get the DDF of the distribution */
      Rayleigh::NumericalPoint Rayleigh::computeDDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return NumericalPoint(1, 0.0);
	const NumericalScalar y(x / sigma_);
	const NumericalScalar sigma2(sigma_ * sigma_);
	return NumericalPoint(1, -exp(-0.5 * y * y) * (x - sigma_) * (x + sigma_) / (sigma2 * sigma2));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Rayleigh::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return 0.0;
	const NumericalScalar y(x / (sigma_ * sigma_));
	return y * exp(-0.5 * x * y);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Rayleigh::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return (tail ? 1.0 : 0.0);
	const NumericalScalar y(x / sigma_);
	if (tail) return exp(-0.5 * y * y);
	return 1.0 - exp(-0.5 * y * y);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X))
	 phi(x) = (1 - sigma * x * exp(-sigma^2 * x^2 / 2) * sqrt(pi / 2) * (erfi(sigma * x / sqrt(2)) - i)) * exp(i * gamma * x)
	 erfi(t) = -i * erf(i * t) = 2 / sqrt(pi) * int(exp(u^2), u=0..t)
	 dawson(t) = exp(-t^2) * int(exp(u^2), t=0..t)
	 erfi(t) = 2 / sqrt(pi) * exp(t^2) * dawson(t)
	 phi(x) = (1 - sigma * x * exp(-sigma^2 * x^2 / 2) * sqrt(pi / 2) * (2 / sqrt(pi) * exp(sigma^2 * x^2 / 2) * dawson(sigma * x / sqrt(2)) - i)) * exp(i * gamma * x)
	        = (1 - sigma * x * (sqrt(2) * dawson(sigma * x / sqrt(2)) - i * exp(-sigma^2 * x^2 / 2) * sqrt(pi/2))) * exp(i * gamma * x)
		= (1 - t * (2 * dawson(t) - i * exp(-t * t) * sqrt(pi))) * exp(i * gamma * x) with t = sigma * x / sqrt(2)
      */
      NumericalComplex Rayleigh::computeCharacteristicFunction(const NumericalScalar x,
							       const Bool logScale) const
      {
	const NumericalScalar t(sigma_ * x / sqrt(2.0));
	if (logScale) return log(NumericalComplex(1 - 2 * t * SpecFunc::Dawson(t), t * exp(-t * t) * sqrt(M_PI))) + NumericalComplex(0.0, x * gamma_);
	return NumericalComplex(1 - 2 * t * SpecFunc::Dawson(t), t * exp(-t * t) * sqrt(M_PI)) * exp(NumericalComplex(0.0, x * gamma_));
      }

      /* Get the PDFGradient of the distribution */
      Rayleigh::NumericalPoint Rayleigh::computePDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	NumericalPoint pdfGradient(2, 0.0);
	if (x <= 0.0) return pdfGradient;
	const NumericalScalar sigma2(sigma_ * sigma_);
	const NumericalScalar factor1(computePDF(point) / sigma2);
	const NumericalScalar factor2((x - sigma_) * (x + sigma_));
	pdfGradient[0] = factor1 * (factor2 - sigma2) / sigma_;
	pdfGradient[1] = factor1 *  factor2 / x;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Rayleigh::NumericalPoint Rayleigh::computeCDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - gamma_);
	NumericalPoint cdfGradient(2, 0.0);
	const NumericalScalar pdf(computePDF(point));
	if (x <= 0.0) return cdfGradient;
	cdfGradient[0] = -x * pdf / sigma_;
	cdfGradient[1] = -pdf;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Rayleigh::computeScalarQuantile(const NumericalScalar prob,
						      const NumericalScalar initialGuess,
						      const NumericalScalar initialStep,
						      const NumericalScalar precision) const
      {
	return gamma_ + sigma_ * sqrt(-2.0 * log(1.0 - prob));
      }

      /* Get the mean of the distribution */
      Rayleigh::NumericalPoint Rayleigh::getMean() const throw(NotDefinedException)
      {
	// 1.253314137315500251207882 = sqrt(pi/2)
	return NumericalPoint(1, gamma_ + 1.253314137315500251207882 * sigma_);
      }

      /* Get the standard deviation of the distribution */
      Rayleigh::NumericalPoint Rayleigh::getStandardDeviation() const throw(NotDefinedException)
      {
	// 0.6551363775620335530939357 = sqrt(2 - pi / 2)
	return NumericalPoint(1, 0.6551363775620335530939357 * sigma_);
      }

      /* Get the skewness of the distribution */
      Rayleigh::NumericalPoint Rayleigh::getSkewness() const throw(NotDefinedException)
      {
	// 0.631110657818937138918970 = 2 * sqrt(pi / (4 - pi)) * (pi - 3) / (4 - pi)
	return NumericalPoint(1, 0.631110657818937138918970);
      }

      /* Get the kurtosis of the distribution */
      Rayleigh::NumericalPoint Rayleigh::getKurtosis() const throw(NotDefinedException)
      {
	// 3.245089300687638062848667 = 3 - (6 * pi^2 - 24 * pi + 16) / (4 - pi)^2
	return NumericalPoint(1, 3.245089300687638062848667);
      }

      /* Get the covariance of the distribution */
      Rayleigh::CovarianceMatrix Rayleigh::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	// 0.429203673205103380768678 = (4 - pi) / 2
	covariance(0, 0) = 0.429203673205103380768678 * sigma_ * sigma_;
	return covariance;
      }

      /* Parameters value and description accessor */
      Rayleigh::NumericalPointWithDescriptionCollection Rayleigh::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(2);
        Description description(point.getDimension());
	point[0] = sigma_;
	description[0] = "sigma";
	point[1] = gamma_;
	description[1] = "gamma";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }

      void Rayleigh::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
	*this = Rayleigh(parametersCollection[0][0], parametersCollection[0][1]);
      }


      /* Sigma accessor */
      void Rayleigh::setSigma(const NumericalScalar sigma)
	throw(InvalidArgumentException)
      {
	if (sigma <= 0.) throw InvalidArgumentException(HERE) << "Sigma MUST be positive";
	sigma_ = sigma;
	computeRange();
      }

      NumericalScalar Rayleigh::getSigma() const
      {
	return sigma_;
      }

      /* Gamma accessor */
      void Rayleigh::setGamma(const NumericalScalar gamma)
      {
	gamma_ = gamma;
	computeRange();
      }

      NumericalScalar Rayleigh::getGamma() const
      {
	return gamma_;
      }

      /* Method save() stores the object through the StorageManager */
      void Rayleigh::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("sigma_", sigma_);
	adv.writeValue("gamma_", gamma_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Rayleigh::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "sigma_") sigma_ = value;
	    if (name == "gamma_") gamma_ = value;
	  }
	}
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
